@extends('admin.admin')

@section('title', 'Edit Paradigm Template - ' . $template->name)

@section('content')
<div class="page-wrapper">
    <div class="content">
        <!-- Clean Header -->
        <div class="page-header mb-4">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h3 class="page-title">{{ __('Edit Paradigm Template') }}</h3>
                    <p class="page-subtitle">{{ $template->name }}</p>
                </div>
                <div>
                    <a href="{{ route('admin.paradigm.templates.show', $template) }}" class="btn btn-outline-secondary me-2">
                        <i class="ti ti-arrow-left me-2"></i>{{ __('Back') }}
                    </a>
                    <button type="submit" form="template-edit-form" class="btn btn-primary">
                        <i class="ti ti-device-floppy me-2"></i>{{ __('Save Changes') }}
                    </button>
                </div>
            </div>
        </div>

        <form id="template-edit-form" data-template-id="{{ $template->id }}">
            @csrf
            @method('PUT')
            
            <!-- Template Basic Info -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="card-title mb-0">{{ __('Template Information') }}</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group mb-3">
                                <label for="name" class="form-label">{{ __('Template Name') }}</label>
                                <input type="text" class="form-control" id="name" name="name" value="{{ $template->name }}" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group mb-3">
                                <label for="is_active" class="form-label">{{ __('Status') }}</label>
                                <select class="form-select" id="is_active" name="is_active">
                                    <option value="1" {{ $template->is_active ? 'selected' : '' }}>{{ __('Active') }}</option>
                                    <option value="0" {{ !$template->is_active ? 'selected' : '' }}>{{ __('Inactive') }}</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description" class="form-label">{{ __('Description') }}</label>
                        <textarea class="form-control" id="description" name="description" rows="3">{{ $template->description }}</textarea>
                    </div>
                </div>
            </div>

            <!-- Part 1 Questions -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="card-title mb-0">{{ __('Part 1: Personality Preferences (30 Questions)') }}</h5>
                    <small class="text-muted">4-option ranking (select 3 out of 4) - Maps to: LU, LL, RU, RL quadrants</small>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="40%">Question</th>
                                    <th width="15%">Option A</th>
                                    <th width="15%">Option B</th>
                                    <th width="15%">Option C</th>
                                    <th width="15%">Option D</th>
                                    <th width="10%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part1Questions = DB::table('paradigm_questions')
                                        ->where('assessment_module', 'paradigm_main')
                                        ->where('axis', 'paradigm_main_part1')
                                        ->orderBy('display_order')
                                        ->get();
                                @endphp
                                @forelse($part1Questions as $question)
                                <tr>
                                    <td>{{ $question->display_order }}</td>
                                    <td>{{ $question->question_text }}</td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_a_label }}</small>
                                        <br><span class="badge bg-primary">{{ $question->a_side }}</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_b_label }}</small>
                                        <br><span class="badge bg-success">{{ $question->b_side }}</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_c_label }}</small>
                                        <br><span class="badge bg-warning">{{ $question->c_side ?? 'N/A' }}</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_d_label }}</small>
                                        <br><span class="badge bg-info">{{ $question->d_side ?? 'N/A' }}</span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="7" class="text-center">
                                        <div class="py-4">
                                            <p class="text-muted">No Part 1 questions found.</p>
                                            <button type="button" class="btn btn-primary" onclick="addPart1Questions()">
                                                <i class="ti ti-plus me-1"></i>{{ __('Add Part 1 Questions') }}
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Part 2 Questions -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-2">{{ __('Part 2: Behavioral Fields') }}</h5>
                    <p class="mb-2 text-muted">Each question maps to one of 8 behavioral fields:</p>
                    <div class="d-flex flex-wrap gap-2">
                        <span class="badge bg-primary">LU – The Question Asker</span>
                        <span class="badge bg-info">LU – The Fact Finder</span>
                        <span class="badge bg-success">LL – The Rule Keeper</span>
                        <span class="badge bg-warning text-dark">LL – The Task Manager</span>
                        <span class="badge bg-danger">RU – The Creator</span>
                        <span class="badge bg-secondary">RU – The Big Thinker</span>
                        <span class="badge bg-dark">RL – The Friend Maker</span>
                        <span class="badge" style="background-color: #6c757d; color: white;">RL – The Empath</span>
                    </div>
                </div>
                <div class="card-body">

                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="40%">Question</th>
                                    <th width="25%">Option A</th>
                                    <th width="25%">Option B</th>
                                    <th width="5%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part2Questions = DB::table('paradigm_questions')
                                        ->where('assessment_module', 'paradigm_main')
                                        ->where('axis', 'part2')
                                        ->orderBy('display_order')
                                        ->get();
                                    
                                    $sideColorMap = [
                                        'LU – The Question Asker' => 'bg-primary',
                                        'LU – The Fact Finder' => 'bg-info',
                                        'LL – The Rule Keeper' => 'bg-success',
                                        'LL – The Task Manager' => 'bg-warning',
                                        'RU – The Creator' => 'bg-danger',
                                        'RU – The Big Thinker' => 'bg-secondary',
                                        'RL – The Friend Maker' => 'bg-dark',
                                        'RL – The Empath' => 'bg-light text-dark',
                                    ];
                                @endphp
                                @forelse($part2Questions as $question)
                                <tr>
                                    <td>{{ $question->display_order }}</td>
                                    <td>{{ $question->question_text }}</td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_a_label ?: 'No option A' }}</small>
                                        <br><span class="badge {{ $sideColorMap[$question->a_side] ?? 'bg-light' }}">{{ $question->a_side ?: 'N/A' }}</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_b_label ?: 'No option B' }}</small>
                                        <br><span class="badge {{ $sideColorMap[$question->b_side] ?? 'bg-light' }}">{{ $question->b_side ?: 'N/A' }}</span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="5" class="text-center">
                                        <div class="py-4">
                                            <p class="text-muted">No Part 2 questions found.</p>
                                            <button type="button" class="btn btn-success" onclick="addPart2Questions()">
                                                <i class="ti ti-plus me-1"></i>{{ __('Add Part 2 Questions') }}
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Part 3: Personality Blueprint -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-2">{{ __('Part 3: Personality Blueprint') }}</h5>
                    <p class="mb-2 text-muted">Each question maps to personality type:</p>
                    <div class="d-flex flex-wrap gap-2">
                        <span class="badge bg-success">E – Extraversion</span>
                        <span class="badge bg-info">I – Introversion</span>
                    </div>
                </div>
                <div class="card-body">
                    
                    <div class="table-responsive">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="35%">Question</th>
                                    <th width="25%">Option A (E)</th>
                                    <th width="25%">Option B (I)</th>
                                    <th width="10%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part3Questions = DB::table('paradigm_questions')
                                        ->where('assessment_module', 'personality_blueprint')
                                        ->where('axis', 'E_I')
                                        ->orderBy('display_order')
                                        ->get();
                                @endphp
                                @forelse($part3Questions as $question)
                                <tr>
                                    <td>{{ $question->display_order }}</td>
                                    <td>{{ $question->question_text }}</td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_a_label ?: 'No option A' }}</small>
                                        <br><span class="badge bg-success">E</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_b_label ?: 'No option B' }}</small>
                                        <br><span class="badge bg-info">I</span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="5" class="text-center">
                                        <div class="py-4">
                                            <p class="text-muted">No Part 3 questions found.</p>
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Part 4: Conflict Styles -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-2">{{ __('Part 4: Conflict Styles') }}</h5>
                    <p class="mb-2 text-muted">Each question maps to one of 5 conflict resolution styles:</p>
                    <div class="d-flex flex-wrap gap-2">
                        <span class="badge" style="background-color: #dc3545;">⑤ Assertive Challenger</span>
                        <span class="badge" style="background-color: #17a2b8;">③ Integrative Bridge Builder</span>
                        <span class="badge" style="background-color: #ffc107; color: #000;">① Fair Negotiator</span>
                        <span class="badge" style="background-color: #6c757d;">④ Quiet Protector</span>
                        <span class="badge" style="background-color: #28a745;">② Peacekeeper</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="35%">Question</th>
                                    <th width="25%">Option A</th>
                                    <th width="25%">Option B</th>
                                    <th width="10%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part4Questions = DB::table('paradigm_questions')
                                        ->where('assessment_module', 'conflict_styles')
                                        ->where('axis', 'conflict')
                                        ->orderBy('display_order')
                                        ->get();
                                    
                                    $conflictStyleColorMap = [
                                        'Assertive Challenger' => ['bg' => '#dc3545', 'text' => '#fff'],
                                        'Integrative Bridge Builder' => ['bg' => '#17a2b8', 'text' => '#fff'],
                                        'Fair Negotiator' => ['bg' => '#ffc107', 'text' => '#000'],
                                        'Quiet Protector' => ['bg' => '#6c757d', 'text' => '#fff'],
                                        'Peacekeeper' => ['bg' => '#28a745', 'text' => '#fff'],
                                    ];
                                @endphp
                                @forelse($part4Questions as $question)
                                <tr>
                                    <td>{{ $question->display_order }}</td>
                                    <td>Question {{ $question->display_order }}</td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_a_label ?: 'No option A' }}</small>
                                        @php $styleA = $conflictStyleColorMap[$question->a_side] ?? ['bg' => '#6c757d', 'text' => '#fff']; @endphp
                                        <br><span class="badge" style="background-color: {{ $styleA['bg'] }}; color: {{ $styleA['text'] }};">{{ $question->a_side }}</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_b_label ?: 'No option B' }}</small>
                                        @php $styleB = $conflictStyleColorMap[$question->b_side] ?? ['bg' => '#6c757d', 'text' => '#fff']; @endphp
                                        <br><span class="badge" style="background-color: {{ $styleB['bg'] }}; color: {{ $styleB['text'] }};">{{ $question->b_side }}</span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="5" class="text-center">
                                        <div class="py-4">
                                            <p class="text-muted">No Part 4 questions found.</p>
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Part 5: Connection Styles -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-2">{{ __('Part 5: Connection Styles') }}</h5>
                    <p class="mb-2 text-muted">Each question maps to one of 5 connection styles (indicated by letters):</p>
                    <div class="d-flex flex-wrap gap-2">
                        <span class="badge" style="background-color: #dc3545;">⑤ Verbal Validator</span>
                        <span class="badge" style="background-color: #6f42c1;">③ Presence Seeker</span>
                        <span class="badge" style="background-color: #fd7e14;">① Grounded Connector</span>
                        <span class="badge" style="background-color: #20c997;">④ Practical Supporter</span>
                        <span class="badge" style="background-color: #ffc107; color: #000;">② Symbolic Appreciator</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="35%">Question</th>
                                    <th width="25%">Option A</th>
                                    <th width="25%">Option B</th>
                                    <th width="10%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part5Questions = DB::table('paradigm_questions')
                                        ->where('assessment_module', 'connection_styles')
                                        ->where('axis', 'connection')
                                        ->orderBy('display_order')
                                        ->get();
                                    
                                    $connectionStyleColorMap = [
                                        'A' => ['bg' => '#dc3545', 'text' => '#fff', 'name' => 'Verbal Validator'],
                                        'B' => ['bg' => '#6f42c1', 'text' => '#fff', 'name' => 'Presence Seeker'],
                                        'C' => ['bg' => '#ffc107', 'text' => '#000', 'name' => 'Symbolic Appreciator'],
                                        'D' => ['bg' => '#20c997', 'text' => '#fff', 'name' => 'Practical Supporter'],
                                        'E' => ['bg' => '#fd7e14', 'text' => '#fff', 'name' => 'Grounded Connector'],
                                    ];
                                @endphp
                                @forelse($part5Questions as $question)
                                <tr>
                                    <td>{{ $question->display_order }}</td>
                                    <td>{{ $question->question_text }}</td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_a_label ?: 'No option A' }}</small>
                                        @php $styleA = $connectionStyleColorMap[$question->a_side] ?? ['bg' => '#6c757d', 'text' => '#fff', 'name' => $question->a_side]; @endphp
                                        <br><span class="badge" style="background-color: {{ $styleA['bg'] }}; color: {{ $styleA['text'] }};">{{ $question->a_side }} - {{ $styleA['name'] }}</span>
                                    </td>
                                    <td>
                                        <small class="text-muted">{{ $question->option_b_label ?: 'No option B' }}</small>
                                        @php $styleB = $connectionStyleColorMap[$question->b_side] ?? ['bg' => '#6c757d', 'text' => '#fff', 'name' => $question->b_side]; @endphp
                                        <br><span class="badge" style="background-color: {{ $styleB['bg'] }}; color: {{ $styleB['text'] }};">{{ $question->b_side }} - {{ $styleB['name'] }}</span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="5" class="text-center">
                                        <div class="py-4">
                                            <p class="text-muted">No Part 5 questions found.</p>
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Part 6: Social, Emotional Intelligence & Relevancy Quotient -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-2">{{ __('Part 6: Intelligence & Relevancy Quotient') }}</h5>
                    <p class="mb-2 text-muted">44 questions across 3 sections (A/B/C format, A=3pts, B=2pts, C=1pt)</p>
                </div>
                <div class="card-body">
                    <!-- Social Intelligence Subsection -->
                    <h6 class="text-primary mb-2">Social Intelligence (12 questions)</h6>
                    <div class="table-responsive mb-4">
                        <table class="table table-sm table-bordered">
                            <thead class="table-light">
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="35%">Question</th>
                                    <th width="15%">A (3pts)</th>
                                    <th width="15%">B (2pts)</th>
                                    <th width="15%">C (1pt)</th>
                                    <th width="15%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part6SiQuestions = DB::table('paradigm_questions')
                                        ->where('question_type', 'part6_si')
                                        ->orderBy('display_order')
                                        ->get();
                                @endphp
                                @forelse($part6SiQuestions as $question)
                                <tr>
                                    <td>{{ $loop->iteration }}</td>
                                    <td><small>{{ $question->question_text }}</small></td>
                                    <td><small>{{ Str::limit($question->option_a_label, 30) }}</small></td>
                                    <td><small>{{ Str::limit($question->option_b_label, 30) }}</small></td>
                                    <td><small>{{ Str::limit($question->option_c_label, 30) }}</small></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr><td colspan="6" class="text-center text-muted">No SI questions</td></tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    <!-- Emotional Intelligence Subsection -->
                    <h6 class="text-success mb-2">Emotional Intelligence (20 questions)</h6>
                    <div class="table-responsive mb-4">
                        <table class="table table-sm table-bordered">
                            <thead class="table-light">
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="35%">Question</th>
                                    <th width="15%">A (3pts)</th>
                                    <th width="15%">B (2pts)</th>
                                    <th width="15%">C (1pt)</th>
                                    <th width="15%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part6EqQuestions = DB::table('paradigm_questions')
                                        ->where('question_type', 'part6_eq')
                                        ->orderBy('display_order')
                                        ->get();
                                @endphp
                                @forelse($part6EqQuestions as $question)
                                <tr>
                                    <td>{{ $loop->iteration }}</td>
                                    <td><small>{{ $question->question_text }}</small></td>
                                    <td><small>{{ Str::limit($question->option_a_label, 30) }}</small></td>
                                    <td><small>{{ Str::limit($question->option_b_label, 30) }}</small></td>
                                    <td><small>{{ Str::limit($question->option_c_label, 30) }}</small></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr><td colspan="6" class="text-center text-muted">No EQ questions</td></tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    <!-- Relevancy Quotient Subsection -->
                    <h6 class="text-warning mb-2">Relevancy Quotient (12 questions)</h6>
                    <div class="table-responsive">
                        <table class="table table-sm table-bordered">
                            <thead class="table-light">
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="35%">Question</th>
                                    <th width="15%">A (3pts)</th>
                                    <th width="15%">B (2pts)</th>
                                    <th width="15%">C (1pt)</th>
                                    <th width="15%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @php
                                    $part6RqQuestions = DB::table('paradigm_questions')
                                        ->where('question_type', 'part6_rq')
                                        ->orderBy('display_order')
                                        ->get();
                                @endphp
                                @forelse($part6RqQuestions as $question)
                                <tr>
                                    <td>{{ $loop->iteration }}</td>
                                    <td><small>{{ $question->question_text }}</small></td>
                                    <td><small>{{ Str::limit($question->option_a_label, 30) }}</small></td>
                                    <td><small>{{ Str::limit($question->option_b_label, 30) }}</small></td>
                                    <td><small>{{ Str::limit($question->option_c_label, 30) }}</small></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="editQuestion({{ $question->id }})">
                                            <i class="ti ti-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                @empty
                                <tr><td colspan="6" class="text-center text-muted">No RQ questions</td></tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Part 7: Paradigm Balance (8 Wellness Dimensions) -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-2">{{ __('Part 7: Paradigm Balance™') }}</h5>
                    <p class="mb-2 text-muted">80 questions across 8 wellness dimensions (1-4 rating scale)</p>
                    <div class="alert alert-info mb-0">
                        <small><strong>Fixed Rating Scale:</strong> 1 = Rarely, if ever | 2 = Sometimes | 3 = Most of the time | 4 = Always</small>
                    </div>
                </div>
                <div class="card-body">
                    @php
                        $part7Sections = [
                            ['type' => 'part7_emotional', 'name' => 'Emotional Balance', 'color' => 'primary'],
                            ['type' => 'part7_spiritual', 'name' => 'Spiritual Balance', 'color' => 'info'],
                            ['type' => 'part7_physical', 'name' => 'Physical Balance', 'color' => 'success'],
                            ['type' => 'part7_social', 'name' => 'Social Balance', 'color' => 'warning'],
                            ['type' => 'part7_financial', 'name' => 'Financial Balance', 'color' => 'danger'],
                            ['type' => 'part7_occupational', 'name' => 'Occupational Balance', 'color' => 'secondary'],
                            ['type' => 'part7_intellectual', 'name' => 'Intellectual Balance', 'color' => 'dark'],
                            ['type' => 'part7_environmental', 'name' => 'Environmental Balance', 'color' => 'success'],
                        ];
                    @endphp
                    
                    @foreach($part7Sections as $section)
                        <h6 class="text-{{ $section['color'] }} mb-2 mt-3">{{ $section['name'] }} (10 questions)</h6>
                        <div class="table-responsive mb-3">
                            <table class="table table-sm table-bordered">
                                <thead class="table-light">
                                    <tr>
                                        <th width="5%">#</th>
                                        <th width="85%">Statement</th>
                                        <th width="10%">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @php
                                        $sectionQuestions = DB::table('paradigm_questions')
                                            ->where('question_type', $section['type'])
                                            ->orderBy('display_order')
                                            ->get();
                                    @endphp
                                    @forelse($sectionQuestions as $question)
                                    <tr>
                                        <td>{{ $loop->iteration }}</td>
                                        <td><small>{{ $question->question_text }}</small></td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="editPart7Question({{ $question->id }})">
                                                <i class="ti ti-edit"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    @empty
                                    <tr><td colspan="3" class="text-center text-muted">No questions</td></tr>
                                    @endforelse
                                </tbody>
                            </table>
                        </div>
                    @endforeach
                </div>
            </div>

            <!-- PDF Builder Link -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="card-title mb-0">{{ __('PDF Configuration') }}</h5>
                </div>
                <div class="card-body">
                    <p class="text-muted mb-3">{{ __('Configure the PDF output elements and layout for the assessment results.') }}</p>
                    <a href="{{ route('admin.paradigm.paradigm-pdf-builder.template', $template->id) }}" class="btn btn-primary">
                        <i class="ti ti-layout-2 me-2"></i>{{ __('Open PDF Builder') }}
                    </a>
                </div>
            </div>

            <!-- Pricing Settings Link -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="card-title mb-0">{{ __('Pricing Settings') }}</h5>
                </div>
                <div class="card-body">
                    <p class="text-muted mb-3">{{ __('Configure pricing for the assessment. Set to 0 to make it free.') }}</p>
                    <a href="{{ route('admin.paradigm.pricing.index') }}" class="btn btn-success">
                        <i class="ti ti-currency-dollar me-2"></i>{{ __('Manage Pricing') }}
                    </a>
                </div>
            </div>

        </form>
    </div>
</div>

<script>
// Global variables
let templateId;

document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('template-edit-form');
    templateId = form.dataset.templateId;
    
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(form);
        
        // Show loading state
        const submitBtn = document.querySelector('button[type="submit"][form="template-edit-form"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="ti ti-loader me-2"></i>{{ __("Saving...") }}';
        submitBtn.disabled = true;
        
        fetch(`/admin/paradigm/templates/${templateId}`, {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'X-HTTP-Method-Override': 'PUT'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (typeof toastr !== 'undefined') {
                    toastr.success('{{ __("Template updated successfully") }}');
                }
                setTimeout(() => {
                    window.location.href = '/admin/paradigm/templates/' + templateId;
                }, 1000);
            } else {
                if (typeof toastr !== 'undefined') {
                    toastr.error(data.message || '{{ __("Error updating template") }}');
                }
            }
        })
        .catch(error => {
            console.error('Error:', error);
            if (typeof toastr !== 'undefined') {
                toastr.error('{{ __("An error occurred. Please try again.") }}');
            }
        })
        .finally(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    });
    
    // Add Part 1 Questions
    window.addPart1Questions = function() {
        if (confirm('{{ __("This will add 30 Part 1 questions. Continue?") }}')) {
            fetch('/admin/paradigm/seed-part1-questions', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (typeof toastr !== 'undefined') {
                        toastr.success('{{ __("Part 1 questions added successfully") }}');
                    }
                    location.reload();
                } else {
                    if (typeof toastr !== 'undefined') {
                        toastr.error(data.message || '{{ __("Error adding questions") }}');
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                if (typeof toastr !== 'undefined') {
                    toastr.error('{{ __("An error occurred") }}');
                }
            });
        }
    };
    
    // Add Part 2 Questions
    window.addPart2Questions = function() {
        if (confirm('{{ __("This will add 64 Part 2 questions. Continue?") }}')) {
            fetch('/admin/paradigm/seed-part2-questions', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (typeof toastr !== 'undefined') {
                        toastr.success('{{ __("Part 2 questions added successfully") }}');
                    }
                    location.reload();
                } else {
                    if (typeof toastr !== 'undefined') {
                        toastr.error(data.message || '{{ __("Error adding questions") }}');
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                if (typeof toastr !== 'undefined') {
                    toastr.error('{{ __("An error occurred") }}');
                }
            });
        }
    };

        window.viewPart1Questions = function() {
            window.open(`/admin/paradigm/templates/${templateId}/questions?part=1`, '_blank');
        };

        window.viewPart2Questions = function() {
            window.open(`/admin/paradigm/templates/${templateId}/questions?part=2`, '_blank');
        };
    });

    // Global functions
    window.editQuestion = function(questionId) {
        // Get question data from the table row
        const row = event.target.closest('tr');
        const questionText = row.cells[1].textContent.trim();
        
        // Check if Part 6 first (no badges, has 6 columns, option C in cell 4)
        const isPart6 = row.cells.length === 6 && !row.cells[2].querySelector('.badge');
        
        let optionA, optionB, sideA = '', sideB = '', optionC = '', optionD = '', sideC = '', sideD = '';
        
        if (isPart6) {
            // Part 6: Just read text options, no badges (fixed scoring)
            optionA = row.cells[2].querySelector('small').textContent.trim();
            optionB = row.cells[3].querySelector('small').textContent.trim();
            optionC = row.cells[4].querySelector('small').textContent.trim();
            sideA = '3'; // Fixed values for detection
            sideB = '2';
        } else {
            // All other parts: Read options and badges
            optionA = row.cells[2].querySelector('small').textContent.trim();
            optionB = row.cells[3].querySelector('small').textContent.trim();
            sideA = row.cells[2].querySelector('.badge').textContent.trim();
            sideB = row.cells[3].querySelector('.badge').textContent.trim();
        }
        
        // Check part types
        const isPart1 = row.cells.length > 6; // Part 1 has 7 columns
        const isPart3 = !isPart1 && !isPart6 && (sideA === 'E' || sideA === 'I'); // Part 3 uses E/I
        const conflictStyles = ['Assertive Challenger', 'Integrative Bridge Builder', 'Fair Negotiator', 'Quiet Protector', 'Peacekeeper'];
        const isPart4 = !isPart1 && !isPart6 && (conflictStyles.includes(sideA) || conflictStyles.includes(sideB)); // Part 4 uses conflict styles
        // Part 5 detection: badges show "A - Verbal Validator" format
        const isPart5 = !isPart1 && !isPart6 && (sideA.match(/^[A-E]\s-/) || sideB.match(/^[A-E]\s-/));
        
        if (isPart1) {
            optionC = row.cells[4].querySelector('small').textContent.trim();
            optionD = row.cells[5].querySelector('small').textContent.trim();
            sideC = row.cells[4].querySelector('.badge').textContent.trim();
            sideD = row.cells[5].querySelector('.badge').textContent.trim();
        }
        
        // Generate options for Part 1 (quadrants), Part 2 (behavioral fields), Part 3 (E/I), Part 4 (conflict styles), or Part 5 (connection styles)
        const getSideOptions = (isPart1, isPart3, isPart4, isPart5, currentSide) => {
            if (isPart1) {
                return `
                    <option value="LU" ${currentSide === 'LU' ? 'selected' : ''}>LU</option>
                    <option value="LL" ${currentSide === 'LL' ? 'selected' : ''}>LL</option>
                    <option value="RU" ${currentSide === 'RU' ? 'selected' : ''}>RU</option>
                    <option value="RL" ${currentSide === 'RL' ? 'selected' : ''}>RL</option>
                `;
            } else if (isPart3) {
                return `
                    <option value="E" ${currentSide === 'E' ? 'selected' : ''}>E – Extraversion</option>
                    <option value="I" ${currentSide === 'I' ? 'selected' : ''}>I – Introversion</option>
                `;
            } else if (isPart4) {
                const styles = [
                    'Assertive Challenger',
                    'Integrative Bridge Builder',
                    'Fair Negotiator',
                    'Quiet Protector',
                    'Peacekeeper'
                ];
                return styles.map(style => `<option value="${style}" ${currentSide === style ? 'selected' : ''}>${style}</option>`).join('');
            } else if (isPart5) {
                // Extract just the letter from "A - Verbal Validator" format
                const currentLetter = currentSide.charAt(0);
                const connections = [
                    {val: 'A', name: 'A - Verbal Validator'},
                    {val: 'B', name: 'B - Presence Seeker'},
                    {val: 'C', name: 'C - Symbolic Appreciator'},
                    {val: 'D', name: 'D - Practical Supporter'},
                    {val: 'E', name: 'E - Grounded Connector'}
                ];
                return connections.map(c => `<option value="${c.val}" ${currentLetter === c.val ? 'selected' : ''}>${c.name}</option>`).join('');
            } else {
                const sides = [
                    'LU – The Question Asker', 'LU – The Fact Finder',
                    'LL – The Rule Keeper', 'LL – The Task Manager',
                    'RU – The Creator', 'RU – The Big Thinker',
                    'RL – The Friend Maker', 'RL – The Empath'
                ];
                return sides.map(side => `<option value="${side}" ${currentSide === side ? 'selected' : ''}>${side}</option>`).join('');
            }
        };

        const modalHtml = `
            <div class="modal fade" id="editModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Edit Question ${questionId} - ${isPart1 ? 'Part 1 (Quadrants)' : isPart6 ? 'Part 6 (Intelligence/Quotient)' : isPart3 ? 'Part 3 (Personality)' : isPart4 ? 'Part 4 (Conflict Styles)' : isPart5 ? 'Part 5 (Connection Styles)' : 'Part 2 (Behavioral Fields)'}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="editForm">
                                <div class="mb-3">
                                    <label class="form-label">Question Text</label>
                                    <textarea class="form-control" id="questionText" rows="3">${questionText}</textarea>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Option A</label>
                                            <input type="text" class="form-control" id="optionA" value="${optionA}">
                                            <select class="form-select mt-2" id="sideA">
                                                ${getSideOptions(isPart1, isPart3, isPart4, isPart5, sideA)}
                                            </select>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">Option B</label>
                                            <input type="text" class="form-control" id="optionB" value="${optionB}">
                                            <select class="form-select mt-2" id="sideB">
                                                ${getSideOptions(isPart1, isPart3, isPart4, isPart5, sideB)}
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        ${isPart1 ? `
                                        <div class="mb-3">
                                            <label class="form-label">Option C</label>
                                            <input type="text" class="form-control" id="optionC" value="${optionC}">
                                            <select class="form-select mt-2" id="sideC">
                                                ${getSideOptions(isPart1, isPart3, isPart4, isPart5, sideC)}
                                            </select>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">Option D</label>
                                            <input type="text" class="form-control" id="optionD" value="${optionD}">
                                            <select class="form-select mt-2" id="sideD">
                                                ${getSideOptions(isPart1, isPart3, isPart4, isPart5, sideD)}
                                            </select>
                                        </div>
                                        ` : isPart6 ? `
                                        <div class="mb-3">
                                            <label class="form-label">Option C (1pt)</label>
                                            <input type="text" class="form-control" id="optionC" value="${optionC}">
                                        </div>
                                        <div class="alert alert-info">
                                            <h6>Part 6: Intelligence/Quotient</h6>
                                            <p class="mb-1">Fixed scoring (cannot be changed):</p>
                                            <div class="d-flex gap-2">
                                                <span class="badge bg-success">A = 3 points</span>
                                                <span class="badge bg-warning text-dark">B = 2 points</span>
                                                <span class="badge bg-danger">C = 1 point</span>
                                            </div>
                                        </div>
                                        ` : isPart3 ? `
                                        <div class="alert alert-info">
                                            <h6>Part 3: Personality Blueprint</h6>
                                            <p class="mb-1">Each question maps to personality type:</p>
                                            <div class="d-flex flex-wrap gap-1">
                                                <span class="badge bg-success">E – Extraversion</span>
                                                <span class="badge bg-info">I – Introversion</span>
                                            </div>
                                        </div>
                                        ` : isPart4 ? `
                                        <div class="alert alert-info">
                                            <h6>Part 4: Conflict Styles</h6>
                                            <p class="mb-1">Each question maps to one of 5 conflict resolution styles:</p>
                                            <div class="d-flex flex-wrap gap-1">
                                                <span class="badge" style="background-color: #dc3545;">Assertive Challenger</span>
                                                <span class="badge" style="background-color: #17a2b8;">Integrative Bridge Builder</span>
                                                <span class="badge" style="background-color: #ffc107; color: #000;">Fair Negotiator</span>
                                                <span class="badge" style="background-color: #6c757d;">Quiet Protector</span>
                                                <span class="badge" style="background-color: #28a745;">Peacekeeper</span>
                                            </div>
                                        </div>
                                        ` : isPart5 ? `
                                        <div class="alert alert-info">
                                            <h6>Part 5: Connection Styles</h6>
                                            <p class="mb-1">Each question maps to one of 5 connection styles (indicated by letters):</p>
                                            <div class="d-flex flex-wrap gap-1">
                                                <span class="badge" style="background-color: #dc3545;">A - Verbal Validator</span>
                                                <span class="badge" style="background-color: #6f42c1;">B - Presence Seeker</span>
                                                <span class="badge" style="background-color: #ffc107; color: #000;">C - Symbolic Appreciator</span>
                                                <span class="badge" style="background-color: #20c997;">D - Practical Supporter</span>
                                                <span class="badge" style="background-color: #fd7e14;">E - Grounded Connector</span>
                                            </div>
                                        </div>
                                        ` : `
                                        <div class="alert alert-info">
                                            <h6>Part 2: Behavioral Fields</h6>
                                            <p class="mb-1">Each question maps to one of 8 behavioral fields:</p>
                                            <div class="d-flex flex-wrap gap-1">
                                                <span class="badge bg-primary">LU – The Question Asker</span>
                                                <span class="badge bg-info">LU – The Fact Finder</span>
                                                <span class="badge bg-success">LL – The Rule Keeper</span>
                                                <span class="badge bg-warning text-dark">LL – The Task Manager</span>
                                                <span class="badge bg-danger">RU – The Creator</span>
                                                <span class="badge bg-secondary">RU – The Big Thinker</span>
                                                <span class="badge bg-dark">RL – The Friend Maker</span>
                                                <span class="badge" style="background-color: #6c757d; color: white;">RL – The Empath</span>
                                            </div>
                                        </div>
                                        `}
                                    </div>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="button" class="btn btn-primary" onclick="saveQuestion(${questionId})">Save Changes</button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if any
        const existingModal = document.getElementById('editModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('editModal'));
        modal.show();
    };
    
    window.saveQuestion = function(questionId) {
        const isPart1 = document.getElementById('optionC') !== null;
        
        const payload = {
            question_text: document.getElementById('questionText').value,
            option_a_label: document.getElementById('optionA').value,
            option_b_label: document.getElementById('optionB').value,
            a_side: document.getElementById('sideA').value,
            b_side: document.getElementById('sideB').value,
            _token: '{{ csrf_token() }}'
        };

        if (isPart1) {
            payload.option_c_label = document.getElementById('optionC').value;
            payload.option_d_label = document.getElementById('optionD').value;
            payload.c_side = document.getElementById('sideC').value;
            payload.d_side = document.getElementById('sideD').value;
        } else if (isPart6) {
            payload.option_c_label = document.getElementById('optionC').value;
            // Part 6 has fixed scoring (a_side=3, b_side=2, c_side=1), no need to send
        }

        fetch(`/admin/paradigm/questions/${questionId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify(payload)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (typeof toastr !== 'undefined') {
                    toastr.success('Question updated successfully');
                }
                location.reload();
            } else {
                if (typeof toastr !== 'undefined') {
                    toastr.error(data.message || 'Error updating question');
                }
            }
        })
        .catch(error => {
            console.error('Error:', error);
            if (typeof toastr !== 'undefined') {
                toastr.error('An error occurred. Please try again.');
            }
        });
    };

    // Part 7 Edit Function (Simpler - only edit statement text)
    window.editPart7Question = function(questionId) {
        const row = event.target.closest('tr');
        const questionText = row.cells[1].querySelector('small').textContent.trim();
        
        const modalHtml = `
            <div class="modal fade" id="editPart7Modal" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Edit Part 7 Question ${questionId}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="editPart7Form">
                                <div class="mb-3">
                                    <label class="form-label">Statement Text</label>
                                    <textarea class="form-control" id="part7QuestionText" rows="3">${questionText}</textarea>
                                </div>
                                <div class="alert alert-info">
                                    <small><strong>Fixed Rating Scale (cannot be changed):</strong></small>
                                    <div class="mt-2">
                                        <small>1 = Rarely, if ever</small><br>
                                        <small>2 = Sometimes</small><br>
                                        <small>3 = Most of the time</small><br>
                                        <small>4 = Always</small>
                                    </div>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-primary" onclick="savePart7Question(${questionId})">Save Changes</button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        const modal = new bootstrap.Modal(document.getElementById('editPart7Modal'));
        
        document.getElementById('editPart7Modal').addEventListener('hidden.bs.modal', function() {
            this.remove();
        });
        
        modal.show();
    };

    window.savePart7Question = function(questionId) {
        const payload = {
            question_text: document.getElementById('part7QuestionText').value,
            _token: '{{ csrf_token() }}'
        };

        fetch(`/admin/paradigm/questions/${questionId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify(payload)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (typeof toastr !== 'undefined') {
                    toastr.success('Question updated successfully');
                }
                bootstrap.Modal.getInstance(document.getElementById('editPart7Modal')).hide();
                location.reload();
            }
        })
        .catch(error => {
            console.error('Error:', error);
            if (typeof toastr !== 'undefined') {
                toastr.error('Failed to update question');
            }
        });
    };
</script>
@endsection