<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PDF Builder - {{ $template->name }}</title>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <link href="https://cdn.jsdelivr.net/npm/remixicon@2.5.0/fonts/remixicon.css" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8f9fa;
            color: #333;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 1rem 2rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .header-content {
            display: flex;
            align-items: center;
            justify-content: space-between;
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .back-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 40px;
            height: 40px;
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 8px;
            color: white;
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .back-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: translateX(-2px);
        }
        
        .header h1 {
            font-size: 1.5rem;
            font-weight: 600;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 0.9rem;
        }
        
        .builder-container {
            display: grid;
            grid-template-columns: 300px 1fr 300px;
            min-height: calc(100vh - 80px);
            max-width: 1400px;
            margin: 0 auto;
            gap: 1rem;
            padding: 1rem;
        }
        
        @media (max-width: 1200px) {
            .builder-container {
                grid-template-columns: 1fr;
                grid-template-rows: auto auto auto;
                height: auto;
            }
            
            .sidebar {
                order: 1;
            }
            
            .pdf-canvas {
                order: 2;
            }
            
            .properties-panel {
                order: 3;
            }
        }
        
        @media (max-width: 768px) {
            .builder-container {
                padding: 0.5rem;
                gap: 0.5rem;
            }
        }
        
        .sidebar {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .sidebar-header {
            background: #f8f9fa;
            padding: 1rem;
            border-bottom: 1px solid #e9ecef;
            font-weight: 600;
        }
        
        .element-library {
            padding: 1rem;
            max-height: calc(100vh - 200px);
            overflow-y: auto;
        }
        
        @media (max-width: 1200px) {
            .element-library {
                max-height: 300px;
            }
        }
        
        @media (max-width: 768px) {
            .element-library {
                max-height: 250px;
            }
        }
        
        .element-item {
            display: flex;
            align-items: center;
            padding: 0.75rem;
            margin-bottom: 0.5rem;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 6px;
            cursor: grab;
            transition: all 0.2s ease;
        }
        
        .element-item:hover {
            background: #e9ecef;
            transform: translateY(-1px);
        }
        
        .element-item:active {
            cursor: grabbing;
        }
        
        .element-icon {
            width: 20px;
            height: 20px;
            margin-right: 0.5rem;
            color: #667eea;
        }
        
        .pdf-canvas {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
            position: relative;
            min-height: 600px;
        }
        
        @media (max-width: 1200px) {
            .pdf-canvas {
                min-height: 500px;
            }
        }
        
        @media (max-width: 768px) {
            .pdf-canvas {
                min-height: 400px;
            }
        }
        
        .canvas-header {
            background: #f8f9fa;
            padding: 1rem;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .page-controls {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .page-info {
            font-weight: 500;
            color: #333;
            background: white;
            padding: 0.25rem 0.75rem;
            border-radius: 15px;
            border: 1px solid #ddd;
            display: flex;
            align-items: center;
            height: 28px; /* Smaller height */
            min-width: 80px;
            justify-content: center;
            font-size: 0.8rem;
        }
        
        .canvas-title {
            font-weight: 600;
            color: #333;
        }
        
        .canvas-actions {
            display: flex;
            gap: 0.25rem;
            align-items: center;
        }
        
        .zoom-controls {
            display: flex;
            align-items: center;
            gap: 0.25rem;
            margin-right: 0.5rem;
            padding: 0.15rem 0.35rem;
            background: #f8f9fa;
            border-radius: 4px;
            border: 1px solid #dee2e6;
        }
        
        .zoom-level {
            font-size: 0.75rem;
            font-weight: 500;
            color: #495057;
            min-width: 35px;
            text-align: center;
        }
        
        .btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.875rem;
            transition: all 0.2s ease;
        }
        
        .btn-sm {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
            height: 28px;
            min-width: 28px;
        }
        
        .btn-primary {
            background: #667eea;
            color: white;
        }
        
        .btn-primary:hover {
            background: #5a6fd8;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .btn-info {
            background: #17a2b8;
            color: white;
        }
        
        .btn-info:hover {
            background: #138496;
        }
        
        /* Preview Modal Styles */
        .preview-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .preview-modal-content {
            background: white;
            border-radius: 8px;
            width: 90%;
            max-width: 800px;
            max-height: 90%;
            overflow: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .preview-header {
            padding: 1rem;
            border-bottom: 1px solid #ddd;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
        }
        
        .preview-header h3 {
            margin: 0;
            color: #333;
        }
        
        .preview-content {
            padding: 1rem;
            overflow-y: auto;
            max-height: calc(90vh - 80px);
        }
        
        .preview-page {
            margin-bottom: 2rem;
        }
        
        .preview-element {
            font-family: Arial, sans-serif;
        }
        
        .canvas-area {
            height: calc(100% - 60px);
            background: #fafafa;
            background-image: 
                linear-gradient(rgba(0,0,0,0.1) 1px, transparent 1px),
                linear-gradient(90deg, rgba(0,0,0,0.1) 1px, transparent 1px);
            background-size: 20px 20px;
            position: relative;
            overflow: auto;
        }
        
        .pdf-page {
            width: 210mm;
            height: 297mm;
            background: white;
            margin: 20px auto;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            position: relative;
            transform-origin: top center;
            transform: scale(0.6);
        }
        
        @media (max-width: 1200px) {
            .pdf-page {
                transform: scale(0.5);
                margin: 15px auto;
            }
        }
        
        @media (max-width: 768px) {
            .pdf-page {
                transform: scale(0.4);
                margin: 10px auto;
            }
        }
        
        @media (max-width: 480px) {
            .pdf-page {
                transform: scale(0.3);
                margin: 5px auto;
            }
        }
        
        .pdf-element {
            position: absolute;
            border: 2px dashed transparent;
            cursor: move;
            min-width: 50px;
            min-height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            color: #333;
            background: rgba(102, 126, 234, 0.1);
            box-sizing: border-box;
            word-wrap: break-word;
            overflow: hidden;
        }
        
        .pdf-element img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 4px;
        }
        
        .pdf-element.resizing {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.2);
        }
        
        .resize-handle {
            position: absolute;
            background: #667eea;
            border: 2px solid white;
            border-radius: 50%;
            width: 12px;
            height: 12px;
            cursor: se-resize;
            opacity: 0;
            transition: opacity 0.2s;
            z-index: 10;
        }
        
        .pdf-element:hover .resize-handle,
        .pdf-element.selected .resize-handle {
            opacity: 1;
        }
        
        .resize-handle.se {
            bottom: -6px;
            right: -6px;
        }
        
        .resize-handle.ne {
            top: -6px;
            right: -6px;
        }
        
        .resize-handle.sw {
            bottom: -6px;
            left: -6px;
        }
        
        .resize-handle.nw {
            top: -6px;
            left: -6px;
        }
        
        .pdf-element:hover {
            border-color: #667eea;
        }
        
        .pdf-element.selected {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.2);
        }
        
        .properties-panel {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
            max-height: calc(100vh - 100px);
            overflow-y: auto;
        }
        
        @media (max-width: 1200px) {
            .properties-panel {
                max-height: 400px;
            }
        }
        
        @media (max-width: 768px) {
            .properties-panel {
                max-height: 300px;
            }
        }
        
        .properties-header {
            background: #f8f9fa;
            padding: 1rem;
            border-bottom: 1px solid #e9ecef;
            font-weight: 600;
        }
        
        .properties-content {
            padding: 1rem;
        }
        
        .form-group {
            margin-bottom: 1rem;
        }
        
        .form-label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #333;
        }
        
        .form-control {
            width: 100%;
            padding: 0.5rem;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 0.875rem;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 2px rgba(102, 126, 234, 0.2);
        }
        
        .color-input {
            width: 50px;
            height: 35px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        
        .range-input {
            width: 100%;
        }
        
        .no-elements {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
            color: #999;
        }
        
        .no-elements i {
            font-size: 3rem;
            margin-bottom: 1rem;
            display: block;
        }
        
        .status-bar {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #333;
            color: white;
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            z-index: 1000;
        }
        
        @media (max-width: 768px) {
            .status-bar {
                padding: 0.25rem 0.5rem;
                font-size: 0.75rem;
            }
        }
        
        .hidden {
            display: none;
        }
        
        .mobile-toggle {
            display: none;
            position: fixed;
            top: 50%;
            right: 10px;
            transform: translateY(-50%);
            background: #667eea;
            color: white;
            border: none;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            font-size: 1.2rem;
            z-index: 1001;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        }
        
        @media (max-width: 1200px) {
            .mobile-toggle {
                display: block;
            }
            
            .sidebar.hidden-mobile,
            .properties-panel.hidden-mobile {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <div>
                <a href="{{ route('admin.paradigm.templates.edit', $template->id) }}" class="back-btn">
                    <i class="ri-arrow-left-line"></i>
                </a>
            </div>
            <div>
                <h1>PDF Builder</h1>
                <p>{{ $template->name }}</p>
            </div>
            <div></div>
        </div>
    </div>

    <div class="builder-container">
        <!-- Element Library -->
        <div class="sidebar">
            <div class="sidebar-header">
                <i class="ri-layout-2-line"></i> Paradigm Elements
            </div>
            <div class="element-library">
                <!-- Paradigm Brain Profile -->
                <div class="element-category">
                    <h6 style="color: #667eea; margin-bottom: 0.5rem; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Brain Profile Results</h6>
                    <div class="element-item" draggable="true" data-type="paradigm_type">
                        <i class="ri-user-heart element-icon"></i>
                        <span>Paradigm Type</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="dominance_category">
                        <i class="ri-award element-icon"></i>
                        <span>Dominance Category</span>
                    </div>
                </div>
                
                <!-- Brain Frames (Part 1) -->
                <div class="element-category">
                    <h6 style="color: #667eea; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Brain Frames (Part 1)</h6>
                    <div class="element-item" draggable="true" data-type="lu_percentage">
                        <i class="ri-bar-chart-line element-icon"></i>
                        <span>LU Score (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="ll_percentage">
                        <i class="ri-bar-chart-line element-icon"></i>
                        <span>LL Score (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="ru_percentage">
                        <i class="ri-bar-chart-line element-icon"></i>
                        <span>RU Score (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="rl_percentage">
                        <i class="ri-bar-chart-line element-icon"></i>
                        <span>RL Score (%)</span>
                    </div>
                </div>
                
                <!-- Brain Fields (Part 2) -->
                <div class="element-category">
                    <h6 style="color: #667eea; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Brain Fields (Part 2)</h6>
                    <div class="element-item" draggable="true" data-type="question_asker_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Question Asker (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="fact_finder_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Fact Finder (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="rule_keeper_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Rule Keeper (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="task_manager_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Task Manager (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="creator_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Creator (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="big_thinker_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Big Thinker (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="friend_maker_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Friend Maker (%)</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="empath_percentage">
                        <i class="ri-bar-chart-box element-icon"></i>
                        <span>Empath (%)</span>
                    </div>
                </div>

                <!-- Part 3: Personality Blueprint -->
                <div class="element-category">
                    <h6 style="color: #17a2b8; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Part 3: Personality</h6>
                    <div class="element-item" draggable="true" data-type="e_score">
                        <i class="ri-hashtag element-icon"></i>
                        <span>E Score</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="i_score">
                        <i class="ri-hashtag element-icon"></i>
                        <span>I Score</span>
                    </div>
                </div>

                <!-- Part 4: Conflict Styles -->
                <div class="element-category">
                    <h6 style="color: #fd7e14; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Part 4: Conflict Styles</h6>
                    <div class="element-item" draggable="true" data-type="assertive_challenger">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Assertive Challenger</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="integrative_bridge_builder">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Integrative Bridge Builder</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="fair_negotiator">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Fair Negotiator</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="quiet_protector">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Quiet Protector</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="peacekeeper">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Peacekeeper</span>
                    </div>
                </div>

                <!-- Part 5: Connection Styles -->
                <div class="element-category">
                    <h6 style="color: #e83e8c; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Part 5: Connection Styles</h6>
                    <div class="element-item" draggable="true" data-type="verbal_validator">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Verbal Validator</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="presence_seeker">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Presence Seeker</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="symbolic_appreciator">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Symbolic Appreciator</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="practical_supporter">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Practical Supporter</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="grounded_connector">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Grounded Connector</span>
                    </div>
                </div>

                <!-- Part 6: Intelligence/Quotient (Checkboxes) -->
                <div class="element-category">
                    <h6 style="color: #20c997; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Part 6: Intelligence (Checkboxes)</h6>
                    <div class="element-item" draggable="true" data-type="si_checkbox">
                        <i class="ri-checkbox-line element-icon"></i>
                        <span>Social Intelligence</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="eq_checkbox">
                        <i class="ri-checkbox-line element-icon"></i>
                        <span>Emotional Intelligence</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="rq_checkbox">
                        <i class="ri-checkbox-line element-icon"></i>
                        <span>Relevancy Quotient</span>
                    </div>
                </div>

                <!-- Part 7: Paradigm Balance (Wellness Dimensions) -->
                <div class="element-category">
                    <h6 style="color: #6610f2; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Part 7: Wellness (Numbers Only)</h6>
                    <div class="element-item" draggable="true" data-type="emotional_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Emotional Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="spiritual_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Spiritual Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="physical_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Physical Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="social_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Social Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="financial_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Financial Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="occupational_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Occupational Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="intellectual_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Intellectual Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="environmental_balance">
                        <i class="ri-hashtag element-icon"></i>
                        <span>Environmental Balance</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="overall_wellness">
                        <i class="ri-bar-chart-2 element-icon"></i>
                        <span>Overall Wellness</span>
                    </div>
                </div>

                <!-- Personal Information -->
                <div class="element-category">
                    <h6 style="color: #667eea; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Personal Info</h6>
                    <div class="element-item" draggable="true" data-type="name">
                        <i class="ri-user element-icon"></i>
                        <span>First Name</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="surname">
                        <i class="ri-user element-icon"></i>
                        <span>Last Name</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="email">
                        <i class="ri-mail element-icon"></i>
                        <span>Email Address</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="gender">
                        <i class="ri-genderless element-icon"></i>
                        <span>Gender</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="age_range">
                        <i class="ri-calendar-event element-icon"></i>
                        <span>Age Range</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="phone">
                        <i class="ri-phone element-icon"></i>
                        <span>Phone Number</span>
                    </div>
                </div>
                
                <!-- Provider Information -->
                <div class="element-category">
                    <h6 style="color: #764ba2; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Provider Info</h6>
                    <div class="element-item" draggable="true" data-type="provider_name">
                        <i class="ri-user-star element-icon"></i>
                        <span>Provider First Name</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="provider_last_name">
                        <i class="ri-user-star element-icon"></i>
                        <span>Provider Last Name</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="provider_email">
                        <i class="ri-mail-star element-icon"></i>
                        <span>Provider Email</span>
                    </div>
                    <div class="element-item" draggable="true" data-type="provider_phone">
                        <i class="ri-phone-line element-icon"></i>
                        <span>Provider Phone</span>
                    </div>
                </div>
                
                <!-- Assessment Details -->
                <div class="element-category">
                    <h6 style="color: #667eea; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px;">Assessment Details</h6>
                    <div class="element-item" draggable="true" data-type="date_completed">
                        <i class="ri-calendar element-icon"></i>
                        <span>Completion Date</span>
                    </div>
                </div>
                
                <!-- NEW ELEMENTS - Auto-sizing Combined Elements -->
                <div class="element-category" style="border-top: 2px solid #667eea; padding-top: 1rem; margin-top: 1rem;">
                    <h6 style="color: #e74c3c; margin: 1rem 0 0.5rem 0; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.5px; font-weight: 700;">
                        <i class="ri-magic-line" style="margin-right: 0.25rem;"></i>New Elements (Auto-sizing)
                    </h6>
                    
                    <div class="element-item" draggable="true" data-type="full_name">
                        <i class="ri-user-star-line element-icon"></i>
                        <span>Full Name (Auto-size)</span>
                    </div>
                    
                    <div class="element-item" draggable="true" data-type="user_details_block">
                        <i class="ri-profile-line element-icon"></i>
                        <span>User Details Block (Auto-size)</span>
                    </div>
                    
                    <div class="element-item" draggable="true" data-type="provider_details_block">
                        <i class="ri-briefcase-line element-icon"></i>
                        <span>Provider Details Block (Auto-size)</span>
                    </div>
                    
                    <div style="background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px; padding: 0.5rem; margin-top: 0.5rem; font-size: 0.65rem; color: #856404;">
                        <i class="ri-information-line" style="margin-right: 0.25rem;"></i>
                        <strong>Auto-sizing:</strong> These elements automatically adjust their size based on content length.
                    </div>
                </div>
            </div>
        </div>

        <!-- PDF Canvas -->
        <div class="pdf-canvas">
            <div class="canvas-header">
                <div class="page-controls">
                    <button class="btn btn-sm btn-outline-secondary" onclick="addPage()" title="Add Page">
                        <i class="ri-add-line"></i>
                    </button>
                    <button class="btn btn-sm btn-outline-secondary" onclick="previousPage()" title="Previous Page" id="prevPageBtn" disabled>
                        <i class="ri-arrow-left-line"></i>
                    </button>
                    <div class="page-info">Page <span id="currentPage">1</span> of <span id="totalPages">1</span></div>
                    <button class="btn btn-sm btn-outline-secondary" onclick="nextPage()" title="Next Page" id="nextPageBtn" disabled>
                        <i class="ri-arrow-right-line"></i>
                    </button>
                    <button class="btn btn-sm btn-outline-secondary" onclick="removePage()" title="Remove Page" id="removePageBtn" disabled>
                        <i class="ri-subtract-line"></i>
                    </button>
                    <button class="btn btn-sm btn-outline-primary" onclick="setPageBackground()" title="Set Background">
                        <i class="ri-image-add-line"></i>
                    </button>
                    <button class="btn btn-sm btn-outline-secondary" onclick="clearPageBackground()" title="Clear Background">
                        <i class="ri-image-line"></i>
                    </button>
                </div>
                <div class="canvas-actions">
                    <div class="zoom-controls">
                        <button class="btn btn-sm btn-outline-secondary" onclick="zoomOut()" title="Zoom Out">
                            <i class="ri-zoom-out-line"></i>
                        </button>
                        <span class="zoom-level" id="zoomLevel">100%</span>
                        <button class="btn btn-sm btn-outline-secondary" onclick="zoomIn()" title="Zoom In">
                            <i class="ri-zoom-in-line"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-secondary" onclick="resetZoom()" title="Reset Zoom">
                            <i class="ri-refresh-line"></i>
                        </button>
                    </div>
                    <button class="btn btn-sm btn-secondary" onclick="clearCanvas()" title="Clear">
                        <i class="ri-delete-bin-line"></i>
                    </button>
                    <button class="btn btn-sm btn-info" onclick="previewPDF()" title="Preview">
                        <i class="ri-eye-line"></i>
                    </button>
                <button class="btn btn-sm btn-primary" onclick="saveElements()" title="Save">
                    <i class="ri-save-line"></i>
                </button>
                <button class="btn btn-sm btn-warning" onclick="testCreateElement()" title="Test">
                    <i class="ri-test-tube-line"></i>
                </button>
                <button class="btn btn-sm btn-info" onclick="debugElements()" title="Debug">
                    <i class="ri-information-line"></i>
                </button>
                </div>
            </div>
            <div class="canvas-area">
                <div class="pdf-page" id="pdfPage">
                    <div class="no-elements" id="noElements">
                        <i class="ri-drag-drop-line"></i>
                        <div>Drag elements here to start building</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Properties Panel -->
        <div class="sidebar">
            <div class="sidebar-header">
                <i class="ri-settings-3-line"></i> Properties
            </div>
            <div class="properties-content">
                <div id="noSelection" class="text-center text-muted">
                    <i class="ri-cursor-line" style="font-size: 2rem; color: #ccc; margin-bottom: 1rem;"></i>
                    <div>Select an element to edit properties</div>
                </div>
                
                <div id="elementProperties" class="hidden">
                    <div class="form-group">
                        <label class="form-label">Element Type</label>
                        <input type="text" class="form-control" id="elementType" readonly>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Text Content</label>
                        <input type="text" class="form-control" id="elementText" placeholder="Enter text...">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Font Size</label>
                        <input type="range" class="form-control range-input" id="fontSize" min="8" max="48" value="12">
                        <small id="fontSizeValue">12px</small>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Text Color</label>
                        <input type="color" class="color-input" id="textColor" value="#000000">
                    </div>
                    
                    
                    <div class="form-group">
                        <label class="form-label">Text Align</label>
                        <select class="form-control" id="textAlign">
                            <option value="left">Left</option>
                            <option value="center">Center</option>
                            <option value="right">Right</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Position</label>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 0.5rem;">
                            <input type="number" class="form-control" id="posX" placeholder="X %" min="0" max="100">
                            <input type="number" class="form-control" id="posY" placeholder="Y %" min="0" max="100">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Size</label>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 0.5rem;">
                            <input type="number" class="form-control" id="width" placeholder="Width %" min="1" max="100">
                            <input type="number" class="form-control" id="height" placeholder="Height %" min="1" max="100">
                        </div>
                    </div>
                    
                    <button class="btn btn-primary" onclick="updateElement()" style="width: 100%;">
                        <i class="ri-refresh-line"></i> Update Element
                    </button>
                    
                    <button class="btn btn-secondary" onclick="deleteElement()" style="width: 100%; margin-top: 0.5rem;">
                        <i class="ri-delete-bin-line"></i> Delete Element
                    </button>
                </div>
            </div>
        </div>
    </div>

    <div class="status-bar">
        <span id="statusText">Ready - Drag elements to start building</span>
    </div>
    
    <button class="mobile-toggle" onclick="toggleMobileSidebar()" id="mobileToggle">
        <i class="ri-menu-line"></i>
    </button>

    <!-- Preview Modal -->
    <div id="previewModal" class="preview-modal" style="display: none;">
        <div class="preview-modal-content">
            <div class="preview-header">
                <h3>PDF Preview</h3>
                <button class="btn btn-secondary" onclick="closePreview()">
                    <i class="ri-close-line"></i> Close
                </button>
            </div>
            <div id="previewContent" class="preview-content">
                <!-- Preview pages will be inserted here -->
            </div>
        </div>
    </div>

    <script>
        let elements = [];
        let selectedElement = null;
        let elementCounter = 0;
        let pages = @json($template->pdfPages->pluck('page_number')->all() ?? [1]);
        if (pages.length === 0) pages = [1];
        let currentPageId = {{ $template->pdfPages->first()->page_number ?? 1 }};
        let pageBackgrounds = {}; // This will now store local changes
        let currentZoom = 1; // Current zoom level (1 = 100%)

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            setupDragAndDrop();
            loadExistingElements();
            updateStatus('PDF Builder loaded successfully');
        });

        // Test function to manually create an element
        function testCreateElement() {
            console.log('Test: Creating element manually');
            createElement('p16_personality_type', 10, 10);
            console.log('Elements after test:', elements);
        }
        
        function debugElements() {
            console.log('Current elements:', elements);
            console.log('Element counter:', elementCounter);
            elements.forEach((el, index) => {
                console.log(`Element ${index}:`, {
                    id: el.id,
                    type: el.type,
                    x: el.x,
                    y: el.y,
                    width: el.width,
                    height: el.height
                });
            });
        }

        function setupDragAndDrop() {
            const elementItems = document.querySelectorAll('.element-item');
            const pdfPage = document.getElementById('pdfPage');

            elementItems.forEach(item => {
                item.addEventListener('dragstart', function(e) {
                    console.log('Drag start:', this.dataset.type);
                    e.dataTransfer.setData('text/plain', this.dataset.type);
                });
            });

            pdfPage.addEventListener('dragover', function(e) {
                e.preventDefault();
                // Add visual feedback during drag
                this.style.backgroundColor = 'rgba(0, 123, 255, 0.1)';
            });
            
            pdfPage.addEventListener('dragleave', function(e) {
                // Remove visual feedback when leaving drop zone
                this.style.backgroundColor = '';
            });

            pdfPage.addEventListener('drop', function(e) {
                e.preventDefault();
                console.log('Drop event triggered!');
                // Remove visual feedback
                this.style.backgroundColor = '';
                
                const elementType = e.dataTransfer.getData('text/plain');
                const elementId = e.dataTransfer.getData('element-id');
                const rect = this.getBoundingClientRect();
                
                console.log('Element type:', elementType);
                console.log('Element ID:', elementId);
                console.log('Drop position:', e.clientX, e.clientY);
                console.log('PDF page rect:', rect);
                
                // ULTRA-PRECISE positioning calculation
                // The canvas has transform: scale(0.6) but getBoundingClientRect() gives us the scaled dimensions
                // We need to calculate based on the actual unscaled canvas dimensions
                const computedStyle = window.getComputedStyle(this);
                const transform = computedStyle.transform;
                let scale = 0.6; // Default scale
                
                if (transform && transform !== 'none') {
                    const matrix = transform.match(/matrix\(([^)]+)\)/);
                    if (matrix) {
                        const values = matrix[1].split(',');
                        scale = parseFloat(values[0]) || 0.6;
                    }
                }
                
                // The actual canvas dimensions (unscaled)
                const actualWidth = 210; // 210mm in CSS units
                const actualHeight = 297; // 297mm in CSS units
                
                // Convert mm to pixels (assuming 96 DPI: 1mm = 3.7795275591 pixels)
                const mmToPx = 3.7795275591;
                const actualWidthPx = actualWidth * mmToPx;
                const actualHeightPx = actualHeight * mmToPx;
                
                // Calculate position relative to the actual unscaled canvas
                const relativeX = (e.clientX - rect.left) / (rect.width / scale);
                const relativeY = (e.clientY - rect.top) / (rect.height / scale);
                
                // Convert to percentage with precise rounding (2 decimal places)
                const x = Math.round(Math.max(0, Math.min(100, relativeX * 100)) * 100) / 100;
                const y = Math.round(Math.max(0, Math.min(100, relativeY * 100)) * 100) / 100;
                
                // Adjust for element center positioning
                const elementWidth = getDefaultWidth(elementType);
                const elementHeight = getDefaultHeight(elementType);
                const adjustedX = Math.max(0, Math.min(100 - elementWidth, x - (elementWidth / 2)));
                const adjustedY = Math.max(0, Math.min(100 - elementHeight, y - (elementHeight / 2)));
                
                console.log('Calculated position:', x, y);
                console.log('Adjusted position:', adjustedX, adjustedY);
                
                if (elementId) {
                    // Moving existing element
                    console.log('Moving existing element');
                    moveElement(elementId, adjustedX, adjustedY);
                } else {
                    // Creating new element
                    console.log('Creating new element');
                    createElement(elementType, adjustedX, adjustedY);
                }
            });
        }

        function createElement(type, x, y) {
            elementCounter++;
            const width = getDefaultWidth(type);
            const defaultHeight = getDefaultHeight(type);
            const fontSize = getDefaultFontSize(type);
            
            // Use default height - don't auto-adjust based on font size
            const height = defaultHeight;
            
            // Ensure element stays within bounds
            const maxX = 100 - width;
            const maxY = 100 - height;
            
            const element = {
                id: `element_${elementCounter}`,
                type: type,
                x: Math.max(0, Math.min(maxX, x)),
                y: Math.max(0, Math.min(maxY, y)),
                width: width,
                height: height,
                text: getDefaultText(type),
                fontSize: fontSize,
                textColor: getDefaultTextColor(type),
                backgroundColor: 'transparent',
                textAlign: 'left',
                pageId: currentPageId
            };
            
            console.log('Creating element with pageId:', currentPageId, 'type:', typeof currentPageId);

            elements.push(element);
            renderElement(element);
            selectElement(element);
            updateStatus(`Created ${getElementDisplayName(type)} at ${Math.round(x)}%, ${Math.round(y)}%`);
            
            // Debug: Log element creation
            console.log('Element created:', element);
            console.log('Total elements:', elements.length);
        }
        
        function moveElement(elementId, x, y) {
            console.log('Moving element:', elementId, 'to', x, y);
            const element = elements.find(el => el.id === elementId);
            if (element) {
                // Ensure element stays within bounds and accounts for element width/height
                const maxX = 100 - element.width;
                const maxY = 100 - element.height;
                
                // Use precise positioning with proper bounds checking
                const newX = Math.max(0, Math.min(maxX, x));
                const newY = Math.max(0, Math.min(maxY, y));
                
                // Update element position
                element.x = newX;
                element.y = newY;
                
                // Update the element in the elements array
                const elementIndex = elements.findIndex(el => el.id === elementId);
                if (elementIndex !== -1) {
                    elements[elementIndex] = { ...element };
                }
                
                // Update visual position immediately
                const elementDiv = document.getElementById(element.id);
                if (elementDiv) {
                    elementDiv.style.left = newX + '%';
                    elementDiv.style.top = newY + '%';
                }
                
                // Update properties panel if this element is selected
                if (selectedElement && selectedElement.id === elementId) {
                    document.getElementById('posX').value = newX;
                    document.getElementById('posY').value = newY;
                }
                
                updateStatus(`Moved element to ${Math.round(element.x * 100) / 100}%, ${Math.round(element.y * 100) / 100}%`);
            }
        }

        function getDefaultText(type) {
            const defaults = {
                // Paradigm Brain Profile
                paradigm_type: '{{ $assessmentData->paradigm_type ?? "The Analyst" }}',
                dominance_category: '{{ $assessmentData->dominance_category ?? "Single-Frame" }}',

                // Brain Frames (Part 1)
                lu_percentage: '{{ round($assessmentData->lu_percentage ?? 73) }}',
                ll_percentage: '{{ round($assessmentData->ll_percentage ?? 58) }}',
                ru_percentage: '{{ round($assessmentData->ru_percentage ?? 76) }}',
                rl_percentage: '{{ round($assessmentData->rl_percentage ?? 71) }}',

                // Brain Fields (Part 2)
                question_asker_percentage: '{{ round($assessmentData->question_asker_percentage ?? 65) }}',
                fact_finder_percentage: '{{ round($assessmentData->fact_finder_percentage ?? 80) }}',
                rule_keeper_percentage: '{{ round($assessmentData->rule_keeper_percentage ?? 70) }}',
                task_manager_percentage: '{{ round($assessmentData->task_manager_percentage ?? 60) }}',
                creator_percentage: '{{ round($assessmentData->creator_percentage ?? 85) }}',
                big_thinker_percentage: '{{ round($assessmentData->big_thinker_percentage ?? 78) }}',
                friend_maker_percentage: '{{ round($assessmentData->friend_maker_percentage ?? 90) }}',
                empath_percentage: '{{ round($assessmentData->empath_percentage ?? 82) }}',

                // Part 3: Personality Blueprint (E/I) - Percentages
                e_score: '70',
                i_score: '30',

                // Part 4: Conflict Styles (Ranks 1-5)
                assertive_challenger: '2',
                integrative_bridge_builder: '5',
                fair_negotiator: '1',
                quiet_protector: '4',
                peacekeeper: '3',

                // Part 5: Connection Styles (Ranks 1-5)
                verbal_validator: '3',
                presence_seeker: '1',
                symbolic_appreciator: '5',
                practical_supporter: '2',
                grounded_connector: '4',

                // Part 6: Intelligence/Quotient (Checkboxes)
                si_checkbox: '☑ Low ☐ Medium ☐ High',
                eq_checkbox: '☐ Low ☑ Medium ☐ High',
                rq_checkbox: '☐ Low ☐ Medium ☑ High',

                // Part 7: Paradigm Balance (Wellness - Numbers Only, NO %)
                emotional_balance: '85',
                spiritual_balance: '72',
                physical_balance: '90',
                social_balance: '65',
                financial_balance: '55',
                occupational_balance: '78',
                intellectual_balance: '82',
                environmental_balance: '68',
                overall_wellness: '74',

                // Personal Information
                name: '{{ $assessmentData->name ?? "John" }}',
                surname: '{{ $assessmentData->surname ?? "Doe" }}',
                email: '{{ $assessmentData->email ?? "participant@email.com" }}',
                gender: '{{ $assessmentData->gender ?? "Not specified" }}',
                age_range: '{{ $assessmentData->age_range ?? "25-34" }}',
                phone: '{{ $assessmentData->phone ?? "Not provided" }}',

                // Provider Information
                provider_name: '{{ $assessmentData->provider_name ?? "Dr. Sarah" }}',
                provider_last_name: '{{ $assessmentData->provider_last_name ?? "Johnson" }}',
                provider_email: '{{ $assessmentData->provider_email ?? "provider@clinic.com" }}',
                provider_phone: '{{ $assessmentData->provider_phone ?? "+1 555-0123" }}',

                // Assessment Details
                date_completed: '{{ $assessmentData->completed_at ?? "2025-10-04" }}',
                
                // NEW ELEMENTS - Auto-sizing
                full_name: '{{ ($assessmentData->name ?? "John") }} {{ ($assessmentData->surname ?? "Doe") }}',
                user_details_block: 'Name: {{ ($assessmentData->name ?? "John") }} {{ ($assessmentData->surname ?? "Doe") }}\\nEmail: {{ $assessmentData->email ?? "participant@email.com" }}\\nPhone: {{ $assessmentData->phone ?? "Not provided" }}\\nGender: {{ $assessmentData->gender ?? "Not specified" }}\\nAge: {{ $assessmentData->age_range ?? "25-34" }}',
                provider_details_block: 'Name: {{ ($assessmentData->provider_name ?? "Dr. Sarah") }} {{ ($assessmentData->provider_last_name ?? "Johnson") }}\\nEmail: {{ $assessmentData->provider_email ?? "provider@clinic.com" }}\\nPhone: {{ $assessmentData->provider_phone ?? "+1 555-0123" }}',
            };
            return defaults[type] || 'Paradigm Element';
        }
        
        function getDefaultWidth(type) {
            const widths = {
                paradigm_type: 25,
                dominance_category: 25,
                lu_percentage: 10,
                ll_percentage: 10,
                ru_percentage: 10,
                rl_percentage: 10,
                question_asker_percentage: 15,
                fact_finder_percentage: 15,
                rule_keeper_percentage: 15,
                task_manager_percentage: 15,
                creator_percentage: 15,
                big_thinker_percentage: 15,
                friend_maker_percentage: 15,
                empath_percentage: 15,
                name: 20,
                surname: 20,
                email: 30,
                gender: 15,
                age_range: 15,
                phone: 25,
                date_completed: 20,
            };
            return widths[type] || 20;
        }
        
        function getDefaultHeight(type) {
            const heights = {
                paradigm_type: 5,
                dominance_category: 5,
                lu_percentage: 4,
                ll_percentage: 4,
                ru_percentage: 4,
                rl_percentage: 4,
                question_asker_percentage: 4,
                fact_finder_percentage: 4,
                rule_keeper_percentage: 4,
                task_manager_percentage: 4,
                creator_percentage: 4,
                big_thinker_percentage: 4,
                friend_maker_percentage: 4,
                empath_percentage: 4,
                name: 5,
                surname: 5,
                email: 5,
                gender: 5,
                age_range: 5,
                phone: 5,
                date_completed: 5,
            };
            return heights[type] || 5;
        }
        
        function getDefaultFontSize(type) {
            const sizes = {
                paradigm_type: 16,
                dominance_category: 14,
                lu_percentage: 12,
                ll_percentage: 12,
                ru_percentage: 12,
                rl_percentage: 12,
                question_asker_percentage: 12,
                fact_finder_percentage: 12,
                rule_keeper_percentage: 12,
                task_manager_percentage: 12,
                creator_percentage: 12,
                big_thinker_percentage: 12,
                friend_maker_percentage: 12,
                empath_percentage: 12,
                name: 14,
                surname: 14,
                email: 10,
                gender: 10,
                age_range: 10,
                phone: 10,
                date_completed: 10,
            };
            return sizes[type] || 12;
        }
        
        function getDefaultTextColor(type) {
            const colors = {
                paradigm_type: '#667eea',
                dominance_category: '#333333',
                lu_percentage: '#000000',
                ll_percentage: '#000000',
                ru_percentage: '#000000',
                rl_percentage: '#000000',
                name: '#000000',
                surname: '#000000',
                email: '#666666',
            };
            return colors[type] || '#000000';
        }
        
        function getDefaultBackgroundColor(type) {
            // All elements have transparent background by default
            return 'transparent';
        }
        
        function getElementDisplayName(type) {
            const names = {
                paradigm_type: 'Paradigm Type',
                dominance_category: 'Dominance Category',
                lu_percentage: 'LU Score (%)',
                ll_percentage: 'LL Score (%)',
                ru_percentage: 'RU Score (%)',
                rl_percentage: 'RL Score (%)',
                question_asker_percentage: 'Question Asker (%)',
                fact_finder_percentage: 'Fact Finder (%)',
                rule_keeper_percentage: 'Rule Keeper (%)',
                task_manager_percentage: 'Task Manager (%)',
                creator_percentage: 'Creator (%)',
                big_thinker_percentage: 'Big Thinker (%)',
                friend_maker_percentage: 'Friend Maker (%)',
                empath_percentage: 'Empath (%)',
                e_score: 'E Score',
                i_score: 'I Score',
                assertive_challenger: 'Assertive Challenger',
                integrative_bridge_builder: 'Integrative Bridge Builder',
                fair_negotiator: 'Fair Negotiator',
                quiet_protector: 'Quiet Protector',
                peacekeeper: 'Peacekeeper',
                verbal_validator: 'Verbal Validator',
                presence_seeker: 'Presence Seeker',
                symbolic_appreciator: 'Symbolic Appreciator',
                practical_supporter: 'Practical Supporter',
                grounded_connector: 'Grounded Connector',
                si_checkbox: 'Social Intelligence (Checkbox)',
                eq_checkbox: 'Emotional Intelligence (Checkbox)',
                rq_checkbox: 'Relevancy Quotient (Checkbox)',
                emotional_balance: 'Emotional Balance',
                spiritual_balance: 'Spiritual Balance',
                physical_balance: 'Physical Balance',
                social_balance: 'Social Balance',
                financial_balance: 'Financial Balance',
                occupational_balance: 'Occupational Balance',
                intellectual_balance: 'Intellectual Balance',
                environmental_balance: 'Environmental Balance',
                overall_wellness: 'Overall Wellness Score',
                name: 'First Name',
                surname: 'Last Name',
                email: 'Email Address',
                gender: 'Gender',
                age_range: 'Age Range',
                phone: 'Phone Number',
                provider_name: 'Provider First Name',
                provider_last_name: 'Provider Last Name',
                provider_email: 'Provider Email',
                provider_phone: 'Provider Phone',
                date_completed: 'Completion Date',
                // NEW ELEMENTS - Auto-sizing
                full_name: 'Full Name (Auto-size)',
                user_details_block: 'User Details Block (Auto-size)',
                provider_details_block: 'Provider Details Block (Auto-size)',
            };
            return names[type] || type;
        }

        function renderElement(element) {
            console.log('Rendering element:', element);
            const pdfPage = document.getElementById('pdfPage');
            const noElements = document.getElementById('noElements');
            
            if (!pdfPage) {
                console.error('PDF page element not found!');
                return;
            }
            
            if (noElements) {
                noElements.remove();
            }

            const elementDiv = document.createElement('div');
            elementDiv.className = 'pdf-element';
            elementDiv.id = element.id;
            elementDiv.style.left = element.x + '%';
            elementDiv.style.top = element.y + '%';
            elementDiv.style.width = element.width + '%';
            // Use original height but ensure minimum for text readability
            const minHeight = Math.max(element.height, 2); // Minimum 2% height
            elementDiv.style.height = minHeight + '%';
            elementDiv.style.fontSize = element.fontSize + 'px';
            elementDiv.style.color = element.textColor;
            elementDiv.style.backgroundColor = 'transparent';
            elementDiv.style.textAlign = element.textAlign;
            // Ensure text fits properly and handles overflow
            elementDiv.style.display = 'flex';
            elementDiv.style.alignItems = 'center';
            elementDiv.style.justifyContent = element.textAlign === 'center' ? 'center' : 
                                            element.textAlign === 'right' ? 'flex-end' : 'flex-start';
            elementDiv.style.overflow = 'hidden';
            elementDiv.style.textOverflow = 'ellipsis';
            elementDiv.style.whiteSpace = 'normal';
            elementDiv.style.wordWrap = 'break-word';
            elementDiv.style.lineHeight = '1.2';
            
            // Handle different element types
            if (element.type === 'image') {
                elementDiv.style.backgroundImage = `url(${element.src})`;
                elementDiv.style.backgroundSize = 'contain';
                elementDiv.style.backgroundRepeat = 'no-repeat';
                elementDiv.style.backgroundPosition = 'center';
            } else if (element.type === 'user_details_block' || element.type === 'provider_details_block') {
                // RIGHT-TO-LEFT AUTO-SIZING: Fixed alignment for PDF builder
                elementDiv.style.fontFamily = "'Courier New', monospace";
                elementDiv.style.textAlign = 'right';
                elementDiv.style.alignItems = 'flex-end';
                elementDiv.style.display = 'block';
                elementDiv.style.lineHeight = '1.4';
                
                const textContent = getDefaultText(element.type);
                const lines = textContent.split('\\n');

                // Find the longest label to pad others to its length
                let maxLabelLength = 0;
                lines.forEach(line => {
                    const label = line.split(':')[0] + ':';
                    if (label.length > maxLabelLength) {
                        maxLabelLength = label.length;
                    }
                });

                let contentHTML = '';
                lines.forEach(line => {
                    const parts = line.split(':');
                    const label = parts.shift() + ':';
                    const value = parts.join(':').trim();
                    // Use padStart to add spaces to the left of shorter labels
                    const paddedLabel = label.padStart(maxLabelLength, ' ');
                    
                    contentHTML += `<div style="white-space: pre; overflow: hidden; text-overflow: ellipsis;"><span style="font-weight: bold;">${paddedLabel}</span> ${value}</div>`;
                });
                
                elementDiv.innerHTML = contentHTML;
            } else {
                // Default text rendering
                const text = getDefaultText(element.type);
                // Replace newline characters with <br> for HTML rendering
                elementDiv.innerHTML = text.replace(/\\n/g, '<br>');
            }

            // Add resize handles
            const resizeHandles = ['se', 'ne', 'sw', 'nw'];
            resizeHandles.forEach(handle => {
                const resizeHandle = document.createElement('div');
                resizeHandle.className = `resize-handle ${handle}`;
                resizeHandle.addEventListener('mousedown', (e) => startResize(e, element, handle));
                elementDiv.appendChild(resizeHandle);
            });

            // Add click-to-move functionality
            let isMoving = false;
            let startX, startY, startElementX, startElementY;
            let mouseDownTime = 0;
            
            elementDiv.addEventListener('mousedown', function(e) {
                if (e.target.classList.contains('resize-handle')) return; // Don't move when resizing
                mouseDownTime = Date.now();
                
                isMoving = true;
                startX = e.clientX;
                startY = e.clientY;
                startElementX = element.x;
                startElementY = element.y;
                
                this.style.cursor = 'grabbing';
                this.style.zIndex = '1000';
                
                e.preventDefault();
            });
            
            document.addEventListener('mousemove', function(e) {
                if (!isMoving) return;
                
                const pdfPage = document.getElementById('pdfPage');
                const rect = pdfPage.getBoundingClientRect();
                
                // Calculate movement delta
                const deltaX = e.clientX - startX;
                const deltaY = e.clientY - startY;
                
                // ULTRA-PRECISE: Convert pixel movement to percentage accounting for scale
                // Get the actual scale factor from the CSS transform
                const computedStyle = window.getComputedStyle(pdfPage);
                const transform = computedStyle.transform;
                let scale = 0.6; // Default scale
                
                if (transform && transform !== 'none') {
                    const matrix = transform.match(/matrix\(([^)]+)\)/);
                    if (matrix) {
                        const values = matrix[1].split(',');
                        scale = parseFloat(values[0]) || 0.6;
                    }
                }
                
                // Calculate movement relative to the actual unscaled canvas
                const deltaXPercent = (deltaX / (rect.width / scale)) * 100;
                const deltaYPercent = (deltaY / (rect.height / scale)) * 100;
                
                // Calculate new position
                const newX = startElementX + deltaXPercent;
                const newY = startElementY + deltaYPercent;
                
                // Ensure element stays within bounds
                const maxX = 100 - element.width;
                const maxY = 100 - element.height;
                
                const clampedX = Math.max(0, Math.min(maxX, newX));
                const clampedY = Math.max(0, Math.min(maxY, newY));
                
                // Update element position
                element.x = clampedX;
                element.y = clampedY;
                
                // Update visual position
                const elementDiv = document.getElementById(element.id);
                if (elementDiv) {
                    elementDiv.style.left = clampedX + '%';
                    elementDiv.style.top = clampedY + '%';
                }
            });
            
            document.addEventListener('mouseup', function(e) {
                if (!isMoving) return;
                
                isMoving = false;
                const elementDiv = document.getElementById(element.id);
                if (elementDiv) {
                    elementDiv.style.cursor = 'grab';
                    elementDiv.style.zIndex = '1';
                }
                
                // Update properties panel if this element is selected
                if (selectedElement && selectedElement.id === element.id) {
                    document.getElementById('posX').value = element.x;
                    document.getElementById('posY').value = element.y;
                }
                
                updateStatus(`Moved element to ${Math.round(element.x * 100) / 100}%, ${Math.round(element.y * 100) / 100}%`);
            });
            
            // Add click event for selection (only if it wasn't a drag)
            elementDiv.addEventListener('click', function(e) {
                e.stopPropagation();
                // Only select if it wasn't a drag operation (mouse was down for less than 200ms)
                if (Date.now() - mouseDownTime < 200) {
                    selectElement(element);
                }
            });
            
            // Make element draggable
            // Use mousedown for element movement (removed dragstart/dragend to prevent conflicts)
            elementDiv.style.cursor = 'grab';

            pdfPage.appendChild(elementDiv);
        }

        function selectElement(element) {
            // Remove previous selection
            document.querySelectorAll('.pdf-element').forEach(el => {
                el.classList.remove('selected');
            });

            // Select new element
            const elementDiv = document.getElementById(element.id);
            if (elementDiv) {
                elementDiv.classList.add('selected');
            }

            selectedElement = element;
            showElementProperties(element);
            updateStatus(`Selected ${element.type} element`);
        }

        function showElementProperties(element) {
            document.getElementById('noSelection').classList.add('hidden');
            document.getElementById('elementProperties').classList.remove('hidden');

            document.getElementById('elementType').value = element.type;
            document.getElementById('elementText').value = element.text;
            document.getElementById('fontSize').value = element.fontSize;
            document.getElementById('fontSizeValue').textContent = element.fontSize + 'px';
            document.getElementById('textColor').value = element.textColor;
            document.getElementById('textAlign').value = element.textAlign;
            document.getElementById('posX').value = element.x;
            document.getElementById('posY').value = element.y;
            document.getElementById('width').value = element.width;
            document.getElementById('height').value = element.height;
        }

        function updateElement() {
            if (!selectedElement) return;

            selectedElement.text = document.getElementById('elementText').value;
            selectedElement.fontSize = parseInt(document.getElementById('fontSize').value);
            selectedElement.textColor = document.getElementById('textColor').value;
            selectedElement.backgroundColor = 'transparent';
            selectedElement.textAlign = document.getElementById('textAlign').value;
            selectedElement.x = parseFloat(document.getElementById('posX').value);
            selectedElement.y = parseFloat(document.getElementById('posY').value);
            selectedElement.width = parseFloat(document.getElementById('width').value);
            selectedElement.height = parseFloat(document.getElementById('height').value);

            // Keep original height - don't auto-adjust based on font size
            // This prevents layout issues with overlapping elements

            renderElement(selectedElement);
            selectElement(selectedElement);
            updateStatus('Element updated');
        }

        function deleteElement() {
            if (!selectedElement) return;

            const elementDiv = document.getElementById(selectedElement.id);
            if (elementDiv) {
                elementDiv.remove();
            }

            elements = elements.filter(el => el.id !== selectedElement.id);
            selectedElement = null;

            document.getElementById('noSelection').classList.remove('hidden');
            document.getElementById('elementProperties').classList.add('hidden');

            if (elements.length === 0) {
                const pdfPage = document.getElementById('pdfPage');
                const noElements = document.createElement('div');
                noElements.id = 'noElements';
                noElements.className = 'no-elements';
                noElements.innerHTML = '<i class="ri-drag-drop-line"></i><div>Drag elements here to start building</div>';
                pdfPage.appendChild(noElements);
            }

            updateStatus('Element deleted');
        }

        function clearCanvas() {
            if (confirm('Are you sure you want to clear all elements?')) {
                elements = [];
                selectedElement = null;
                document.getElementById('pdfPage').innerHTML = '<div class="no-elements" id="noElements"><i class="ri-drag-drop-line"></i><div>Drag elements here to start building</div></div>';
                document.getElementById('noSelection').classList.remove('hidden');
                document.getElementById('elementProperties').classList.add('hidden');
                updateStatus('Canvas cleared');
            }
        }

        function saveElements() {
            try {
                // Get or create page record for current page
                let pageId = null;
                let pageNumber = currentPageId;
                
                // Check if page exists in database
                const allPages = @json($template->pdfPages ?? []);
                const existingPage = allPages.find(p => p.page_number == currentPageId);
                
                if (existingPage) {
                    pageId = existingPage.id;
                } else {
                    // Create new page record
                    updateStatus('Creating new page record...');
                    // For now, we'll use the first page ID and handle page creation separately
                    pageId = {{ $template->pdfPages->first()->id ?? 'null' }};
                }
                
                if (!pageId) {
                    updateStatus('Error: No PDF page found for this template');
                    return;
                }
                
                console.log('Saving elements:', elements);
                console.log('Page ID:', pageId);
                console.log('Current Page ID:', currentPageId);
                console.log('Template ID:', {{ $template->id }});
                
                // Filter elements for current page only
                console.log('All elements:', elements);
                console.log('Element pageIds:', elements.map(el => el.pageId));
                console.log('Current page ID for filtering:', currentPageId);
                console.log('Type of currentPageId:', typeof currentPageId);
                console.log('Type of element pageId:', elements.length > 0 ? typeof elements[0].pageId : 'no elements');
                
                const currentPageElements = elements.filter(el => String(el.pageId) === String(currentPageId));
                console.log('Current page elements:', currentPageElements);
                
                const elementsData = currentPageElements.map((element, index) => ({
                    id: element.db_id || null, // Pass existing DB ID if available
                    element_type: element.type,
                    element_key: element.type, // Add element_key
                    text_template: element.text,
                    x_pct: element.x,
                    y_pct: element.y,
                    w_pct: element.width,
                    h_pct: element.height,
                    font_px: element.fontSize,
                    font_family: 'Arial',
                    font_style: 'normal',
                    text_align: element.textAlign,
                    font_weight: 400,
                    color: element.textColor,
                    bg_color: 'transparent',
                    fg_color: null,
                    display_order: index + 1
                }));

                console.log('Elements data to send:', elementsData);

                fetch(`{{ route('admin.paradigm.paradigm-pdf-builder.save-elements') }}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({
                        template_id: {{ $template->id }},
                        page_number: currentPageId,
                        elements: elementsData,
                        background_image: pageBackgrounds[currentPageId] || null // Send staged background change
                    })
                })
                .then(response => {
                    console.log('Raw response received');
                    if (!response.ok) {
                        return response.text().then(text => {
                            throw new Error(`Server error: ${response.status} - ${text}`);
                        });
                    }
                    return response.json();
                })
                .then(data => {
                    console.log('Parsed response:', data);
                    if (data.success) {
                        updateStatus('Elements saved successfully! Reloading...');
                        // Reload to get fresh data from the server
                        setTimeout(() => {
                            window.location.reload();
                        }, 1000);
                    } else {
                        updateStatus('Error saving elements: ' + data.message);
                        alert('Save failed: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Save error:', error);
                    updateStatus('Fatal error saving elements: ' + error.message);
                    alert('A fatal error occurred during save. Check the console for details.');
                });
            } catch (error) {
                console.error('Save function error:', error);
                updateStatus('Error saving elements: ' + error.message);
            }
        }

        function loadExistingElements() {
            const allPages = @json($template->pdfPages->keyBy('page_number')->toArray() ?? []);
            const existingElementsJson = `{!! addslashes(json_encode($template->pdfPages->flatMap(function($page) { return $page->pdfElements; }))) !!}`;
            let existingElements = [];
            try {
                existingElements = JSON.parse(existingElementsJson);
            } catch(e) {
                console.error("Could not parse existing elements JSON:", e);
            }

            console.log('All pages loaded:', allPages);
            console.log('Existing backgrounds:', Object.keys(allPages).reduce((acc, pageNum) => {
                if(allPages[pageNum].background_image_path) {
                    acc[pageNum] = allPages[pageNum].background_image_path;
                }
                return acc;
            }, {}));
            console.log('Raw Existing elements:', existingElements);

            // Clear local elements before loading
            elements = [];
            
            existingElements.forEach(elementData => {
                elements.push({
                    db_id: elementData.id,
                    id: `element_${elementData.id}`,
                    type: elementData.element_type,
                    x: parseFloat(elementData.x_pct),
                    y: parseFloat(elementData.y_pct),
                    width: parseFloat(elementData.w_pct),
                    height: parseFloat(elementData.h_pct),
                    text: elementData.text_template || getDefaultText(elementData.element_type),
                    fontSize: parseInt(elementData.font_px),
                    textColor: elementData.color,
                    textAlign: elementData.text_align,
                    pageId: elementData.page_number
                });
            });
            
            // Load background images from the server data
            Object.keys(allPages).forEach(pageNumber => {
                const page = allPages[pageNumber];
                if (page.background_image_path) {
                    const fullUrl = `{{ asset('storage') }}/${page.background_image_path}`;
                    pageBackgrounds[pageNumber] = fullUrl;
                    console.log(`Loaded background for page ${pageNumber}: ${fullUrl}`);
                }
            });

            renderCurrentPage();
            updatePageDisplay();

            if (elements.length > 0) {
                updateStatus(`Loaded ${elements.length} existing elements.`);
            }
        }

        function renderCurrentPage() {
            const pdfPage = document.getElementById('pdfPage');
            pdfPage.innerHTML = '';
            
            const pageElements = elements.filter(el => String(el.pageId) === String(currentPageId));
            
            if (pageElements.length === 0) {
                pdfPage.innerHTML = `<div class="no-elements" id="noElements"><i class="ri-drag-drop-line"></i><div>Drag elements here to start building</div></div>`;
            } else {
                pageElements.forEach(element => {
                    renderElement(element);
                });
            }

            // Apply background for the current page
            if (pageBackgrounds[currentPageId]) {
                 // Check if it's a base64 string or a URL
                const backgroundUrl = pageBackgrounds[currentPageId].startsWith('data:image') 
                    ? pageBackgrounds[currentPageId] 
                    : pageBackgrounds[currentPageId];
                pdfPage.style.backgroundImage = `url('${backgroundUrl}')`;
                pdfPage.style.backgroundSize = '100% 100%';
                pdfPage.style.backgroundPosition = 'center';
                pdfPage.style.backgroundRepeat = 'no-repeat';
            } else {
                pdfPage.style.backgroundImage = 'none';
            }
        }

        function updateStatus(message) {
            document.getElementById('statusText').textContent = message;
        }

        // Font size slider
        document.getElementById('fontSize').addEventListener('input', function() {
            document.getElementById('fontSizeValue').textContent = this.value + 'px';
            if (selectedElement) {
                selectedElement.fontSize = parseInt(this.value);
                renderElement(selectedElement);
                selectElement(selectedElement);
            }
        });

        // Click outside to deselect
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.pdf-element') && !e.target.closest('#elementProperties')) {
                selectedElement = null;
                document.querySelectorAll('.pdf-element').forEach(el => {
                    el.classList.remove('selected');
                });
                document.getElementById('noSelection').classList.remove('hidden');
                document.getElementById('elementProperties').classList.add('hidden');
            }
        });
        
        // Page Management Functions
        function addPage() {
            const newPageId = Math.max(...pages) + 1;
            pages.push(newPageId);
            currentPageId = newPageId;
            
            // Initialize new page without background
            pageBackgrounds[newPageId] = null;
            
            updatePageDisplay();
            clearCurrentPage();
            
            // Clear background for new page
            const pdfPage = document.getElementById('pdfPage');
            pdfPage.style.backgroundImage = 'none';
            pdfPage.style.backgroundSize = '';
            pdfPage.style.backgroundPosition = '';
            pdfPage.style.backgroundRepeat = '';
            
            updateStatus(`Added page ${newPageId}`);
        }

        function nextPage() {
            console.log('Next page clicked');
            console.log('Current page ID:', currentPageId);
            console.log('Pages array:', pages);
            const currentIndex = pages.indexOf(currentPageId);
            console.log('Current index:', currentIndex);
            if (currentIndex < pages.length - 1) {
                console.log('Switching to page:', pages[currentIndex + 1]);
                switchPage(pages[currentIndex + 1]);
            } else {
                console.log('Already on last page');
            }
        }

        function previousPage() {
            console.log('Previous page clicked');
            console.log('Current page ID:', currentPageId);
            console.log('Pages array:', pages);
            const currentIndex = pages.indexOf(currentPageId);
            console.log('Current index:', currentIndex);
            if (currentIndex > 0) {
                console.log('Switching to page:', pages[currentIndex - 1]);
                switchPage(pages[currentIndex - 1]);
            } else {
                console.log('Already on first page');
            }
        }
        
        function removePage() {
            if (pages.length <= 1) return;
            
            const pageIndex = pages.indexOf(currentPageId);
            if (pageIndex > -1) {
                const pageToRemove = currentPageId;
                pages.splice(pageIndex, 1);
                
                // Remove elements from this page
                elements = elements.filter(el => el.pageId !== pageToRemove);
                
                // Remove background for this page
                delete pageBackgrounds[pageToRemove];
                
                // Switch to previous page
                currentPageId = pages[Math.max(0, pageIndex - 1)];
                updatePageDisplay();
                renderCurrentPage();
                
                // Apply background for the switched page
                const pdfPage = document.getElementById('pdfPage');
                if (pageBackgrounds[currentPageId]) {
                    pdfPage.style.backgroundImage = `url(${pageBackgrounds[currentPageId]})`;
                    pdfPage.style.backgroundSize = 'cover';
                    pdfPage.style.backgroundPosition = 'center';
                    pdfPage.style.backgroundRepeat = 'no-repeat';
                } else {
                    pdfPage.style.backgroundImage = 'none';
                    pdfPage.style.backgroundSize = '';
                    pdfPage.style.backgroundPosition = '';
                    pdfPage.style.backgroundRepeat = '';
                }
                
                updateStatus(`Removed page, switched to page ${currentPageId}`);
            }
        }
        
        function switchPage(pageId) {
            console.log('Switching to page:', pageId);
            console.log('Current page ID before switch:', currentPageId);
            currentPageId = pageId;
            console.log('Current page ID after switch:', currentPageId);
            updatePageDisplay();
            renderCurrentPage();
            
            // Apply background for the switched page
            const pdfPage = document.getElementById('pdfPage');
            if (pageBackgrounds[pageId]) {
                pdfPage.style.backgroundImage = `url(${pageBackgrounds[pageId]})`;
                pdfPage.style.backgroundSize = 'cover';
                pdfPage.style.backgroundPosition = 'center';
                pdfPage.style.backgroundRepeat = 'no-repeat';
            } else {
                pdfPage.style.backgroundImage = 'none';
                pdfPage.style.backgroundSize = '';
                pdfPage.style.backgroundPosition = '';
                pdfPage.style.backgroundRepeat = '';
            }
            
            updateStatus(`Switched to page ${pageId}`);
        }
        
        function updatePageDisplay() {
            document.getElementById('currentPage').textContent = currentPageId;
            document.getElementById('totalPages').textContent = pages.length;
            document.getElementById('removePageBtn').disabled = pages.length <= 1;
            
            // Enable/disable navigation buttons
            const currentIndex = pages.indexOf(currentPageId);
            document.getElementById('prevPageBtn').disabled = currentIndex <= 0;
            document.getElementById('nextPageBtn').disabled = currentIndex >= pages.length - 1;
        }
        
        function clearCurrentPage() {
            const pdfPage = document.getElementById('pdfPage');
            pdfPage.innerHTML = '<div class="no-elements" id="noElements"><i class="ri-drag-drop-line"></i><div>Drag elements here to start building</div></div>';
        }
        
        // Image upload functionality
        function handleImageUpload(event) {
            const file = event.target.files[0];
            if (file && file.type.startsWith('image/')) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    if (selectedElement && selectedElement.type === 'image') {
                        selectedElement.imageUrl = e.target.result;
                        renderElement(selectedElement);
                        updateStatus('Image updated');
                    }
                };
                reader.readAsDataURL(file);
            }
        }
        
        // Mobile sidebar toggle
        function toggleMobileSidebar() {
            const sidebar = document.querySelector('.sidebar');
            const properties = document.querySelector('.properties-panel');
            const toggle = document.getElementById('mobileToggle');
            
            // Check if all elements exist before proceeding
            if (!sidebar || !properties || !toggle) {
                console.warn('Mobile sidebar elements not found');
                return;
            }
            
            if (sidebar.classList.contains('hidden-mobile')) {
                sidebar.classList.remove('hidden-mobile');
                properties.classList.add('hidden-mobile');
                toggle.innerHTML = '<i class="ri-settings-line"></i>';
            } else if (properties.classList.contains('hidden-mobile')) {
                sidebar.classList.add('hidden-mobile');
                properties.classList.remove('hidden-mobile');
                toggle.innerHTML = '<i class="ri-layout-2-line"></i>';
            } else {
                sidebar.classList.add('hidden-mobile');
                properties.classList.add('hidden-mobile');
                toggle.innerHTML = '<i class="ri-menu-line"></i>';
            }
        }
        
        // Resize functionality
        let isResizing = false;
        let resizeElement = null;
        let resizeHandle = null;
        let startX, startY, startWidth, startHeight, startLeft, startTop;

        function startResize(e, element, handle) {
            e.preventDefault();
            e.stopPropagation();
            
            isResizing = true;
            resizeElement = element;
            resizeHandle = handle;
            
            const rect = e.target.closest('.pdf-element').getBoundingClientRect();
            const pdfPage = document.getElementById('pdfPage');
            const pdfRect = pdfPage.getBoundingClientRect();
            
            startX = e.clientX;
            startY = e.clientY;
            startWidth = element.width;
            startHeight = element.height;
            startLeft = element.x;
            startTop = element.y;
            
            document.addEventListener('mousemove', handleResize);
            document.addEventListener('mouseup', stopResize);
            
            e.target.closest('.pdf-element').classList.add('resizing');
        }

        function handleResize(e) {
            if (!isResizing) return;
            
            const deltaX = e.clientX - startX;
            const deltaY = e.clientY - startY;
            
            const pdfPage = document.getElementById('pdfPage');
            const pdfRect = pdfPage.getBoundingClientRect();
            
            // ULTRA-PRECISE: Convert pixel changes to percentage changes accounting for scale
            // Get the actual scale factor from the CSS transform
            const computedStyle = window.getComputedStyle(pdfPage);
            const transform = computedStyle.transform;
            let scale = 0.6; // Default scale
            
            if (transform && transform !== 'none') {
                const matrix = transform.match(/matrix\(([^)]+)\)/);
                if (matrix) {
                    const values = matrix[1].split(',');
                    scale = parseFloat(values[0]) || 0.6;
                }
            }
            
            // Calculate size changes relative to the actual unscaled canvas
            const scaleX = (deltaX / (pdfRect.width / scale)) * 100;
            const scaleY = (deltaY / (pdfRect.height / scale)) * 100;
            
            let newWidth = startWidth;
            let newHeight = startHeight;
            let newLeft = startLeft;
            let newTop = startTop;
            
            // Handle width changes
            if (resizeHandle.includes('e')) {
                newWidth = Math.max(5, startWidth + scaleX);
            }
            if (resizeHandle.includes('w')) {
                newWidth = Math.max(5, startWidth - scaleX);
                newLeft = startLeft + (startWidth - newWidth);
            }
            
            // Handle height changes
            if (resizeHandle.includes('s')) {
                newHeight = Math.max(3, startHeight + scaleY);
            }
            if (resizeHandle.includes('n')) {
                newHeight = Math.max(3, startHeight - scaleY);
                newTop = startTop + (startHeight - newHeight);
            }
            
            // Update element with proper bounds
            resizeElement.width = Math.min(95, Math.max(5, newWidth));
            resizeElement.height = Math.min(95, Math.max(3, newHeight));
            resizeElement.x = Math.min(95, Math.max(0, newLeft));
            resizeElement.y = Math.min(95, Math.max(0, newTop));
            
            // Update the element in the elements array
            const elementIndex = elements.findIndex(el => el.id === resizeElement.id);
            if (elementIndex !== -1) {
                elements[elementIndex] = { ...resizeElement };
            }
            
            // Re-render the current page to maintain selection
            renderCurrentPage();
            
            // Re-select the element to show resize handles
            setTimeout(() => {
                selectElement(resizeElement);
            }, 10);
        }

        function stopResize() {
            isResizing = false;
            
            // Keep the element selected after resize
            if (resizeElement) {
                selectedElement = resizeElement;
                selectElement(resizeElement);
            }
            
            resizeElement = null;
            resizeHandle = null;
            
            document.removeEventListener('mousemove', handleResize);
            document.removeEventListener('mouseup', stopResize);
            
            document.querySelectorAll('.pdf-element').forEach(el => {
                el.classList.remove('resizing');
            });
        }

        // Page background functionality
        function setPageBackground() {
            const pageId = getCurrentPageId();
            if (!pageId) {
                alert('Could not determine the current page. Please save and reload.');
                return;
            }

            const input = document.createElement('input');
            input.type = 'file';
            input.accept = 'image/*';
            input.onchange = (e) => {
                const file = e.target.files[0];
                if (!file) return;

                compressImage(file, 0.7, (base64data) => {
                    pageBackgrounds[currentPageId] = base64data; // Store base64 data locally
                    
                    const pdfPage = document.getElementById('pdfPage');
                    pdfPage.style.backgroundImage = `url('${base64data}')`;
                    pdfPage.style.backgroundSize = '100% 100%';
                    pdfPage.style.backgroundPosition = 'center';
                    pdfPage.style.backgroundRepeat = 'no-repeat';

                    updateStatus(`Staged background for Page ${currentPageId}. Click Save to apply.`);
                });
            };
            input.click();
        }

        function clearPageBackground() {
            if (!confirm('Are you sure you want to remove the background for this page? This cannot be undone until you save.')) {
                return;
            }

            pageBackgrounds[currentPageId] = 'DELETE'; // Mark for deletion on save

            const pdfPage = document.getElementById('pdfPage');
            pdfPage.style.backgroundImage = 'none';

            updateStatus(`Staged background removal for Page ${currentPageId}. Click Save to apply.`);
        }

        // Image compression function
        function compressImage(file, quality, callback) {
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            const img = new Image();
            
            img.onload = function() {
                // Set canvas dimensions (max 1200px width for PDF)
                const maxWidth = 1200;
                const maxHeight = 1600;
                let { width, height } = img;
                
                if (width > maxWidth) {
                    height = (height * maxWidth) / width;
                    width = maxWidth;
                }
                if (height > maxHeight) {
                    width = (width * maxHeight) / height;
                    height = maxHeight;
                }
                
                canvas.width = width;
                canvas.height = height;
                
                // Draw and compress
                ctx.drawImage(img, 0, 0, width, height);
                const compressedDataUrl = canvas.toDataURL('image/jpeg', quality); // 70% quality
                callback(compressedDataUrl);
            };
            
            const reader = new FileReader();
            reader.onload = function(e) {
                img.src = e.target.result;
            };
            reader.readAsDataURL(file);
        }

        // Preview functionality
        function previewPDF() {
            const previewModal = document.getElementById('previewModal');
            const previewContent = document.getElementById('previewContent');
            
            // Clear previous content
            previewContent.innerHTML = '';
            
            // Get all pages
            const allPages = [...new Set(elements.map(el => el.pageId))].sort();
            
            allPages.forEach((pageId, index) => {
                const pageDiv = document.createElement('div');
                pageDiv.className = 'preview-page';
                
                // Apply background if it exists for this page
                let backgroundStyle = 'background: white;';
                if (pageBackgrounds[pageId]) {
                    backgroundStyle = `
                        background-image: url(${pageBackgrounds[pageId]});
                        background-size: cover;
                        background-position: center;
                        background-repeat: no-repeat;
                    `;
                }
                
                pageDiv.style.cssText = `
                    width: 400px;
                    height: 600px;
                    ${backgroundStyle}
                    border: 1px solid #ddd;
                    margin: 20px auto;
                    position: relative;
                    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
                    page-break-after: always;
                    aspect-ratio: 2/3;
                `;
                
                // Add page number
                const pageNumber = document.createElement('div');
                pageNumber.textContent = `Page ${index + 1}`;
                pageNumber.style.cssText = `
                    position: absolute;
                    bottom: 10px;
                    right: 10px;
                    font-size: 12px;
                    color: #666;
                `;
                pageDiv.appendChild(pageNumber);
                
                // Add elements for this page
                const pageElements = elements.filter(el => el.pageId === pageId);
                pageElements.forEach(element => {
                    const elementDiv = document.createElement('div');
                    elementDiv.className = 'preview-element';
                    elementDiv.style.cssText = `
                        position: absolute;
                        left: ${element.x}%;
                        top: ${element.y}%;
                        width: ${element.width}%;
                        height: ${element.height}%;
                        font-size: ${element.fontSize}px;
                        color: ${element.textColor};
                        background: transparent;
                        border: 1px solid #ddd;
                        padding: 5px;
                        box-sizing: border-box;
                        overflow: hidden;
                        word-wrap: break-word;
                    `;
                    
                    if (element.type === 'image' && element.imageUrl) {
                        elementDiv.innerHTML = `<img src="${element.imageUrl}" style="width: 100%; height: 100%; object-fit: cover;">`;
                    } else {
                        elementDiv.textContent = element.text;
                    }
                    
                    pageDiv.appendChild(elementDiv);
                });
                
                previewContent.appendChild(pageDiv);
            });
            
            // Show modal
            previewModal.style.display = 'flex';
        }
        
        function closePreview() {
            document.getElementById('previewModal').style.display = 'none';
        }
        
        // Zoom functionality
        function zoomIn() {
            currentZoom = Math.min(3, currentZoom + 0.25);
            applyZoom();
        }
        
        function zoomOut() {
            currentZoom = Math.max(0.25, currentZoom - 0.25);
            applyZoom();
        }
        
        function resetZoom() {
            currentZoom = 1;
            applyZoom();
        }
        
        function applyZoom() {
            const pdfPage = document.getElementById('pdfPage');
            const zoomLevel = document.getElementById('zoomLevel');
            
            // Apply zoom to the PDF page
            pdfPage.style.transform = `scale(${currentZoom})`;
            pdfPage.style.transformOrigin = 'top center';
            
            // Update zoom level display
            zoomLevel.textContent = Math.round(currentZoom * 100) + '%';
            
            // Adjust canvas area to accommodate zoom
            const canvasArea = document.querySelector('.canvas-area');
            if (currentZoom > 1) {
                canvasArea.style.overflow = 'auto';
                canvasArea.style.padding = '20px';
            } else {
                canvasArea.style.overflow = 'hidden';
                canvasArea.style.padding = '0';
            }
        }
        
        // Keyboard shortcuts for zoom
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey || e.metaKey) {
                switch(e.key) {
                    case '=':
                    case '+':
                        e.preventDefault();
                        zoomIn();
                        break;
                    case '-':
                        e.preventDefault();
                        zoomOut();
                        break;
                    case '0':
                        e.preventDefault();
                        resetZoom();
                        break;
                }
            }
        });
        
        // Initialize page management
        document.addEventListener('DOMContentLoaded', function() {
            updatePageDisplay();
            applyZoom(); // Initialize zoom
        });

        function getCurrentPageId() {
            return currentPageId;
        }
    </script>
</body>
</html>
