<?php
/**
 * PayFast ITN Handler for LIVE Production Server
 * URL: https://app.iseekhelpoffice.com/payfast-itn.php
 * 
 * This is a standalone file that handles PayFast ITN notifications
 * without going through Laravel routing/middleware to avoid issues.
 */

// Respond immediately to PayFast (required!)
http_response_code(200);
header('Content-Type: text/plain');
echo "OK";

// Flush output immediately so PayFast gets the response
if (function_exists('fastcgi_finish_request')) {
    fastcgi_finish_request();
}

// Now process the ITN in the background
try {
    // Bootstrap Laravel minimally
    require __DIR__.'/../vendor/autoload.php';
    $app = require_once __DIR__.'/../bootstrap/app.php';
    $kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
    $kernel->bootstrap();
    
    // Log the ITN received
    \Illuminate\Support\Facades\Log::info('PayFast ITN received', [
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'data' => $_POST
    ]);
    
    // Validate required fields
    $requiredFields = ['m_payment_id', 'payment_status', 'amount_gross'];
    foreach ($requiredFields as $field) {
        if (!isset($_POST[$field])) {
            \Illuminate\Support\Facades\Log::error('PayFast ITN missing required field', [
                'missing_field' => $field,
                'data' => $_POST
            ]);
            exit;
        }
    }
    
    // Get transaction data
    $transactionId = $_POST['m_payment_id'];
    $paymentStatus = strtolower($_POST['payment_status']);
    $amountGross = $_POST['amount_gross'];
    
    // Find the transaction in database
    $wallet = \Illuminate\Support\Facades\DB::table('wallet_history')
        ->where('transaction_id', $transactionId)
        ->first();
    
    if (!$wallet) {
        \Illuminate\Support\Facades\Log::error('Transaction not found in database', [
            'transaction_id' => $transactionId
        ]);
        exit;
    }
    
    // Verify amount matches (security check)
    $expectedAmount = number_format($wallet->amount, 2, '.', '');
    $receivedAmount = number_format($amountGross, 2, '.', '');
    
    if ($expectedAmount !== $receivedAmount) {
        \Illuminate\Support\Facades\Log::error('Amount mismatch', [
            'transaction_id' => $transactionId,
            'expected' => $expectedAmount,
            'received' => $receivedAmount
        ]);
        exit;
    }
    
    // Update transaction status
    if ($paymentStatus === 'complete') {
        \Illuminate\Support\Facades\DB::table('wallet_history')
            ->where('transaction_id', $transactionId)
            ->update([
                'status' => 'completed',
                'updated_at' => now()
            ]);
        
        \Illuminate\Support\Facades\Log::info('Transaction marked as completed', [
            'transaction_id' => $transactionId,
            'user_id' => $wallet->user_id,
            'amount' => $wallet->amount,
            'credits' => $wallet->credits ?? $wallet->amount
        ]);
        
        // Send wallet top-up email (Template 103)
        try {
            $user = \Illuminate\Support\Facades\DB::table('users')
                ->where('id', $wallet->user_id)
                ->first();
            
            if ($user) {
                \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
                
                $template = \Illuminate\Support\Facades\DB::table('templates')
                    ->where('notification_type', 20) // Wallet Top-up Success
                    ->where('type', 1) // Email
                    ->where('status', 1) // Active
                    ->first();
                
                if ($template) {
                    // Get company settings
                    $settings = \Illuminate\Support\Facades\DB::table('global_settings')
                        ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no'])
                        ->pluck('value', 'key');
                    
                    $companyName = $settings['company_name'] ?? config('app.name');
                    $credits = $wallet->credits ?? $wallet->amount;
                    
                    // Replace placeholders
                    $subject = str_replace(
                        ['{{amount}}', '{{credits}}', '{{company_name}}'],
                        [$wallet->amount, $credits, $companyName],
                        $template->subject
                    );
                    
                    $content = str_replace(
                        ['{{user_name}}', '{{amount}}', '{{credits}}', '{{balance}}', '{{company_name}}'],
                        [$user->name, $wallet->amount, $credits, $credits, $companyName],
                        $template->content
                    );
                    
                    // Send email
                    \Illuminate\Support\Facades\Mail::raw($content, function ($message) use ($user, $subject) {
                        $message->to($user->email)
                                ->subject($subject);
                    });
                    
                    \Illuminate\Support\Facades\Log::info('Wallet top-up email sent', [
                        'user_id' => $user->id,
                        'email' => $user->email
                    ]);
                }
            }
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send wallet top-up email', [
                'error' => $e->getMessage()
            ]);
        }
        
    } else {
        \Illuminate\Support\Facades\Log::warning('Payment not complete', [
            'transaction_id' => $transactionId,
            'status' => $paymentStatus
        ]);
    }
    
} catch (\Exception $e) {
    \Illuminate\Support\Facades\Log::error('PayFast ITN processing failed', [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
}
?>
