/**
 * Search City Auto-Fill
 * Handles postal code auto-fill when city is selected in search forms
 */

class SearchCityAutoFill {
    constructor() {
        this.apiKey = null;
        this.cache = new Map();
        this.init();
    }

    async init() {
        try {
            const response = await fetch('/api/get-google-maps-key');
            const data = await response.json();
            if (data.success) {
                this.apiKey = data.api_key;
                this.attachEventListeners();
            } else {
                console.error('Google Maps API key not found. Postal code auto-fill disabled.');
            }
        } catch (error) {
            console.error('Failed to initialize search postal code auto-fill:', error);
        }
    }

    attachEventListeners() {
        const self = this;
        
        // Handle city selection in search forms
        $(document).on('change', '#search_city, #search_city_location', function() {
            const selectedCity = $(this).val();
            const selectedState = $('#search_state_location').val() || $('#search_state').val();
            const selectedCountry = $('#search_country_location').val() || $('#search_country').val();
            const citySelect = $(this);
            
            if (selectedCity && selectedCity !== '') {
                // Show postal code suggestion with state and country context
                self.suggestPostalCode(selectedCity, selectedState, selectedCountry, citySelect);
            }
        });
    }

    async suggestPostalCode(city, state, country, citySelectElement) {
        if (!this.apiKey) {
            console.warn('API key not available');
            return;
        }

        // Build full address for better geocoding
        let address = city;
        if (state) address += ', ' + state;
        if (country) address += ', ' + country;

        // Check cache first
        const cacheKey = address.toLowerCase();
        if (this.cache.has(cacheKey)) {
            this.showPostalCodeSuggestion(citySelectElement, this.cache.get(cacheKey));
            return;
        }

        try {
            // Try to get postal code for the full address
            const geocodeUrl = `https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${this.apiKey}`;

            const response = await fetch(geocodeUrl);
            const data = await response.json();

            if (data.status === 'OK' && data.results.length > 0) {
                // Extract postal code from the first result
                const result = data.results[0];
                const postalCodeComponent = result.address_components.find(component => 
                    component.types.includes('postal_code') || component.types.includes('postal_code_prefix')
                );

                if (postalCodeComponent) {
                    const postalCode = postalCodeComponent.long_name;
                    
                    // Cache the result
                    this.cache.set(cacheKey, postalCode);
                    
                    // Show suggestion
                    this.showPostalCodeSuggestion(citySelectElement, postalCode);
                }
            }
        } catch (error) {
            console.error('Error fetching postal code for city:', city, error);
        }
    }

    showPostalCodeSuggestion(citySelectElement, postalCode) {
        // Remove any existing suggestion
        citySelectElement.closest('.input-group').find('.postal-suggestion').remove();
        
        // Create suggestion element
        const suggestionHtml = `
            <div class="postal-suggestion mt-1">
                <small class="text-muted">
                    <i class="ti ti-map-pin me-1"></i>
                    Postal Code: <strong>${postalCode}</strong>
                </small>
            </div>
        `;
        
        // Add suggestion after the input group
        citySelectElement.closest('.input-group').after(suggestionHtml);
        
        // Auto-hide after 5 seconds
        setTimeout(() => {
            citySelectElement.closest('.input-group').siblings('.postal-suggestion').fadeOut();
        }, 5000);
    }
}

// Initialize when document is ready
$(document).ready(function() {
    // Add CSS for postal suggestion styling
    if (!$('#search-postal-styles').length) {
        $('head').append(`
            <style id="search-postal-styles">
                .postal-suggestion {
                    background: #f8f9fa;
                    border: 1px solid #dee2e6;
                    border-radius: 0.375rem;
                    padding: 0.5rem;
                    margin-top: 0.25rem;
                    animation: fadeIn 0.3s ease-in;
                }
                @keyframes fadeIn {
                    from { opacity: 0; transform: translateY(-10px); }
                    to { opacity: 1; transform: translateY(0); }
                }
            </style>
        `);
    }
    
    // Initialize search city auto-fill
    new SearchCityAutoFill();
});

// Export for manual use if needed
window.SearchCityAutoFill = SearchCityAutoFill;
