/**
 * Request Help Form - Marketplace Lead Submission
 * Simplified 3-step form: Service → Questions → Message → Submit
 */

(function() {
    'use strict';

    // Form state
    let currentStep = 1;
    let totalSteps = 3; // Will be updated based on login status
    let formData = {};
    let categories = [];
    let services = [];
    let isUserLoggedIn = false;
    let iti; // Global intlTelInput instance (EXACT SAME as Join Us form)

    // Initialize form when document is ready
    $(document).ready(function() {
        if ($('#requestHelpForm').length) {
            initRequestHelpForm();
        }
    });

    /**
     * Initialize the form
     */
    function initRequestHelpForm() {
        // Check if user is logged in
        checkUserLoginStatus();

        // Load categories
        loadCategories();

        // Setup event listeners
        setupEventListeners();

        // Initialize character counter
        updateCharCounter();
    }

    /**
     * Check if user is logged in
     */
    function checkUserLoginStatus() {
        // Check if user is logged in by looking for auth data in the page (using existing data-authid)
        const authId = $('body').data('authid');
        isUserLoggedIn = authId !== undefined && authId !== null && authId !== '';
        
        // Update total steps based on login status
        if (isUserLoggedIn) {
            totalSteps = 3; // Service → Questions → Message → Submit
            $('#registration-step-indicator').hide();
        } else {
            totalSteps = 4; // Service → Questions → Message → Register → Submit
            $('#registration-step-indicator').show();
        }
        
        console.log('User logged in:', isUserLoggedIn, 'Auth ID:', authId, 'Total steps:', totalSteps);
    }

    /**
     * Load categories from API
     */
    function loadCategories() {
        $.ajax({
            url: '/api/leads/request-help/categories',
            type: 'GET',
            success: function(response) {
                if (response.success) {
                    categories = response.categories;
                    populateCategoryDropdown();
                } else {
                    console.error('Failed to load categories:', response.message);
                    toastr.error('Failed to load categories. Please refresh and try again.');
                }
            },
            error: function(xhr) {
                console.error('Error loading categories:', xhr.responseJSON);
                toastr.error('Failed to load categories. Please refresh and try again.');
            }
        });
    }

    /**
     * Populate category dropdown
     */
    function populateCategoryDropdown() {
        const $categorySelect = $('#rh_category_id');
        $categorySelect.empty().append('<option value="">Choose a category...</option>');
        
        categories.forEach(function(category) {
            $categorySelect.append(`<option value="${category.id}">${category.name}</option>`);
        });
    }

    /**
     * Load dynamic questions for step 2
     */
    function loadDynamicQuestions() {
        $.ajax({
            url: '/api/leads/help-form/questions',
            type: 'GET',
            success: function(response) {
                if (response.success && response.questions.length > 0) {
                    displayDynamicQuestions(response.questions);
                    } else {
                    // No questions available, show message and allow to proceed
                    $('#dynamic_questions_container').html(`
                        <div class="alert alert-info">
                            <i class="ri-information-line me-2"></i>
                            No additional questions at this time. You can proceed to the next step.
                        </div>
                    `);
                }
            },
            error: function(xhr) {
                console.error('Error loading questions:', xhr.responseJSON);
                $('#dynamic_questions_container').html(`
                    <div class="alert alert-warning">
                        <i class="ri-error-warning-line me-2"></i>
                        Unable to load questions. You can proceed to the next step.
                    </div>
                `);
            }
        });
    }

    /**
     * Display dynamic questions
     */
    function displayDynamicQuestions(questions) {
        let html = '';
        
        questions.forEach(function(question, index) {
            html += `
                <div class="mb-4">
                    <label class="form-label fw-bold">${question.question_text}</label>
                    <div class="question-container" data-question-id="${question.id}">
                        ${renderQuestionOptions(question)}
                    </div>
                </div>
            `;
        });
        
        $('#dynamic_questions_container').html(html);
    }

    /**
     * Render question options based on type
     */
    function renderQuestionOptions(question) {
        let html = '';
        
        switch(question.question_type) {
            case 'radio':
                question.answers.forEach(function(answer) {
                    html += `
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="question_${question.id}" 
                                   id="q${question.id}_${answer.id}" value="${answer.id}">
                            <label class="form-check-label" for="q${question.id}_${answer.id}">
                                ${answer.answer_text}
                            </label>
                        </div>
                    `;
                });
                break;
                
            case 'checkbox':
                question.answers.forEach(function(answer) {
                    html += `
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="question_${question.id}[]" 
                                   id="q${question.id}_${answer.id}" value="${answer.id}">
                            <label class="form-check-label" for="q${question.id}_${answer.id}">
                                ${answer.answer_text}
                            </label>
                        </div>
                    `;
                });
                break;
                
            case 'select':
                html += `<select class="form-select" name="question_${question.id}">`;
                html += '<option value="">Choose an answer...</option>';
                question.answers.forEach(function(answer) {
                    html += `<option value="${answer.id}">${answer.answer_text}</option>`;
                });
                html += '</select>';
                break;
                
            case 'text':
                html += `<textarea class="form-control" name="question_${question.id}" rows="3" placeholder="Your answer..."></textarea>`;
                break;
        }
        
        return html;
    }

    /**
     * Collect question responses
     */
    function collectQuestionResponses() {
        let responses = [];
        
        $('.question-container').each(function() {
            let questionId = $(this).data('question-id');
            let questionType = $(this).find('input, select, textarea').first().attr('type') || 
                              ($(this).find('select').length ? 'select' : 'text');
            
            let answers = [];
            
            if (questionType === 'radio') {
                let selected = $(this).find('input[type="radio"]:checked');
                if (selected.length) {
                    answers.push(selected.val());
                }
            } else if (questionType === 'checkbox') {
                $(this).find('input[type="checkbox"]:checked').each(function() {
                    answers.push($(this).val());
                });
            } else if (questionType === 'select') {
                let selected = $(this).find('select option:selected');
                if (selected.val()) {
                    answers.push(selected.val());
                }
            } else if (questionType === 'text') {
                let text = $(this).find('textarea').val();
                if (text.trim()) {
                    answers.push(text.trim());
                }
            }
            
            if (answers.length > 0) {
                responses.push({
                    question_id: questionId,
                    answers: answers
                });
            }
        });
        
        return responses;
    }

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Category change
        $(document).on('change', '#rh_category_id', function() {
            const categoryId = $(this).val();
            loadServices(categoryId);
        });

        // Next button
        $(document).on('click', '.btn-next', function(e) {
            e.preventDefault();
            console.log('🔵 Next button clicked - Current step:', currentStep, 'Total steps:', totalSteps);
            if (validateCurrentStep()) {
                console.log('✅ Validation passed');
                // If on last step and logged in, submit the form
                if (currentStep === 3 && isUserLoggedIn) {
                    console.log('📤 Logged in user on step 3 - submitting form');
                    submitForm();
                } else {
                    console.log('➡️ Moving to next step');
                    nextStep();
                }
            } else {
                console.log('❌ Validation failed');
            }
        });

        // Back button
        $(document).on('click', '.btn-prev', function(e) {
            e.preventDefault();
            prevStep();
        });

        // Form submission
        $(document).on('submit', '#requestHelpForm', function(e) {
            e.preventDefault();
            submitForm();
        });

        // Message character counter
        $(document).on('input', '#rh_message_text', function() {
            updateCharCounter();
        });
    }

    /**
     * Load services for selected category
     */
    function loadServices(categoryId) {
        if (!categoryId) {
            $('#rh_service_id').empty().append('<option value="">Choose a service...</option>');
            return;
        }

        $.ajax({
            url: '/api/leads/request-help/services',
            type: 'GET',
            data: { category_id: categoryId },
            success: function(response) {
                if (response.success) {
                    services = response.services;
                    populateServiceDropdown();
        } else {
                    console.error('Failed to load services:', response.message);
                }
            },
            error: function(xhr) {
                console.error('Error loading services:', xhr.responseJSON);
            }
        });
    }

    /**
     * Populate service dropdown
     */
    function populateServiceDropdown() {
        const $serviceSelect = $('#rh_service_id');
        $serviceSelect.empty().append('<option value="">Choose a service...</option>');
        
        services.forEach(function(service) {
            $serviceSelect.append(`<option value="${service.id}">${service.name}</option>`);
        });
    }

    /**
     * Validate current step
     */
    function validateCurrentStep() {
        let isValid = true;

        if (currentStep === 1) {
            // Validate category selection
            const categoryId = $('#rh_category_id').val();
            if (!categoryId) {
                $('#rh_category_id').addClass('is-invalid');
                isValid = false;
            } else {
                $('#rh_category_id').removeClass('is-invalid');
            }
        } else if (currentStep === 2) {
            // Validate questions (if any) - always valid for now
            console.log('✅ Step 2 validation passed');
        } else if (currentStep === 3) {
            // Validate message
            const message = $('#rh_message_text').val();
            if (!message || message.length < 50) {
                $('#rh_message_text').addClass('is-invalid');
                toastr.error('Please enter at least 50 characters in your message');
                console.log('❌ Message validation failed - Length:', message ? message.length : 0);
                        isValid = false;
            } else {
                $('#rh_message_text').removeClass('is-invalid');
                console.log('✅ Message validation passed - Length:', message.length);
            }
        }

        return isValid;
    }

    /**
     * Move to next step
     */
    function nextStep() {
        if (currentStep < totalSteps) {
        // Hide current step
        $(`.form-step[data-step="${currentStep}"]`).hide();

            // Update progress indicator
            $(`.step[data-step="${currentStep}"]`).removeClass('active').addClass('completed');
            
            // Move to next step
            currentStep++;
            
            // Special handling for Step 2 (Questions)
        if (currentStep === 2) {
            loadDynamicQuestions();
        }

            // Special handling for Step 4
        if (currentStep === 4) {
                if (!isUserLoggedIn) {
                    // Show registration step for non-logged users
                    $('#registration-step').show();
                    loadRegistrationFields();
                } else {
                    // Show submit step for logged-in users
                    $('#submit-step').show();
                }
            } else {
                // Show next step (for steps 1-3)
                $(`.form-step[data-step="${currentStep}"]`).show();
            }
            $(`.step[data-step="${currentStep}"]`).addClass('active');

        // Scroll to form
        $('html, body').animate({
            scrollTop: $('#requestHelpForm').offset().top - 100
        }, 500);
        }
    }

    /**
     * Move to previous step
     */
    function prevStep() {
        if (currentStep > 1) {
            // Hide current step
            $(`.form-step[data-step="${currentStep}"]`).hide();
            
            // Update progress indicator
            $(`.step[data-step="${currentStep}"]`).removeClass('active');
            
            // Move to previous step
            currentStep--;
            
            // Show previous step
            $(`.form-step[data-step="${currentStep}"]`).show();
            $(`.step[data-step="${currentStep}"]`).addClass('active').removeClass('completed');
            
            // Scroll to form
            $('html, body').animate({
                scrollTop: $('#requestHelpForm').offset().top - 100
            }, 500);
        }
    }

    /**
     * Update character counter
     */
    function updateCharCounter() {
        const message = $('#rh_message_text').val();
        const length = message ? message.length : 0;
        const $counter = $('#rh_char_counter');
        
        $counter.text(`${length} / 2000`);
        
        if (length < 50) {
            $counter.removeClass('text-success').addClass('text-danger');
        } else {
            $counter.removeClass('text-danger').addClass('text-success');
        }
    }

    /**
     * Submit form
     */
    function submitForm() {
        if (!validateCurrentStep()) {
            return;
        }
        
        // Collect form data
        const formData = {
            category_id: $('#rh_category_id').val(),
            service_id: $('#rh_service_id').val(),
            message_text: $('#rh_message_text').val(),
            question_responses: collectQuestionResponses(),
            _token: $('meta[name="csrf-token"]').attr('content')
        };

        // If user is not logged in, add registration fields
        if (!isUserLoggedIn) {
            formData.first_name = $('#rh_first_name').val();
            formData.last_name = $('#rh_last_name').val();
            formData.name = $('#rh_name').val(); // Username field
            formData.email = $('#rh_email').val();
            formData.password = $('#rh_password').val();
            // Get properly formatted international phone number (EXACT SAME as Join Us form)
            const fullPhoneNumber = iti.getNumber();
            formData.phone_number = fullPhoneNumber;
            formData.gender = $('#rh_gender').val();
            formData.dob = $('#rh_dob').val();
            formData.address = $('#rh_address').val();
            // Get text values instead of IDs for location fields
            formData.country = $('#rh_country option:selected').text() || $('#rh_country').val();
            formData.state = $('#rh_state option:selected').text() || $('#rh_state').val();
            formData.city = $('#rh_city option:selected').text() || $('#rh_city').val();
            formData.postal_code = $('#rh_postal_code').val();
            formData.terms_policy = $('#rh_terms_policy').is(':checked') ? 1 : 0;
        }

        // Debug: Log form data
        console.log('📤 Submitting form data:', formData);

        // Show loading state - handle both submit buttons
        const $submitBtn = $('#btn_submit_request_help, #btn_submit_with_registration');
        $submitBtn.prop('disabled', true);
        $submitBtn.find('.spinner-border').removeClass('d-none');
        $submitBtn.find('.btn-text').text('Submitting...');

        // Submit form
        $.ajax({
            url: '/api/leads/request-help/submit',
            type: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    if (!isUserLoggedIn && response.register_status === '1') {
                        // Handle OTP verification - EXACT SAME as Join Us form
                        handleOtpFlow(response);
                } else {
                        // For logged-in users - show success and reload
                        toastr.success('Request submitted successfully! Redirecting...');
                        setTimeout(function() {
                            window.location.href = '/user/submitted-requests';
                        }, 1500);
                    }
                } else {
                    toastr.error(response.message || 'Failed to submit request');
                }
            },
            error: function(xhr) {
                console.error('Form submission error:', xhr.responseJSON);
                
                // Show validation errors if present
                if (xhr.status === 422 && xhr.responseJSON?.errors) {
                    const errors = xhr.responseJSON.errors;
                    Object.keys(errors).forEach(function(field) {
                        errors[field].forEach(function(error) {
                            toastr.error(error);
                    });
                    });
                } else {
                    const message = xhr.responseJSON?.message || 'Failed to submit request. Please try again.';
                    toastr.error(message);
                }
            },
            complete: function() {
                // Reset button state
        $submitBtn.prop('disabled', false);
        $submitBtn.find('.spinner-border').addClass('d-none');
        $submitBtn.find('.btn-text').text('Submit Request');
            }
        });
    }

    /**
     * Collect question responses
     */
    function collectQuestionResponses() {
        const responses = [];
        $('.question-response').each(function() {
            const questionId = $(this).data('question-id');
            const value = $(this).val();
            if (questionId && value) {
                    responses.push({
                        question_id: questionId,
                    value: value
                });
            }
        });
        return responses;
    }

    /**
     * Show success message
     */
    function showSuccessMessage(message) {
        toastr.success(message);
    }
        
    /**
     * Send SMS OTP - EXACT SAME as Join Us form
     */
    function sendSms(phoneNumber, emailData, userName, otp) {
        return new Promise((resolve, reject) => {
        $.ajax({
                url: "/api/sms/sendsms",
                type: "POST",
                dataType: "json",
                data: {
                    otp_type: "sms",
                    to_number: phoneNumber,
                    notification_type: 2,
                    type: 1,
                    user_name: userName,
                    otp: otp,
                    subject: emailData.subject,
                    content: emailData.content,
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                    Accept: "application/json",
                },
                success: function (response) {
                    resolve(response);
                },
                error: function (error) {
                    console.error('❌ SMS sending failed:', error);
                    console.error('❌ SMS response:', error.responseJSON);
                    console.error('❌ SMS response text:', error.responseText);
                    console.error('❌ SMS status:', error.status);
                    console.error('❌ SMS full error:', JSON.stringify(error.responseJSON));
                    
                    if (error.status === 422) {
                        const errors = error.responseJSON?.errors;
                        if (errors) {
                            toastr.error("Validation error while sending OTP.");
                } else {
                            toastr.error("An error occurred while sending OTP.");
                        }
                    } else {
                        const msg = error.responseJSON?.message || "An error occurred while sending OTP.";
                        toastr.error(msg);
                    }
                    reject(error);
                },
            });
        });
    }

    /**
     * Start SMS timer - EXACT SAME as Join Us form
     */
    let helpFormSmsTimerInterval;
    let helpFormSmsTimerTime;
    
    function startHelpFormSmsTimer(expireSmsTime) {
        clearInterval(helpFormSmsTimerInterval); // Clear any existing timer
        helpFormSmsTimerTime = expireSmsTime * 60; // Convert minutes to seconds

        const helpFormOtpTimerDisplay = document.getElementById("help-form-otp-timer");
        
        helpFormSmsTimerInterval = setInterval(() => {
            let minutes = Math.floor(helpFormSmsTimerTime / 60);
            let seconds = helpFormSmsTimerTime % 60;
            if (helpFormOtpTimerDisplay) {
                helpFormOtpTimerDisplay.textContent = `${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;
            }
            helpFormSmsTimerTime--;

            if (helpFormSmsTimerTime < 0) {
                clearInterval(helpFormSmsTimerInterval);
                if (helpFormOtpTimerDisplay) {
                    helpFormOtpTimerDisplay.textContent = "00:00";
                }
            }
        }, 1000);
    }

    /**
     * Handle OTP flow - EXACT SAME as Join Us form
     */
    function handleOtpFlow(response) {
        const userName = response.name;
        const otp = response.otp;
        const otpExpireTime = parseInt(response.otp_expire_time.split(" ")[0]);
        const phoneNumber = response.phone_number;
        const otpDigitLimit = parseInt(response.otp_digit_limit);
        
        // Use dedicated Help Form OTP modal (completely separate from Join Us)
        const helpFormOtpModal = new bootstrap.Modal(document.getElementById("help-form-otp-modal"));
        
        // Create OTP input boxes in the dedicated Help Form container
        const helpFormOtpContainer = $(".helpFormOtpContainer");
        helpFormOtpContainer.empty();
        
        let inputsSMSHtml = '<div class="d-flex align-items-center mb-3">';
        for (let i = 1; i <= otpDigitLimit; i++) {
            const nextId = `help-otp-${i + 1}`;
            const prevId = `help-otp-${i - 1}`;
            inputsSMSHtml += `
                <input type="text" 
                       class="rounded w-100 py-sm-3 py-2 text-center fs-26 fw-bold me-3 help-otp-${i}" 
                       id="help-otp-${i}" 
                       name="help-otp-${i}" 
                       data-next="${nextId}" 
                       data-previous="${prevId}" 
                       maxlength="1">
            `;
        }
        inputsSMSHtml += "</div>";
        helpFormOtpContainer.append(inputsSMSHtml);
        
        // Add input navigation for Help Form OTP inputs
        $(".helpFormOtpContainer").on("input", "input", function () {
            const maxLength = $(this).attr("maxlength") || 1;
            if (this.value.length >= maxLength) {
                const next = $(this).data("next");
                if (next) {
                    $("#" + next).focus();
                }
            }
        });
        
        $(".helpFormOtpContainer").on("keydown", "input", function (e) {
            if (e.key === "Backspace" && this.value === "") {
                const prev = $(this).data("previous");
                if (prev) {
                    $("#" + prev).focus();
                }
            }
        });
        
        $(".helpFormOtpContainer").on("click", "input", function () {
            $(this).select();
        });
        
        // THEN send SMS (EXACT SAME as Join Us)
        if (response.otp_type === "sms") {
            const emailData = {
                subject: response.email_subject,
                content: response.email_content,
            };
            
            sendSms(phoneNumber, emailData, userName, otp)
                .then(() => {
                    // Update Help Form modal message
                    const helpFormOtpMessage = document.getElementById("help-form-otp-message");
                    if (helpFormOtpMessage) {
                        helpFormOtpMessage.textContent = `OTP sent to your mobile number ending ******${phoneNumber.slice(-4)}`;
                    }
                    
                    // Show Help Form OTP modal
                    helpFormOtpModal.show();
                    
                    // Start timer on Help Form modal
                    startHelpFormSmsTimer(otpExpireTime);
                    
                    console.log('✅ SMS OTP sent successfully');
                })
                .catch((error) => {
                    console.error('❌ Failed to send SMS OTP:', error);
                    toastr.error('Failed to send verification code. Please try again.');
                });
        }
        
        // Add Help Form specific resend OTP button handler
        $('.helpFormResendOtp').off('click').on('click', function(e) {
            e.preventDefault();
            console.log('✅ Help Form resend handler called');
            
            // Call Help Form resend endpoint
            $.ajax({
                url: '/api/leads/request-help/resend-otp',
                type: 'POST',
                data: {
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                success: function(data) {
                    const newOtp = data.otp;
                    const newOtpExpireTime = parseInt(data.otp_expire_time.split(" ")[0]);
                    const newPhoneNumber = data.phone_number || phoneNumber;
                    const newUserName = data.name || userName;
                    
                    const emailData = {
                        subject: data.email_subject || response.email_subject,
                        content: data.email_content || response.email_content,
                    };
                    
                    // Send new SMS
                    sendSms(newPhoneNumber, emailData, newUserName, newOtp)
                        .then(() => {
                            toastr.success('New verification code sent!');
                            // Restart timer
                            startHelpFormSmsTimer(newOtpExpireTime);
                        })
                        .catch((error) => {
                            toastr.error('Failed to resend code. Please try again.');
                        });
                },
                error: function(xhr) {
                    const errorMessage = xhr.responseJSON?.message || "Failed to resend OTP. Please try again.";
                    toastr.error(errorMessage);
                }
            });
        });
        
        // Add Help Form specific verify handler
        $('#help-form-verify-otp-btn').off('click').on('click', function(e) {
            e.preventDefault();
            console.log('✅ Help Form verify handler called');
            console.log('🎯 Button ID:', $(this).attr('id'));
            console.log('📍 Endpoint:', '/api/leads/request-help/verify-otp');
            
            // Collect OTP digits from Help Form inputs
            const otp = [];
            for (let i = 1; i <= otpDigitLimit; i++) {
                const digit = $(`#help-otp-${i}`).val();
                console.log(`OTP Digit ${i}:`, digit);
                otp.push(digit);
            }
            const otpString = otp.join("");
            console.log('📝 Complete OTP:', otpString);
            
            if (otpString.length !== otpDigitLimit) {
                toastr.error('Please enter the complete verification code');
            return;
        }

            // Show loading
            $(this).attr("disabled", true);
            $(this).html('<div class="spinner-border text-light" role="status"></div>');
            
            console.log('🚀 Calling Help Form verification endpoint...');
            
            // Verify OTP via Help Form endpoint
        $.ajax({
                url: '/api/leads/request-help/verify-otp',
            type: 'POST',
            data: {
                    otp: otpString,
                    _token: $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                    if (response.success) {
                        $('#help-form-otp-modal').modal('hide');
                        toastr.success('Request submitted successfully!');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                } else {
                        $("#help-form-otp-error").text(response.message || 'Invalid OTP');
                }
            },
            error: function(xhr) {
                    console.error('❌ Verification failed:', xhr);
                    console.error('❌ Status:', xhr.status);
                    console.error('❌ Response:', xhr.responseJSON);
                    const errorMessage = xhr.responseJSON?.message || 'OTP verification failed';
                    $("#help-form-otp-error").text(errorMessage);
                    toastr.error(errorMessage);
            },
            complete: function() {
                    $('#help-form-verify-otp-btn').attr("disabled", false);
                    $('#help-form-verify-otp-btn').html("Verify & Proceed");
                }
            });
        });
    }

    /**
     * Load registration fields dynamically
     */
    function loadRegistrationFields() {
        const $container = $('#help_form_registration_container');
        
        // Only load if not already loaded
        if ($container.children().length === 1) {
            $.ajax({
                url: '/user/auth/register-fields',
                type: 'GET',
                success: function(response) {
                    $container.html(response);
                    console.log('✅ Registration fields loaded successfully');
                    
                    // Initialize phone number with country selector
                    initializePhoneNumber();
                    
                    // Initialize country/state/city dropdowns
                    initializeLocationDropdowns();
                    
                    // Auto-detect and select country
                    autoDetectCountry();
                    
                    // Initialize postal code auto-fill
                    initializePostalCodeAutoFill();
                    
                    // Initialize password toggle
                    initializePasswordToggle();
                },
                error: function(xhr) {
                    console.error('❌ Failed to load registration fields:', xhr.responseJSON);
                    $container.html('<div class="alert alert-danger">Failed to load registration form. Please refresh and try again.</div>');
                }
            });
        }
    }

    /**
     * Initialize phone number with country selector
     */
    function initializePhoneNumber() {
        const phoneInput = document.querySelector('#rh_phone');
        if (phoneInput && window.intlTelInput) {
            // Destroy existing instance if any
            if (phoneInput._iti) {
                phoneInput._iti.destroy();
            }
            
            // Initialize intlTelInput (EXACT SAME as Join Us form)
            iti = intlTelInput(phoneInput, {
                utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js",
                separateDialCode: true,
                initialCountry: "za",
                placeholderNumberType: "",
                autoPlaceholder: "off"
            });
            
            // Store instance for later use
            phoneInput._iti = iti;
            
            console.log('✅ Phone number with country selector initialized');
        }
    }

    /**
     * Initialize location dropdowns (country/state/city)
     */
    function initializeLocationDropdowns() {
        // Initialize country dropdown
        if ($('#rh_country').length && typeof $.fn.select2 !== 'undefined') {
            $('#rh_country').select2({
                placeholder: 'Select Country',
                allowClear: true
            });
            
            // Load countries
            loadCountries();
        }
        
        // Initialize state dropdown
        if ($('#rh_state').length && typeof $.fn.select2 !== 'undefined') {
            $('#rh_state').select2({
                placeholder: 'Select State',
                allowClear: true
            });
        }
        
        // Initialize city dropdown
        if ($('#rh_city').length && typeof $.fn.select2 !== 'undefined') {
            $('#rh_city').select2({
                placeholder: 'Select City',
                allowClear: true
            });
        }
        
        // Set up cascading dropdowns
        setupCascadingDropdowns();
    }

    /**
     * Load countries for dropdown - SAME AS JOIN US FORM
     */
    function loadCountries() {
        $.getJSON("/countries.json", function(data) {
            const $countrySelect = $('#rh_country');
            $countrySelect.empty().append('<option value="">Select Country</option>');
            $.each(data.countries, function(index, country) {
                $countrySelect.append($("<option>", {
                    value: country.id,
                    text: country.name
                }));
            });
            console.log('✅ Countries loaded successfully');
        });
    }

    /**
     * Set up cascading dropdowns for country/state/city - SAME AS JOIN US FORM
     */
    function setupCascadingDropdowns() {
        // Country change handler
        $('#rh_country').on('change', function() {
            const selectedCountry = $(this).val();
            const $stateSelect = $('#rh_state');
            const $citySelect = $('#rh_city');
            
            // Clear state and city
            $stateSelect.empty().append('<option value="">Select State</option>');
            $citySelect.empty().append('<option value="">Select City</option>');
            
            if (selectedCountry) {
                $.getJSON("/states.json", function(data) {
                    const states = data.states.filter(state => state.country_id == selectedCountry);
                    $.each(states, function(index, state) {
                        $stateSelect.append($("<option>", {
                            value: state.id,
                            text: state.name
                        }));
                    });
                });
            }
        });
        
        // State change handler
        $('#rh_state').on('change', function() {
            const selectedState = $(this).val();
            const $citySelect = $('#rh_city');
            
            // Clear city
            $citySelect.empty().append('<option value="">Select City</option>');
            
            if (selectedState) {
                $.getJSON("/cities.json", function(data) {
                    const cities = data.cities.filter(city => city.state_id == selectedState);
                    $.each(cities, function(index, city) {
                        $citySelect.append($("<option>", {
                            value: city.id,
                            text: city.name
                        }));
                    });
                });
            }
        });
    }

    /**
     * Initialize password toggle functionality
     */
    function initializePasswordToggle() {
        $('#rh_togglePassword').on('click', function() {
            const passwordInput = $('#rh_password');
            const toggleIcon = $('#rh_toggleIcon');
            
            if (passwordInput.attr('type') === 'password') {
                passwordInput.attr('type', 'text');
                toggleIcon.removeClass('fa-eye').addClass('fa-eye-slash');
                } else {
                passwordInput.attr('type', 'password');
                toggleIcon.removeClass('fa-eye-slash').addClass('fa-eye');
            }
        });
    }

    /**
     * Auto-detect and select country
     */
    function autoDetectCountry() {
        // Use the existing geo-country-detect functionality
        if (window.GeoCountryDetect) {
            const geoDetect = new GeoCountryDetect();
            // The geo detection will automatically work with #rh_country selector
        }
        
        // Also try to detect from browser timezone
        const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
        const timezoneCountryMap = {
            'Africa/Johannesburg': 'South Africa',
            'America/New_York': 'United States',
            'America/Los_Angeles': 'United States',
            'America/Chicago': 'United States',
            'Europe/London': 'United Kingdom',
            'America/Toronto': 'Canada',
            'Australia/Sydney': 'Australia',
            'Asia/Kolkata': 'India',
            'Europe/Berlin': 'Germany',
            'Europe/Paris': 'France',
            'Europe/Rome': 'Italy',
            'Europe/Madrid': 'Spain',
            'Asia/Singapore': 'Singapore',
            'Asia/Dubai': 'United Arab Emirates',
            'Asia/Tokyo': 'Japan',
            'Asia/Shanghai': 'China',
            'America/Sao_Paulo': 'Brazil',
            'America/Mexico_City': 'Mexico'
        };

        if (timezoneCountryMap[timezone]) {
            const detectedCountry = timezoneCountryMap[timezone];
            console.log('Country detected from timezone:', detectedCountry);
            
            // Wait for countries to load, then auto-select
            setTimeout(() => {
                const $countrySelect = $('#rh_country');
                if ($countrySelect.length && $countrySelect.find('option').length > 1) {
                    const matchingOption = $countrySelect.find(`option:contains("${detectedCountry}")`);
                    if (matchingOption.length) {
                        $countrySelect.val(matchingOption.val()).trigger('change');
                        console.log('✅ Country auto-selected:', detectedCountry);
                    }
                }
            }, 1000);
        }
    }

    /**
     * Initialize postal code auto-fill
     */
    function initializePostalCodeAutoFill() {
        // Set up postal code auto-fill when city is selected
        $('#rh_city').on('change', function() {
            const city = $(this).find('option:selected').text();
            const state = $('#rh_state').find('option:selected').text();
            const country = $('#rh_country').find('option:selected').text();
            
            if (city && city !== 'Select City' && state && state !== 'Select State' && country && country !== 'Select Country') {
                fetchPostalCodeREST(city, state, country);
            }
        });
    }

    /**
     * Fetch postal code for selected city/state/country
     */
    async function fetchPostalCodeREST(city, state, country) {
        const $postalCodeInput = $('#rh_postal_code');
        
        // Show loading state
        $postalCodeInput.attr('placeholder', 'Fetching postal code...');
        
        try {
            // Get Google Maps API key
            const keyResponse = await fetch('/api/get-google-maps-key');
            const keyData = await keyResponse.json();
            
            if (!keyData.success || !keyData.api_key) {
                console.log('❌ Google Maps API key not available');
                $postalCodeInput.attr('placeholder', 'Enter Postal Code');
            return;
        }
        
            // Try multiple address formats for better results
            const addressFormats = [
                `${city}, ${state}, ${country}`,
                `${city} ${state} ${country}`,
                `${city}, ${country}`,
                `${city} Central, ${state}, ${country}`,
                `${city} CBD, ${state}, ${country}`
            ];
            
            for (const address of addressFormats) {
                const geocodeUrl = `https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${keyData.api_key}`;
                
                try {
                    const response = await fetch(geocodeUrl);
                    const data = await response.json();
                    
                    if (data.status === 'OK' && data.results.length > 0) {
                        for (const result of data.results) {
                            const postalCodeComponent = result.address_components.find(component => 
                                component.types.includes('postal_code') || 
                                component.types.includes('postal_code_prefix')
                            );
                            
                            if (postalCodeComponent) {
                                const postalCode = postalCodeComponent.long_name;
                                $postalCodeInput.val(postalCode);
                                $postalCodeInput.attr('placeholder', 'Enter Postal Code');
                                console.log('✅ Postal code auto-filled:', postalCode);
                                return; // Success, exit
                            }
                        }
                    }
                } catch (error) {
                    console.log('❌ Geocoding request failed:', error);
                }
            }
            
            // If no postal code found
            $postalCodeInput.attr('placeholder', 'Postal code not found');
            console.log('❌ No postal code found for:', city, state, country);
            
        } catch (error) {
            console.error('❌ Failed to fetch postal code:', error);
            $postalCodeInput.attr('placeholder', 'Enter Postal Code');
        }
    }

    /**
     * Extract postal code from geocoding result
     */
    function extractPostalCode(result) {
        for (let component of result.address_components) {
            if (component.types.includes('postal_code')) {
                return component.long_name;
            }
        }
        return null;
    }

    /**
     * Close help form
     */
    function closeHelpForm() {
        // Close any modal or hide form
        $('.request-help-form-wrapper').fadeOut();
    }

})();
