/**
 * Request Help Form - Marketplace Lead Submission
 * Simplified 3-step form: Service → Questions → Message → Submit
 */

(function() {
    'use strict';

    // Form state
    let currentStep = 1;
    const totalSteps = 3;
    let formData = {};
    let categories = [];
    let services = [];

    // Initialize form when document is ready
    $(document).ready(function() {
        if ($('#requestHelpForm').length) {
            initRequestHelpForm();
        }
    });

    /**
     * Initialize the form
     */
    function initRequestHelpForm() {
        // Load categories
        loadCategories();

        // Setup event listeners
        setupEventListeners();

        // Initialize character counter
        updateCharCounter();
    }

    /**
     * Load categories from API
     */
    function loadCategories() {
        $.ajax({
            url: '/api/leads/request-help/categories',
            type: 'GET',
            success: function(response) {
                if (response.success) {
                    categories = response.categories;
                    populateCategoryDropdown();
                } else {
                    console.error('Failed to load categories:', response.message);
                    toastr.error('Failed to load categories. Please refresh and try again.');
                }
            },
            error: function(xhr) {
                console.error('Error loading categories:', xhr.responseJSON);
                toastr.error('Failed to load categories. Please refresh and try again.');
            }
        });
    }

    /**
     * Populate category dropdown
     */
    function populateCategoryDropdown() {
        const $categorySelect = $('#rh_category_id');
        $categorySelect.empty().append('<option value="">Choose a category...</option>');
        
        categories.forEach(function(category) {
            $categorySelect.append(`<option value="${category.id}">${category.name}</option>`);
        });
    }

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Category change
        $(document).on('change', '#rh_category_id', function() {
            const categoryId = $(this).val();
            loadServices(categoryId);
        });

        // Next button
        $(document).on('click', '.btn-next', function(e) {
            e.preventDefault();
            if (validateCurrentStep()) {
                nextStep();
            }
        });

        // Back button
        $(document).on('click', '.btn-prev', function(e) {
            e.preventDefault();
            prevStep();
        });

        // Form submission
        $(document).on('submit', '#requestHelpForm', function(e) {
            e.preventDefault();
            submitForm();
        });

        // Message character counter
        $(document).on('input', '#rh_message_text', function() {
            updateCharCounter();
        });
    }

    /**
     * Load services for selected category
     */
    function loadServices(categoryId) {
        if (!categoryId) {
            $('#rh_service_id').empty().append('<option value="">Choose a service...</option>');
            return;
        }

        $.ajax({
            url: '/api/leads/request-help/services',
            type: 'GET',
            data: { category_id: categoryId },
            success: function(response) {
                if (response.success) {
                    services = response.services;
                    populateServiceDropdown();
                } else {
                    console.error('Failed to load services:', response.message);
                }
            },
            error: function(xhr) {
                console.error('Error loading services:', xhr.responseJSON);
            }
        });
    }

    /**
     * Populate service dropdown
     */
    function populateServiceDropdown() {
        const $serviceSelect = $('#rh_service_id');
        $serviceSelect.empty().append('<option value="">Choose a service...</option>');
        
        services.forEach(function(service) {
            $serviceSelect.append(`<option value="${service.id}">${service.name}</option>`);
        });
    }

    /**
     * Validate current step
     */
    function validateCurrentStep() {
        let isValid = true;
        
        if (currentStep === 1) {
            // Validate category selection
            const categoryId = $('#rh_category_id').val();
            if (!categoryId) {
                $('#rh_category_id').addClass('is-invalid');
                isValid = false;
            } else {
                $('#rh_category_id').removeClass('is-invalid');
            }
        } else if (currentStep === 2) {
            // Validate questions (if any)
            // Questions validation can be added here if needed
        } else if (currentStep === 3) {
            // Validate message
            const message = $('#rh_message_text').val();
            if (!message || message.length < 50) {
                $('#rh_message_text').addClass('is-invalid');
                isValid = false;
            } else {
                $('#rh_message_text').removeClass('is-invalid');
            }
        }
        
        return isValid;
    }

    /**
     * Move to next step
     */
    function nextStep() {
        if (currentStep < totalSteps) {
            // Hide current step
            $(`.form-step[data-step="${currentStep}"]`).hide();
            
            // Update progress indicator
            $(`.step[data-step="${currentStep}"]`).removeClass('active').addClass('completed');
            
            // Move to next step
            currentStep++;
            
            // Show next step
            $(`.form-step[data-step="${currentStep}"]`).show();
            $(`.step[data-step="${currentStep}"]`).addClass('active');
            
            // Scroll to form
            $('html, body').animate({
                scrollTop: $('#requestHelpForm').offset().top - 100
            }, 500);
        }
    }

    /**
     * Move to previous step
     */
    function prevStep() {
        if (currentStep > 1) {
            // Hide current step
            $(`.form-step[data-step="${currentStep}"]`).hide();
            
            // Update progress indicator
            $(`.step[data-step="${currentStep}"]`).removeClass('active');
            
            // Move to previous step
            currentStep--;
            
            // Show previous step
            $(`.form-step[data-step="${currentStep}"]`).show();
            $(`.step[data-step="${currentStep}"]`).addClass('active').removeClass('completed');
            
            // Scroll to form
            $('html, body').animate({
                scrollTop: $('#requestHelpForm').offset().top - 100
            }, 500);
        }
    }

    /**
     * Update character counter
     */
    function updateCharCounter() {
        const message = $('#rh_message_text').val();
        const length = message ? message.length : 0;
        const $counter = $('#rh_char_counter');
        
        $counter.text(`${length} / 2000`);
        
        if (length < 50) {
            $counter.removeClass('text-success').addClass('text-danger');
        } else {
            $counter.removeClass('text-danger').addClass('text-success');
        }
    }

    /**
     * Submit form
     */
    function submitForm() {
        if (!validateCurrentStep()) {
            return;
        }

        // Collect form data
        const formData = {
            category_id: $('#rh_category_id').val(),
            service_id: $('#rh_service_id').val(),
            message_text: $('#rh_message_text').val(),
            question_responses: collectQuestionResponses(),
            _token: $('meta[name="csrf-token"]').attr('content')
        };

        // Show loading state
        const $submitBtn = $('#btn_submit_request_help');
        $submitBtn.prop('disabled', true);
        $submitBtn.find('.spinner-border').removeClass('d-none');
        $submitBtn.find('.btn-text').text('Submitting...');

        // Submit form
        $.ajax({
            url: '/api/leads/request-help',
            type: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    showSuccessMessage('Request submitted successfully!');
                    closeHelpForm();
                } else {
                    toastr.error(response.message || 'Failed to submit request');
                }
            },
            error: function(xhr) {
                console.error('Form submission error:', xhr.responseJSON);
                const message = xhr.responseJSON?.message || 'Failed to submit request. Please try again.';
                toastr.error(message);
            },
            complete: function() {
                // Reset button state
                $submitBtn.prop('disabled', false);
                $submitBtn.find('.spinner-border').addClass('d-none');
                $submitBtn.find('.btn-text').text('Submit Request');
            }
        });
    }

    /**
     * Collect question responses
     */
    function collectQuestionResponses() {
        const responses = [];
        $('.question-response').each(function() {
            const questionId = $(this).data('question-id');
            const value = $(this).val();
            if (questionId && value) {
                responses.push({
                    question_id: questionId,
                    value: value
                });
            }
        });
        return responses;
    }

    /**
     * Show success message
     */
    function showSuccessMessage(message) {
        toastr.success(message);
    }

    /**
     * Close help form
     */
    function closeHelpForm() {
        // Close any modal or hide form
        $('.request-help-form-wrapper').fadeOut();
    }

})();
