/**
 * Provider Marketplace - Lead Browsing and Unlocking
 * Handles marketplace lead display, filtering, and unlocking functionality
 */

(function() {
    'use strict';

    // Global variables
    let currentLeads = [];
    let currentPage = 1;
    let totalPages = 1;
    let currentLeadId = null;
    let filters = {
        category: '',
        priceRange: '',
        quality: '',
        search: ''
    };

    // Initialize when document is ready
    $(document).ready(function() {
        initProviderMarketplace();
        
        // Load unlocked leads when tab is clicked
        $('#unlocked-tab').on('click', function() {
            loadUnlockedLeads();
        });
        
        // Handle connection request button
        $(document).on('click', '.send-connection-btn', function() {
            const unlockId = $(this).data('unlock-id');
            sendConnectionRequest(unlockId);
        });
    });

    /**
     * Initialize provider marketplace
     */
    function initProviderMarketplace() {
        // Setup CSRF token for all AJAX requests
        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });
        
        loadWalletBalance();
        loadCategories();
        setupEventListeners();
        loadLeads();
    }

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Filter changes
        $('#categoryFilter, #priceFilter, #qualityFilter').on('change', function() {
            applyFilters();
        });

        // Search functionality
        $('#searchBtn').on('click', function() {
            applyFilters();
        });

        $('#searchInput').on('keypress', function(e) {
            if (e.which === 13) {
                applyFilters();
            }
        });

        // Modal events
        $('#leadDetailsModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const leadId = button.data('lead-id');
            if (leadId) {
                loadLeadDetails(leadId);
            }
        });

        $('#unlockConfirmModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const leadId = button.data('lead-id');
            const price = button.data('price');
            if (leadId && price) {
                currentLeadId = leadId;
                $('#unlockPrice').text(price + ' Credits');
            }
        });

        // Unlock button click handler (direct from card)
        $(document).on('click', '.unlock-btn', function(e) {
            e.preventDefault();
            const leadId = $(this).data('lead-id');
            const price = $(this).data('price');
            if (leadId && price) {
                currentLeadId = leadId;
                $('#unlockPrice').text(price + ' Credits');
                $('#unlockConfirmModal').modal('show');
            }
        });

        // Unlock button click handler (in lead details modal)
        $(document).on('click', '#unlockLeadBtn', function(e) {
            e.preventDefault();
            const leadId = $(this).data('lead-id');
            const price = $(this).data('price');
            if (leadId && price) {
                currentLeadId = leadId;
                $('#unlockPrice').text(price + ' Credits');
                $('#leadDetailsModal').modal('hide');
                $('#unlockConfirmModal').modal('show');
            }
        });

        // Unlock confirmation
        $('#confirmUnlockBtn').on('click', function() {
            if (currentLeadId) {
                unlockLead(currentLeadId);
            }
        });

        // Pagination
        $(document).on('click', '.pagination-link', function(e) {
            e.preventDefault();
            const page = $(this).data('page');
            if (page && page !== currentPage) {
                currentPage = page;
                loadLeads();
            }
        });
    }

    /**
     * Load wallet balance
     */
    function loadWalletBalance() {
        $.ajax({
            url: '/provider/marketplace/wallet',
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    $('#walletBalance').text(response.balance + ' Credits');
                }
            },
            error: function(xhr) {
                console.error('Error loading wallet balance:', xhr.responseJSON?.message || 'Unknown error');
                $('#walletBalance').text('0 Credits');
            }
        });
    }

    /**
     * Load categories for filter
     */
    function loadCategories() {
        $.ajax({
            url: '/api/leads/request-help/categories',
            type: 'GET',
            success: function(response) {
                if (response.success && response.categories) {
                    const $select = $('#categoryFilter');
                    response.categories.forEach(function(category) {
                        $select.append(`<option value="${category.id}">${category.name}</option>`);
                    });
                }
            },
            error: function(xhr) {
                console.error('Error loading categories:', xhr.responseJSON?.message || 'Unknown error');
            }
        });
    }

    /**
     * Load marketplace leads
     */
    function loadLeads() {
        const requestData = {
            page: currentPage,
            ...filters
        };

        $.ajax({
            url: '/provider/marketplace/leads',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: JSON.stringify(requestData),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    currentLeads = response.data || [];
                    totalPages = response.pagination?.last_page || 1;
                    updateStats({
                        total: response.stats?.total || 0,
                        unlocked: response.stats?.unlocked || 0,
                        pending: response.stats?.pending || 0
                    });
                    renderLeads();
                    renderPagination();
                } else {
                    toastr.error(response.message || 'Failed to load leads');
                }
            },
            error: function(xhr) {
                console.error('Error loading leads:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load marketplace leads');
            }
        });
    }

    /**
     * Apply filters and reload leads
     */
    function applyFilters() {
        filters = {
            category: $('#categoryFilter').val(),
            priceRange: $('#priceFilter').val(),
            quality: $('#qualityFilter').val(),
            search: $('#searchInput').val()
        };
        currentPage = 1;
        loadLeads();
    }

    /**
     * Update statistics
     */
    function updateStats(stats) {
        $('#totalLeads').text(stats.total || 0);
        $('#unlockedLeads').text(stats.unlocked || 0);
        $('#pendingLeads').text(stats.pending || 0);
    }

    /**
     * Render leads grid
     */
    function renderLeads() {
        const $grid = $('#leadsGrid');
        $grid.empty();

        if (currentLeads.length === 0) {
            $grid.html(`
                <div class="col-12">
                    <div class="card text-center py-5">
                        <div class="card-body">
                            <i class="fas fa-search fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No leads found</h5>
                            <p class="text-muted">Try adjusting your filters or check back later for new leads.</p>
                        </div>
                    </div>
                </div>
            `);
            return;
        }

        currentLeads.forEach(function(lead) {
            const leadCard = createLeadCard(lead);
            $grid.append(leadCard);
        });
    }

    /**
     * Create lead card HTML
     */
    function createLeadCard(lead) {
        const isUnlocked = lead.already_unlocked || false;
        
        return $(`
            <div class="col-lg-12 col-md-12 mb-3">
                <div class="card lead-card">
                    <div class="card-body p-3">
                        <div class="row align-items-center">
                            <!-- Left Column: Main Info -->
                            <div class="col-lg-4 col-md-4">
                                <div class="d-flex align-items-center mb-2">
                                    <span class="avatar p-1 me-2 bg-teal-transparent flex-shrink-0">
                                        <i class="ti ti-user-edit text-info fs-20"></i>
                                    </span>
                                    <div>
                                        <h6 class="mb-0">${lead.user?.name || 'N/A'}</h6>
                                        <small class="text-muted">${lead.user?.email || 'N/A'}</small>
                                    </div>
                                </div>
                                <div class="mb-2">
                                    <h6 class="card-title mb-1">${lead.category || 'N/A'}</h6>
                                    <small class="text-muted">${lead.service || 'General Service'}</small>
                                </div>
                            </div>
                            
                            <!-- Middle Column: Message & Questions -->
                            <div class="col-lg-4 col-md-4">
                                <div class="mb-2">
                                    <h6 class="mb-1 small">Message:</h6>
                                    <p class="card-text small mb-1">${lead.message_preview || 'No message provided'}</p>
                                </div>
                                ${lead.questions_answered > 0 ? `
                                    <div class="mb-2 ${!isUnlocked ? 'blur-content' : ''}">
                                        <h6 class="mb-1 small">Questions (${lead.questions_answered}):</h6>
                                        ${isUnlocked && lead.question_responses ? lead.question_responses.map(qr => `
                                            <div class="small p-1 bg-light rounded mb-1">
                                                <strong>Q:</strong> ${qr.question}<br>
                                                <strong>A:</strong> ${qr.answer}
                                            </div>
                                        `).join('') : `
                                            <div class="small p-1 bg-light rounded mb-1">
                                                <strong>Q:</strong> ████████████████<br>
                                                <strong>A:</strong> ████████████████
                                            </div>
                                        `}
                                    </div>
                                ` : ''}
                            </div>
                            
                            <!-- Right Column: Details & Actions -->
                            <div class="col-lg-4 col-md-4">
                                <div class="text-end mb-2">
                                    <span class="badge bg-success">${lead.unlock_price || '0.00'} Credits</span>
                                </div>
                                <div class="row small text-muted mb-2">
                                    <div class="col-6">
                                        <p class="mb-1">
                                            <i class="fas fa-map-marker-alt me-1"></i>
                                            ${lead.location || 'N/A'}
                                        </p>
                                        <p class="mb-1">
                                            <i class="fas fa-phone me-1"></i>
                                            ${lead.user?.phone || 'N/A'}
                                        </p>
                                    </div>
                                    <div class="col-6">
                                        <p class="mb-1">
                                            <i class="fas fa-calendar me-1"></i>
                                            ${lead.posted_at || 'N/A'}
                                        </p>
                                        <p class="mb-1">
                                            <i class="fas fa-file-alt me-1"></i>
                                            ${lead.fields_filled || 0} fields
                                        </p>
                                    </div>
                                </div>
                                <div class="d-flex justify-content-between align-items-center">
                                    <small class="text-muted">
                                        ${isUnlocked ? 'Unlocked' : 'No unlocks yet'}
                                    </small>
                                    ${!isUnlocked ? `
                                        <button class="btn btn-sm btn-primary unlock-btn" 
                                                data-lead-id="${lead.id}"
                                                data-price="${lead.unlock_price || '0.00'}">
                                            <i class="fas fa-unlock me-1"></i>Unlock
                                        </button>
                                    ` : `
                                        <button class="btn btn-sm btn-success" disabled>
                                            <i class="fas fa-check me-1"></i>Unlocked
                                        </button>
                                    `}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `);
    }

    /**
     * Load lead details for modal
     */
    function loadLeadDetails(leadId) {
        const lead = currentLeads.find(l => l.id == leadId);
        if (!lead) return;

        const isUnlocked = lead.is_unlocked || false;
        const $content = $('#leadDetailsContent');
        
        let detailsHtml = `
            <div class="row">
                <div class="col-md-6">
                    <h6>Service Information</h6>
                    <p><strong>Category:</strong> ${isUnlocked ? (lead.category?.name || 'N/A') : 'Hidden'}</p>
                    <p><strong>Service:</strong> ${isUnlocked ? (lead.service?.source_name || 'General Service') : 'Hidden'}</p>
                    <p><strong>Quality Score:</strong> ${isUnlocked ? (lead.quality_score || 'N/A') : 'Hidden'}</p>
                    <p><strong>Unlock Price:</strong> ${lead.calculated_unlock_price || '0.00'} Credits</p>
                </div>
                <div class="col-md-6">
                    <h6>Lead Details</h6>
                    <p><strong>Posted:</strong> ${formatDate(lead.created_at)}</p>
                    ${lead.expires_at ? `<p><strong>Expires:</strong> ${formatDate(lead.expires_at)}</p>` : ''}
                    <p><strong>Status:</strong> <span class="badge bg-${getStatusColor(lead.marketplace_status)}">${lead.marketplace_status}</span></p>
                    <p><strong>Unlocks:</strong> ${lead.unlock_count || 0}</p>
                </div>
            </div>
        `;

        if (isUnlocked) {
            detailsHtml += `
                <hr>
                <div class="row">
                    <div class="col-12">
                        <h6>Client Information</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong>Name:</strong> ${lead.form_inputs?.first_name || 'N/A'} ${lead.form_inputs?.last_name || 'N/A'}</p>
                                <p><strong>Email:</strong> ${lead.form_inputs?.email || 'N/A'}</p>
                                <p><strong>Phone:</strong> ${lead.form_inputs?.phone || 'N/A'}</p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>City:</strong> ${lead.form_inputs?.city || 'N/A'}</p>
                                <p><strong>Location:</strong> ${lead.form_inputs?.city || 'N/A'}, ${lead.form_inputs?.state || 'N/A'}</p>
                            </div>
                        </div>
                        ${lead.message_text ? `
                            <h6 class="mt-3">Message</h6>
                            <div class="alert alert-light">
                                <p class="mb-0">${lead.message_text}</p>
                            </div>
                        ` : ''}
                        
                        <!-- Question Responses Section -->
                        <div id="leadQuestionResponses_${lead.id}" class="mt-3">
                            <h6>Questions Answered</h6>
                            <div class="text-center py-2">
                                <div class="spinner-border spinner-border-sm text-primary" role="status"></div>
                                <small class="text-muted d-block mt-1">Loading answers...</small>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            // Load question responses asynchronously
            setTimeout(() => loadLeadQuestionResponses(lead.id), 100);
        } else {
            detailsHtml += `
                <hr>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>
                    Unlock this lead to see the client's contact information and message.
                </div>
            `;
        }

        $content.html(detailsHtml);

        // Show/hide unlock button
        if (!isUnlocked) {
            $('#unlockLeadBtn').show().attr('data-lead-id', leadId).attr('data-price', lead.calculated_unlock_price);
        } else {
            $('#unlockLeadBtn').hide();
        }
    }

    /**
     * Unlock lead
     */
    function unlockLead(leadId) {
        $('#confirmUnlockBtn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin me-1"></i>Unlocking...');

        $.ajax({
            url: '/provider/marketplace/unlock',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: JSON.stringify({ lead_id: leadId }),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message || 'Lead unlocked successfully!');
                    $('#unlockConfirmModal').modal('hide');
                    loadWalletBalance();
                    loadLeads();
                    
                    // Automatically switch to "My Unlocked Leads" tab
                    $('#unlocked-tab').tab('show');
                    loadUnlockedLeads();
                } else {
                    toastr.error(response.message || 'Failed to unlock lead');
                }
            },
            error: function(xhr) {
                console.error('Unlock failed:', xhr);
                const response = xhr.responseJSON;
                const errorMsg = response?.message || 'Failed to unlock lead';
                
                // Log the full error for debugging
                console.log('Error response:', response);
                console.log('Status:', xhr.status);
                
                if (response?.message === 'Insufficient wallet balance') {
                    // Show detailed insufficient balance message with top-up prompt
                    const required = response.required || '0.00';
                    const current = response.current || '0.00';
                    const shortfall = response.shortfall || '0.00';
                    
                    toastr.error(`
                        <strong>Insufficient Credits!</strong><br>
                        Required: ${required} Credits<br>
                        Current: ${current} Credits<br>
                        Shortfall: ${shortfall} Credits<br><br>
                        <a href="/provider/wallet" class="btn btn-sm btn-primary mt-2">Top Up Wallet</a>
                    `, '', {
                        timeOut: 10000,
                        extendedTimeOut: 5000,
                        closeButton: true,
                        escapeHtml: false
                    });
                } else {
                    toastr.error(errorMsg, 'Error', {
                        timeOut: 8000,
                        closeButton: true
                    });
                }
            },
            complete: function() {
                $('#confirmUnlockBtn').prop('disabled', false).html('<i class="fas fa-unlock me-1"></i>Unlock Lead');
            }
        });
    }

    /**
     * Render pagination
     */
    function renderPagination() {
        const $nav = $('#paginationNav');
        $nav.empty();

        if (totalPages <= 1) return;

        let paginationHtml = '<ul class="pagination">';

        // Previous button
        if (currentPage > 1) {
            paginationHtml += `<li class="page-item"><a class="page-link pagination-link" href="#" data-page="${currentPage - 1}">Previous</a></li>`;
        }

        // Page numbers
        for (let i = 1; i <= totalPages; i++) {
            if (i === currentPage) {
                paginationHtml += `<li class="page-item active"><span class="page-link">${i}</span></li>`;
            } else {
                paginationHtml += `<li class="page-item"><a class="page-link pagination-link" href="#" data-page="${i}">${i}</a></li>`;
            }
        }

        // Next button
        if (currentPage < totalPages) {
            paginationHtml += `<li class="page-item"><a class="page-link pagination-link" href="#" data-page="${currentPage + 1}">Next</a></li>`;
        }

        paginationHtml += '</ul>';
        $nav.html(paginationHtml);
    }

    /**
     * Get status color class
     */
    function getStatusColor(status) {
        const colors = {
            'pending': 'warning',
            'unlocked': 'info',
            'in_progress': 'primary',
            'completed': 'success',
            'expired': 'danger',
            'cancelled': 'secondary'
        };
        return colors[status] || 'secondary';
    }

    /**
     * Load question responses for a lead
     */
    function loadLeadQuestionResponses(leadId) {
        $.ajax({
            url: `/api/leads/${leadId}/question-responses`,
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                const container = $(`#leadQuestionResponses_${leadId}`);
                
                if (response.success && response.responses && response.responses.length > 0) {
                    let html = '<h6 class="mb-3">Questions Answered</h6>';
                    
                    response.responses.forEach(function(resp, index) {
                        html += `
                            <div class="card mb-2 border-0 shadow-sm">
                                <div class="card-body p-3" style="background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);">
                                    <p class="mb-1 small text-muted">Q${index + 1}:</p>
                                    <p class="mb-2 fw-bold">${resp.question.question_text}</p>
                                    <div class="alert alert-light mb-0 py-2 px-3 border-0" style="background-color: white;">
                                        <i class="fas fa-check-circle text-success me-1"></i>
                                        <strong>${resp.answer_text || (resp.answer ? resp.answer.answer_text : 'N/A')}</strong>
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                    
                    container.html(html);
                } else {
                    container.html('<p class="text-muted small">No questions were answered</p>');
                }
            },
            error: function(xhr) {
                console.error('Error loading question responses:', xhr);
                const container = $(`#leadQuestionResponses_${leadId}`);
                container.html('<p class="text-muted small">No questions answered</p>');
            }
        });
    }

    /**
     * Format date
     */
    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }

    /**
     * Load unlocked leads
     */
    function loadUnlockedLeads(page = 1) {
        $('#unlockedLeadsGrid').html('<div class="col-12 text-center py-5"><i class="fas fa-spinner fa-spin fa-2x text-primary"></i></div>');
        
        $.ajax({
            url: '/provider/marketplace/unlocked',
            type: 'POST',
            data: { page: page },
            success: function(response) {
                if (response.success && response.data.length > 0) {
                    displayUnlockedLeads(response.data);
                    displayUnlockedPagination(response.pagination);
                } else {
                    $('#unlockedLeadsGrid').html(`
                        <div class="col-12 text-center py-5">
                            <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                            <p class="text-muted">You haven't unlocked any leads yet</p>
                        </div>
                    `);
                }
            },
            error: function(xhr) {
                console.error('Error loading unlocked leads:', xhr);
                $('#unlockedLeadsGrid').html(`
                    <div class="col-12 text-center py-5">
                        <i class="fas fa-exclamation-circle fa-3x text-danger mb-3"></i>
                        <p class="text-danger">Error loading unlocked leads</p>
                    </div>
                `);
            }
        });
    }

    /**
     * Display unlocked leads
     */
    function displayUnlockedLeads(leads) {
        let html = '';
        
        leads.forEach(function(lead) {
            const statusBadge = getConnectionStatusBadge(lead.connection_status);
            
            html += `
                <div class="col-md-6 col-lg-4 mb-4">
                    <div class="card lead-card h-100">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-start mb-2">
                                <h6 class="card-title mb-0">${lead.category}</h6>
                                ${statusBadge}
                            </div>
                            ${lead.service ? `<p class="text-muted small mb-2"><i class="fas fa-wrench me-1"></i>${lead.service}</p>` : ''}
                            
                            <div class="mb-2">
                                <strong><i class="fas fa-user me-1"></i></strong>${lead.client_name}<br>
                                <strong><i class="fas fa-envelope me-1"></i></strong><a href="mailto:${lead.client_email}">${lead.client_email}</a><br>
                                <strong><i class="fas fa-phone me-1"></i></strong><a href="tel:${lead.client_phone}">${lead.client_phone}</a><br>
                                <strong><i class="fas fa-map-marker-alt me-1"></i></strong>${lead.location}
                            </div>
                            
                            <p class="small text-muted mb-3">${lead.message_preview}</p>
                            
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <small class="text-muted">
                                    <i class="fas fa-clock me-1"></i>${lead.unlocked_at}
                                </small>
                                <span class="badge bg-info">${lead.credits_spent} Credits</span>
                            </div>
                            
                            <!-- Action Buttons -->
                            <div class="btn-group w-100 mb-2" role="group">
                                <a href="mailto:${lead.client_email}" class="btn btn-sm btn-primary">
                                    <i class="fas fa-envelope me-1"></i>Email
                                </a>
                                <a href="tel:${lead.client_phone}" class="btn btn-sm btn-success">
                                    <i class="fas fa-phone me-1"></i>Call
                                </a>
                            </div>
                            ${lead.connection_status === 'pending' ? `
                                <button class="btn btn-sm btn-outline-primary w-100 send-connection-btn" data-unlock-id="${lead.unlock_id}">
                                    <i class="fas fa-handshake me-1"></i>Request Connection
                                </button>
                            ` : ''}
                            ${lead.connection_status === 'accepted' ? `
                                <a href="/bookings/create?lead_id=${lead.lead_id}" class="btn btn-sm btn-primary w-100">
                                    <i class="fas fa-calendar-check me-1"></i>Book Appointment
                                </a>
                            ` : ''}
                        </div>
                    </div>
                </div>
            `;
        });
        
        $('#unlockedLeadsGrid').html(html);
    }

    /**
     * Get connection status badge
     */
    function getConnectionStatusBadge(status) {
        const badges = {
            'pending': '<span class="badge bg-warning">Pending</span>',
            'accepted': '<span class="badge bg-success">Accepted</span>',
            'declined': '<span class="badge bg-danger">Declined</span>',
            'expired': '<span class="badge bg-secondary">Expired</span>'
        };
        return badges[status] || '<span class="badge bg-secondary">Unknown</span>';
    }

    /**
     * Send connection request to client
     */
    function sendConnectionRequest(unlockId) {
        if (!confirm('Send a connection request to this client? They will be able to see your profile and contact details.')) {
            return;
        }
        
        $.ajax({
            url: '/provider/marketplace/connect',
            type: 'POST',
            data: { unlock_id: unlockId },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message || 'Connection request sent!');
                    loadUnlockedLeads(); // Reload to update status
                } else {
                    toastr.error(response.message || 'Failed to send connection request');
                }
            },
            error: function(xhr) {
                console.error('Connection request failed:', xhr);
                const response = xhr.responseJSON;
                toastr.error(response?.message || 'Failed to send connection request');
            }
        });
    }

    /**
     * Display unlocked pagination
     */
    function displayUnlockedPagination(pagination) {
        if (pagination.last_page <= 1) {
            $('#unlockedPaginationNav').html('');
            return;
        }

        let html = '<ul class="pagination">';
        for (let i = 1; i <= pagination.last_page; i++) {
            const active = i === pagination.current_page ? 'active' : '';
            html += `<li class="page-item ${active}"><a class="page-link unlocked-pagination-link" href="#" data-page="${i}">${i}</a></li>`;
        }
        html += '</ul>';
        
        $('#unlockedPaginationNav').html(html);
        
        // Handle pagination clicks
        $('.unlocked-pagination-link').on('click', function(e) {
            e.preventDefault();
            const page = $(this).data('page');
            loadUnlockedLeads(page);
        });
    }

})();
