/**
 * Client Submitted Requests Dashboard
 * Handles client's submitted marketplace requests management
 */

(function() {
    'use strict';

    // Helper function to escape HTML
    function escapeHtml(text) {
        if (!text) return '';
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    // Global variables
    let currentRequests = [];
    let currentPage = 1;
    let totalPages = 1;
    let currentRequestId = null;
    let currentUnlockId = null;
    let filters = {
        status: ''
    };

    // Initialize when document is ready
    $(document).ready(function() {
        initClientDashboard();
    });

    /**
     * Initialize client dashboard
     */
    function initClientDashboard() {
        setupEventListeners();
        loadRequests();
    }

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Tab filter changes
        $('.nav-tabs button[data-status]').on('click', function() {
            filters.status = $(this).data('status');
            currentPage = 1;
            loadRequests();
        });

        // Refresh button
        $('#refreshBtn').on('click', function() {
            loadRequests();
        });

        // Modal events
        $('#requestDetailsModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const requestId = button.data('request-id');
            if (requestId) {
                loadRequestDetails(requestId);
            }
        });

        $('#providersModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const requestId = button.data('request-id');
            if (requestId) {
                loadInterestedProviders(requestId);
            }
        });

        $('#connectionResponseModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const unlockId = button.data('provider-id'); // Actually unlock_id from data attribute
            const requestId = button.data('request-id');
            
            console.log('Modal opening:', {
                button: button,
                unlockId: unlockId,
                requestId: requestId,
                buttonData: button.data()
            });
            
            if (unlockId && requestId) {
                currentUnlockId = unlockId;
                currentRequestId = requestId;
                loadProviderInfo(unlockId);
            } else {
                console.error('Missing unlock_id or request_id!', {unlockId, requestId});
            }
        });

        $('#cancelConfirmModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const requestId = button.data('request-id');
            if (requestId) {
                currentRequestId = requestId;
            }
        });

        // Action buttons
        $('#acceptConnectionBtn').on('click', function() {
            const message = $('#connectionMessage').val();
            handleConnectionResponse('accept', message);
        });

        $('#declineConnectionBtn').on('click', function() {
            handleConnectionResponse('decline');
        });

        $('#confirmCancelBtn').on('click', function() {
            cancelRequest();
        });

        // Pagination
        $(document).on('click', '.pagination-link', function(e) {
            e.preventDefault();
            const page = $(this).data('page');
            if (page && page !== currentPage) {
                currentPage = page;
                loadRequests();
            }
        });
    }

    /**
     * Load client's submitted requests
     */
    function loadRequests() {
        const requestData = {
            page: currentPage,
            status: filters.status
        };

        $.ajax({
            url: '/user/submitted-requests/list',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: JSON.stringify(requestData),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    currentRequests = response.data || [];
                    totalPages = response.pagination?.last_page || 1;
                    renderRequests();
                    renderPagination();
                } else {
                    toastr.error(response.message || 'Failed to load requests');
                }
            },
            error: function(xhr) {
                console.error('Error loading requests:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load submitted requests');
            }
        });
    }

    /**
     * Render requests grid
     */
    function renderRequests() {
        const $grid = $('#requestsGrid');
        $grid.empty();

        if (currentRequests.length === 0) {
            $grid.html(`
                <div class="col-12">
                    <div class="card text-center py-5">
                        <div class="card-body">
                            <i class="ti ti-file-text fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No requests found</h5>
                            <p class="text-muted">You haven't submitted any requests yet, or try adjusting your filter.</p>
                        </div>
                    </div>
                </div>
            `);
            return;
        }

        currentRequests.forEach(function(request) {
            const requestCard = createRequestCard(request);
            $grid.append(requestCard);
        });
    }

    /**
     * Create request card HTML
     */
    function createRequestCard(request) {
        const statusColor = getStatusColor(request.marketplace_status);
        const statusText = request.marketplace_status ? request.marketplace_status.charAt(0).toUpperCase() + request.marketplace_status.slice(1) : 'Unknown';
        
        return $(`
            <div class="col-lg-6 col-xl-4 mb-4">
                <div class="card request-card h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <div>
                                <h6 class="card-title mb-1">${request.category?.name || 'General Request'}</h6>
                                <small class="text-muted">${request.service?.source_name || 'General Service'}</small>
                            </div>
                            <div class="text-end">
                                <span class="badge bg-${statusColor}">${statusText}</span>
                            </div>
                        </div>
                        
                        <p class="card-text text-muted small mb-3" style="height: 60px; overflow: hidden;">
                            ${request.message_text ? escapeHtml(request.message_text.substring(0, 100) + (request.message_text.length > 100 ? '...' : '')) : 'No message provided'}
                        </p>
                        
                        <div class="mb-3">
                            <p class="card-text small text-muted mb-2">
                                <i class="ti ti-calendar me-1"></i>
                                ${request.time_ago || formatDate(request.created_at)}
                            </p>
                            ${request.expires_at ? `
                                <p class="card-text small text-muted mb-2">
                                    <i class="ti ti-clock me-1"></i>
                                    Expires ${formatDate(request.expires_at)}
                                </p>
                            ` : ''}
                            <p class="card-text small text-muted mb-0">
                                <i class="ti ti-users me-1"></i>
                                ${request.providers_interested || 0} provider(s) interested
                            </p>
                        </div>

                        <div class="d-flex gap-2">
                            <button class="btn btn-sm btn-outline-primary flex-fill" 
                                    data-bs-toggle="modal" 
                                    data-bs-target="#requestDetailsModal"
                                    data-request-id="${request.id}">
                                <i class="ti ti-eye me-1"></i>Details
                            </button>
                            ${(request.providers_interested || 0) > 0 ? `
                                <button class="btn btn-sm btn-primary flex-fill" 
                                        data-bs-toggle="modal" 
                                        data-bs-target="#providersModal"
                                        data-request-id="${request.id}">
                                    <i class="ti ti-users me-1"></i>Providers (${request.providers_interested})
                                </button>
                            ` : ''}
                        </div>
                    </div>
                </div>
            </div>
        `);
    }

    /**
     * Load request details for modal
     */
    function loadRequestDetails(requestId) {
        const request = currentRequests.find(r => r.id == requestId);
        if (!request) return;

        const $content = $('#requestDetailsContent');
        
        let detailsHtml = `
            <div class="row">
                <div class="col-md-6">
                    <h6>Service Information</h6>
                    <p><strong>Category:</strong> ${request.category?.name || 'N/A'}</p>
                    <p><strong>Service:</strong> ${request.service?.source_name || 'General Service'}</p>
                    <p><strong>Quality Score:</strong> ${request.quality_score || 'N/A'}</p>
                </div>
                <div class="col-md-6">
                    <h6>Request Details</h6>
                    <p><strong>Status:</strong> <span class="badge bg-${getStatusColor(request.marketplace_status)}">${request.marketplace_status}</span></p>
                    <p><strong>Submitted:</strong> ${formatDate(request.created_at)}</p>
                    ${request.expires_at ? `<p><strong>Expires:</strong> ${formatDate(request.expires_at)}</p>` : ''}
                    <p><strong>Interested Providers:</strong> ${request.unlock_count || 0}</p>
                </div>
            </div>
            <hr>
            <div class="row">
                <div class="col-12">
                    <h6>Your Information</h6>
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Name:</strong> ${request.form_inputs?.first_name || 'N/A'} ${request.form_inputs?.last_name || 'N/A'}</p>
                            <p><strong>Email:</strong> ${request.form_inputs?.email || 'N/A'}</p>
                            <p><strong>Phone:</strong> ${request.form_inputs?.phone || 'N/A'}</p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>Location:</strong> ${request.form_inputs?.city || 'N/A'}, ${request.form_inputs?.state || 'N/A'}</p>
                            <p><strong>Postal Code:</strong> ${request.form_inputs?.postal_code || 'N/A'}</p>
                        </div>
                    </div>
                    ${request.message_text ? `
                        <h6 class="mt-3">Your Message</h6>
                        <div class="alert alert-light">
                            <p class="mb-0">${escapeHtml(request.message_text)}</p>
                        </div>
                    ` : ''}
                </div>
            </div>
        `;

        $content.html(detailsHtml);

        // Show/hide cancel button
        if (request.marketplace_status === 'pending' || request.marketplace_status === 'unlocked') {
            $('#cancelRequestBtn').show().attr('data-request-id', requestId);
        } else {
            $('#cancelRequestBtn').hide();
        }
    }

    /**
     * Load interested providers
     */
    function loadInterestedProviders(requestId) {
        $.ajax({
            url: `/user/submitted-requests/${requestId}/providers`,
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    renderProviders(response.data.providers || []);
                } else {
                    toastr.error(response.message || 'Failed to load providers');
                }
            },
            error: function(xhr) {
                console.error('Error loading providers:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load interested providers');
            }
        });
    }

    /**
     * Render providers list
     */
    function renderProviders(providers) {
        const $content = $('#providersContent');
        
        if (providers.length === 0) {
            $content.html(`
                <div class="text-center py-4">
                    <i class="fas fa-users fa-3x text-muted mb-3"></i>
                    <h5 class="text-muted">No providers interested yet</h5>
                    <p class="text-muted">Providers will appear here when they unlock your request.</p>
                </div>
            `);
            return;
        }

        let providersHtml = '<div class="row">';
        
        providers.forEach(function(provider) {
            providersHtml += `
                <div class="col-md-6 mb-3">
                    <div class="provider-card p-3 border rounded">
                        <div class="d-flex align-items-center mb-3">
                            <img src="${provider.provider_profile_image || '/assets/img/profile-default.png'}" 
                                 class="rounded-circle me-3" 
                                 style="width: 50px; height: 50px; object-fit: cover;"
                                 alt="Provider Avatar">
                            <div class="flex-grow-1">
                                <h6 class="mb-1">${provider.provider_name || 'N/A'}</h6>
                                <small class="text-muted">${provider.provider_email || 'N/A'}</small>
                                <div class="mt-1">
                                    <small class="text-muted">
                                        <i class="ti ti-clock me-1"></i>Unlocked ${provider.unlocked_at || 'recently'}
                                    </small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="d-flex justify-content-between align-items-center">
                            <span class="badge ${provider.connection_status === 'pending' ? 'bg-warning' : provider.connection_status === 'accepted' ? 'bg-success' : 'bg-secondary'}">
                                ${provider.connection_status === 'pending' ? 'Pending' : provider.connection_status === 'accepted' ? 'Connected' : 'Declined'}
                            </span>
                            <div>
                                ${provider.connection_status === 'pending' ? `
                                    <button class="btn btn-sm btn-outline-primary" 
                                            data-bs-toggle="modal" 
                                            data-bs-target="#connectionResponseModal"
                                            data-provider-id="${provider.unlock_id}"
                                            data-request-id="${provider.lead_id}">
                                        <i class="ti ti-message me-1"></i>Respond
                                    </button>
                                ` : provider.connection_status === 'accepted' ? `
                                    <a href="mailto:${provider.provider_email}" class="btn btn-sm btn-outline-info me-1">
                                        <i class="ti ti-mail me-1"></i>Email
                                    </a>
                                    <a href="tel:${provider.provider_phone}" class="btn btn-sm btn-outline-primary">
                                        <i class="ti ti-phone me-1"></i>Call
                                    </a>
                                ` : `
                                    <small class="text-muted">Declined</small>
                                `}
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        providersHtml += '</div>';
        $content.html(providersHtml);
    }

    /**
     * Load provider info for connection response
     */
    function loadProviderInfo(providerId) {
        // This would typically load provider details
        $('#connectionProviderInfo').html(`
            <div class="alert alert-info">
                <h6>Provider wants to connect with you</h6>
                <p class="mb-0">This provider has unlocked your request and wants to work with you.</p>
            </div>
        `);
    }

    /**
     * Handle connection response (accept/decline)
     */
    function handleConnectionResponse(action, message = '') {
        const requestData = {
            unlock_id: currentUnlockId,
            message: message
        };

        console.log('Sending connection response:', {
            action: action,
            requestData: requestData,
            currentUnlockId: currentUnlockId
        });

        $.ajax({
            url: action === 'accept' ? '/user/submitted-requests/accept' : '/user/submitted-requests/decline',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: JSON.stringify(requestData),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message || `Connection ${action}ed successfully!`);
                    $('#connectionResponseModal').modal('hide');
                    loadRequests();
                    // Reload providers modal if open
                    if ($('#providersModal').hasClass('show') && currentRequestId) {
                        loadInterestedProviders(currentRequestId);
                    }
                } else {
                    toastr.error(response.message || `Failed to ${action} connection`);
                }
            },
            error: function(xhr) {
                const errorMsg = xhr.responseJSON?.message || `Failed to ${action} connection`;
                toastr.error(errorMsg);
            }
        });
    }

    /**
     * Cancel request
     */
    function cancelRequest() {
        $('#confirmCancelBtn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin me-1"></i>Cancelling...');

        $.ajax({
            url: '/user/submitted-requests/cancel',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: JSON.stringify({ lead_id: currentRequestId }),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message || 'Request cancelled successfully!');
                    $('#cancelConfirmModal').modal('hide');
                    loadRequests();
                } else {
                    toastr.error(response.message || 'Failed to cancel request');
                }
            },
            error: function(xhr) {
                const errorMsg = xhr.responseJSON?.message || 'Failed to cancel request';
                toastr.error(errorMsg);
            },
            complete: function() {
                $('#confirmCancelBtn').prop('disabled', false).html('<i class="fas fa-times me-1"></i>Yes, Cancel Request');
            }
        });
    }

    /**
     * Render pagination
     */
    function renderPagination() {
        const $nav = $('#paginationNav');
        $nav.empty();

        if (totalPages <= 1) return;

        let paginationHtml = '<ul class="pagination">';

        // Previous button
        if (currentPage > 1) {
            paginationHtml += `<li class="page-item"><a class="page-link pagination-link" href="#" data-page="${currentPage - 1}">Previous</a></li>`;
        }

        // Page numbers
        for (let i = 1; i <= totalPages; i++) {
            if (i === currentPage) {
                paginationHtml += `<li class="page-item active"><span class="page-link">${i}</span></li>`;
            } else {
                paginationHtml += `<li class="page-item"><a class="page-link pagination-link" href="#" data-page="${i}">${i}</a></li>`;
            }
        }

        // Next button
        if (currentPage < totalPages) {
            paginationHtml += `<li class="page-item"><a class="page-link pagination-link" href="#" data-page="${currentPage + 1}">Next</a></li>`;
        }

        paginationHtml += '</ul>';
        $nav.html(paginationHtml);
    }

    /**
     * Get status color class
     */
    function getStatusColor(status) {
        const colors = {
            'pending': 'warning',
            'unlocked': 'info',
            'in_progress': 'primary',
            'completed': 'success',
            'expired': 'danger',
            'cancelled': 'secondary'
        };
        return colors[status] || 'secondary';
    }

    /**
     * Format date
     */
    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }

})();
