/**
 * Client Linked Providers Management
 * Shows all providers linked to user's requests with delink functionality
 */

(function() {
    'use strict';

    let currentUnlockId = null;

    // Initialize when document is ready
    $(document).ready(function() {
        loadLinkedProviders();
        setupEventListeners();
    });

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Delink confirmation modal
        $('#delinkConfirmModal').on('show.bs.modal', function(e) {
            const button = $(e.relatedTarget);
            const unlockId = button.data('unlock-id');
            const providerName = button.data('provider-name');
            const requestCategory = button.data('request-category');
            
            if (unlockId) {
                currentUnlockId = unlockId;
                $('#delinkProviderInfo').html(`
                    <div class="alert alert-warning">
                        <p class="mb-1"><strong>Provider:</strong> ${providerName}</p>
                        <p class="mb-0"><strong>Request:</strong> ${requestCategory}</p>
                    </div>
                `);
            }
        });

        // Confirm delink
        $('#confirmDelinkBtn').on('click', function() {
            delinkProvider();
        });
    }

    /**
     * Load all linked providers across all user requests
     */
    function loadLinkedProviders() {
        $.ajax({
            url: '/user/linked-providers/list',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    renderLinkedProviders(response.data || []);
                } else {
                    showNoLinkedProviders();
                }
            },
            error: function(xhr) {
                console.error('Error loading linked providers:', xhr.responseJSON?.message || 'Unknown error');
                showNoLinkedProviders();
            }
        });
    }

    /**
     * Render linked providers
     */
    function renderLinkedProviders(linkedProviders) {
        const container = $('#linkedProvidersList');
        
        if (linkedProviders.length === 0) {
            showNoLinkedProviders();
            return;
        }
        
        let html = '';
        linkedProviders.forEach(link => {
            html += `
                <div class="col-lg-6 mb-4">
                    <div class="card h-100">
                        <div class="card-body">
                            <div class="d-flex align-items-start justify-content-between mb-3">
                                <div class="d-flex align-items-center">
                                    <img src="${link.provider_profile_image || '/assets/img/profile-default.png'}" 
                                         class="rounded-circle me-3" 
                                         style="width: 50px; height: 50px; object-fit: cover;"
                                         alt="Provider">
                                    <div>
                                        <h6 class="mb-1">${link.provider_name}</h6>
                                        <small class="text-muted">${link.provider_email}</small>
                                    </div>
                                </div>
                                <span class="badge bg-success">Connected</span>
                            </div>
                            
                            <div class="mb-3">
                                <p class="mb-2"><strong>Request:</strong> ${link.request_category}</p>
                                <p class="mb-2"><strong>Service:</strong> ${link.request_service || 'General Service'}</p>
                                <p class="mb-0"><small class="text-muted">Connected ${link.connected_at}</small></p>
                            </div>
                            
                            <div class="d-flex gap-2">
                                <a href="/user/chat/${link.provider_id}" class="btn btn-sm btn-outline-primary flex-fill">
                                    <i class="ti ti-message-circle me-1"></i>Chat
                                </a>
                                <a href="tel:${link.provider_phone}" class="btn btn-sm btn-outline-success flex-fill">
                                    <i class="ti ti-phone me-1"></i>Call
                                </a>
                                <button class="btn btn-sm btn-outline-danger flex-fill" 
                                        data-bs-toggle="modal" 
                                        data-bs-target="#delinkConfirmModal"
                                        data-unlock-id="${link.unlock_id}"
                                        data-provider-name="${link.provider_name}"
                                        data-request-category="${link.request_category}">
                                    <i class="ti ti-unlink me-1"></i>Delink
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        container.html(html);
    }

    /**
     * Show no linked providers message
     */
    function showNoLinkedProviders() {
        const container = $('#linkedProvidersList');
        container.html(`
            <div class="col-12">
                <div class="text-center py-5">
                    <i class="ti ti-link-off fs-48 text-muted mb-3"></i>
                    <h5 class="text-muted">{{ __('No Linked Providers') }}</h5>
                    <p class="text-muted">{{ __('You haven\'t connected with any providers yet.') }}</p>
                    <a href="{{ route('user.submitted-requests') }}" class="btn btn-primary">
                        <i class="ti ti-arrow-left me-1"></i>{{ __('View My Requests') }}
                    </a>
                </div>
            </div>
        `);
    }

    /**
     * Delink provider
     */
    function delinkProvider() {
        if (!currentUnlockId) return;

        $('#confirmDelinkBtn').prop('disabled', true).html('<i class="ti ti-loader fa-spin me-1"></i>Delinking...');

        $.ajax({
            url: '/user/submitted-requests/delink',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            data: JSON.stringify({ unlock_id: currentUnlockId }),
            contentType: 'application/json',
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message || 'Provider delinked successfully!');
                    $('#delinkConfirmModal').modal('hide');
                    loadLinkedProviders(); // Reload list
                } else {
                    toastr.error(response.message || 'Failed to delink provider');
                }
            },
            error: function(xhr) {
                const errorMsg = xhr.responseJSON?.message || 'Failed to delink provider';
                toastr.error(errorMsg);
            },
            complete: function() {
                $('#confirmDelinkBtn').prop('disabled', false).html('<i class="ti ti-unlink me-1"></i>Yes, Delink Provider');
            }
        });
    }

})();

