/**
 * Admin Marketplace Analytics Dashboard
 * Handles analytics data visualization and reporting
 */

(function() {
    'use strict';

    // Global variables
    let performanceChart = null;
    let categoryChart = null;
    let currentDateRange = 30;

    // Initialize when document is ready
    $(document).ready(function() {
        initAnalyticsDashboard();
    });

    /**
     * Initialize analytics dashboard
     */
    function initAnalyticsDashboard() {
        setupEventListeners();
        loadAllAnalytics();
    }

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Date range filter change
        $('#dateRangeFilter').on('change', function() {
            currentDateRange = $(this).val();
            loadAllAnalytics();
        });

        // Refresh button
        $('#refreshAnalytics').on('click', function() {
            loadAllAnalytics();
        });
    }

    /**
     * Load all analytics data
     */
    function loadAllAnalytics() {
        loadOverviewStats();
        loadPerformanceChart();
        loadCategoryChart();
        loadTopProviders();
        loadPricingAnalytics();
    }

    /**
     * Load overview statistics
     */
    function loadOverviewStats() {
        $.ajax({
            url: '/admin/leads/analytics/overview',
            type: 'GET',
            data: { date_range: currentDateRange },
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    updateOverviewStats(response.data);
                } else {
                    toastr.error(response.message || 'Failed to load overview stats');
                }
            },
            error: function(xhr) {
                console.error('Error loading overview stats:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load overview statistics');
            }
        });
    }

    /**
     * Update overview statistics display
     */
    function updateOverviewStats(data) {
        $('#totalLeads').text(data.total_leads || 0);
        $('#totalRevenue').text('$' + (data.total_revenue || 0).toFixed(2));
        $('#totalUnlocks').text(data.total_unlocks || 0);
        $('#activeProviders').text(data.active_providers || 0);
        $('#conversionRate').text((data.conversion_rate || 0).toFixed(2) + '%');
        $('#avgUnlockPrice').text('$' + (data.avg_unlock_price || 0).toFixed(2));
    }

    /**
     * Load performance over time chart
     */
    function loadPerformanceChart() {
        $.ajax({
            url: '/admin/leads/analytics/performance',
            type: 'GET',
            data: { date_range: currentDateRange },
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    renderPerformanceChart(response.data);
                } else {
                    toastr.error(response.message || 'Failed to load performance data');
                }
            },
            error: function(xhr) {
                console.error('Error loading performance data:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load performance chart');
            }
        });
    }

    /**
     * Render performance chart
     */
    function renderPerformanceChart(data) {
        const ctx = document.getElementById('performanceChart');
        if (!ctx) return;

        // Destroy existing chart
        if (performanceChart) {
            performanceChart.destroy();
        }

        const labels = data.map(item => item.date);
        const leadsData = data.map(item => item.leads);
        const unlocksData = data.map(item => item.unlocks);
        const revenueData = data.map(item => item.revenue);

        performanceChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Leads',
                        data: leadsData,
                        borderColor: 'rgb(75, 192, 192)',
                        backgroundColor: 'rgba(75, 192, 192, 0.1)',
                        yAxisID: 'y',
                    },
                    {
                        label: 'Unlocks',
                        data: unlocksData,
                        borderColor: 'rgb(54, 162, 235)',
                        backgroundColor: 'rgba(54, 162, 235, 0.1)',
                        yAxisID: 'y',
                    },
                    {
                        label: 'Revenue ($)',
                        data: revenueData,
                        borderColor: 'rgb(255, 99, 132)',
                        backgroundColor: 'rgba(255, 99, 132, 0.1)',
                        yAxisID: 'y1',
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false,
                },
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: {
                            display: true,
                            text: 'Count'
                        }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: {
                            display: true,
                            text: 'Revenue ($)'
                        },
                        grid: {
                            drawOnChartArea: false,
                        },
                    },
                }
            }
        });
    }

    /**
     * Load leads by category chart
     */
    function loadCategoryChart() {
        $.ajax({
            url: '/admin/leads/analytics/categories',
            type: 'GET',
            data: { date_range: currentDateRange },
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    renderCategoryChart(response.data);
                } else {
                    toastr.error(response.message || 'Failed to load category data');
                }
            },
            error: function(xhr) {
                console.error('Error loading category data:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load category chart');
            }
        });
    }

    /**
     * Render category chart
     */
    function renderCategoryChart(data) {
        const ctx = document.getElementById('categoryChart');
        if (!ctx) return;

        // Destroy existing chart
        if (categoryChart) {
            categoryChart.destroy();
        }

        const labels = data.map(item => item.category_name);
        const counts = data.map(item => item.lead_count);

        categoryChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Leads',
                    data: counts,
                    backgroundColor: [
                        'rgba(255, 99, 132, 0.8)',
                        'rgba(54, 162, 235, 0.8)',
                        'rgba(255, 206, 86, 0.8)',
                        'rgba(75, 192, 192, 0.8)',
                        'rgba(153, 102, 255, 0.8)',
                        'rgba(255, 159, 64, 0.8)',
                        'rgba(199, 199, 199, 0.8)',
                        'rgba(83, 102, 255, 0.8)',
                        'rgba(255, 99, 255, 0.8)',
                        'rgba(99, 255, 132, 0.8)'
                    ],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.parsed || 0;
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((value / total) * 100).toFixed(1);
                                return `${label}: ${value} (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
    }

    /**
     * Load top providers table
     */
    function loadTopProviders() {
        $.ajax({
            url: '/admin/leads/analytics/top-providers',
            type: 'GET',
            data: { date_range: currentDateRange },
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    renderTopProvidersTable(response.data);
                } else {
                    toastr.error(response.message || 'Failed to load providers data');
                }
            },
            error: function(xhr) {
                console.error('Error loading providers data:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load top providers');
            }
        });
    }

    /**
     * Render top providers table
     */
    function renderTopProvidersTable(data) {
        const $tbody = $('#topProvidersTable tbody');
        $tbody.empty();

        if (data.length === 0) {
            $tbody.html('<tr><td colspan="4" class="text-center">No provider data available</td></tr>');
            return;
        }

        data.forEach(function(provider) {
            const row = `
                <tr>
                    <td>
                        <div>${provider.provider_name}</div>
                        <small class="text-muted">${provider.provider_email}</small>
                    </td>
                    <td>${provider.unlocks}</td>
                    <td>$${provider.total_spent}</td>
                    <td>$${provider.avg_spent}</td>
                </tr>
            `;
            $tbody.append(row);
        });
    }

    /**
     * Load pricing analytics
     */
    function loadPricingAnalytics() {
        $.ajax({
            url: '/admin/leads/analytics/pricing',
            type: 'GET',
            data: { date_range: currentDateRange },
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    renderPricingTable(response.data.pricing_distribution);
                } else {
                    toastr.error(response.message || 'Failed to load pricing data');
                }
            },
            error: function(xhr) {
                console.error('Error loading pricing data:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to load pricing analytics');
            }
        });
    }

    /**
     * Render pricing distribution table
     */
    function renderPricingTable(data) {
        const $tbody = $('#pricingTable tbody');
        $tbody.empty();

        if (data.length === 0) {
            $tbody.html('<tr><td colspan="4" class="text-center">No pricing data available</td></tr>');
            return;
        }

        data.forEach(function(item) {
            const revenue = item.count * item.avg_price;
            const row = `
                <tr>
                    <td>$${item.price_range}</td>
                    <td>${item.count}</td>
                    <td>$${parseFloat(item.avg_price).toFixed(2)}</td>
                    <td>$${revenue.toFixed(2)}</td>
                </tr>
            `;
            $tbody.append(row);
        });
    }

    /**
     * Export analytics data
     */
    window.exportAnalytics = function(type) {
        $.ajax({
            url: '/admin/leads/analytics/export',
            type: 'GET',
            data: { 
                type: type,
                date_range: currentDateRange 
            },
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    // Convert data to CSV
                    const csv = convertToCSV(response.data, type);
                    const filename = `marketplace-${type}-${new Date().toISOString().split('T')[0]}.csv`;
                    downloadCSV(csv, filename);
                    toastr.success('Data exported successfully');
                } else {
                    toastr.error(response.message || 'Failed to export data');
                }
            },
            error: function(xhr) {
                console.error('Error exporting data:', xhr.responseJSON?.message || 'Unknown error');
                toastr.error('Failed to export analytics data');
            }
        });
    };

    /**
     * Convert data to CSV format
     */
    function convertToCSV(data, type) {
        if (!data || data.length === 0) {
            return '';
        }

        // Handle overview type differently
        if (type === 'overview' && !Array.isArray(data)) {
            const headers = Object.keys(data).join(',');
            const values = Object.values(data).join(',');
            return `${headers}\n${values}`;
        }

        // Get headers from first object
        const headers = Object.keys(data[0]).join(',');
        
        // Get rows
        const rows = data.map(row => {
            return Object.values(row).map(value => {
                // Escape quotes and wrap in quotes if contains comma
                if (typeof value === 'string' && (value.includes(',') || value.includes('"'))) {
                    return `"${value.replace(/"/g, '""')}"`;
                }
                return value;
            }).join(',');
        });

        return `${headers}\n${rows.join('\n')}`;
    }

    /**
     * Download CSV file
     */
    function downloadCSV(csv, filename) {
        const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        
        if (navigator.msSaveBlob) {
            // IE 10+
            navigator.msSaveBlob(blob, filename);
        } else {
            link.href = URL.createObjectURL(blob);
            link.download = filename;
            link.style.display = 'none';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    }

})();
