/**
 * Admin Help Form Questions Management
 * Handles CRUD operations for dynamic help form questions
 */

(function() {
    'use strict';

    let isEditMode = false;
    let currentQuestionId = null;
    let answerCount = 0;

    // Initialize when document is ready
    $(document).ready(function() {
        loadQuestions();
        setupEventListeners();
    });

    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        // Question type change
        $('#question_type').on('change', function() {
            const type = $(this).val();
            if (type === 'radio' || type === 'checkbox' || type === 'select') {
                $('#answersSection').show();
                if ($('#answersList').children().length === 0) {
                    addAnswerField();
                    addAnswerField();
                }
            } else {
                $('#answersSection').hide();
                $('#answersList').empty();
            }
        });

        // Add answer button
        $('#addAnswerBtn').on('click', function() {
            addAnswerField();
        });

        // Form submission
        $('#questionForm').on('submit', function(e) {
            e.preventDefault();
            saveQuestion();
        });

        // Modal reset
        $('#addQuestionModal').on('hidden.bs.modal', function() {
            resetForm();
        });

        // Toggle status
        $(document).on('change', '.toggle-status', function() {
            const questionId = $(this).data('id');
            toggleStatus(questionId);
        });

        // Edit question
        $(document).on('click', '.btn-edit', function() {
            const questionId = $(this).data('id');
            editQuestion(questionId);
        });

        // Delete question
        $(document).on('click', '.btn-delete', function() {
            const questionId = $(this).data('id');
            deleteQuestion(questionId);
        });
    }

    /**
     * Load all questions
     */
    function loadQuestions() {
        $.ajax({
            url: '/admin/leads/help-form-questions/list',
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    displayQuestions(response.data);
                } else {
                    toastr.error('Failed to load questions');
                }
            },
            error: function(xhr) {
                console.error('Error loading questions:', xhr.responseJSON);
                toastr.error('Error loading questions');
            }
        });
    }

    /**
     * Display questions in table
     */
    function displayQuestions(questions) {
        const tbody = $('#questionsList');
        tbody.empty();

        if (questions.length === 0) {
            tbody.html(`
                <tr>
                    <td colspan="7" class="text-center text-muted py-4">
                        {{ __('No questions found. Click "Add Question" to create one.') }}
                    </td>
                </tr>
            `);
            return;
        }

        questions.forEach(function(question) {
            const checked = question.is_active ? 'checked' : '';
            const statusBadge = question.is_active ? 
                '<span class="badge bg-success">Active</span>' : 
                '<span class="badge bg-secondary">Inactive</span>';
            const requiredBadge = question.is_required ? 
                '<span class="badge bg-warning">Required</span>' : 
                '<span class="badge bg-secondary">Optional</span>';
            const answerCount = question.answers ? question.answers.length : 0;
            const typeLabels = {
                'radio': 'Single Choice',
                'checkbox': 'Multiple Choice',
                'select': 'Dropdown',
                'text': 'Text Input'
            };

            const row = `
                <tr>
                    <td>
                        <span class="badge bg-primary">${question.order}</span>
                    </td>
                    <td>${question.question_text}</td>
                    <td>${typeLabels[question.question_type] || question.question_type}</td>
                    <td>${requiredBadge}</td>
                    <td>
                        ${answerCount > 0 ? `<span class="badge bg-info">${answerCount} options</span>` : '-'}
                    </td>
                    <td>
                        <div class="form-check form-switch">
                            <input class="form-check-input toggle-status" type="checkbox" 
                                   ${checked} data-id="${question.id}">
                        </div>
                    </td>
                    <td>
                        <button class="btn btn-sm btn-info me-1 btn-edit" data-id="${question.id}" title="Edit">
                            <i class="fa fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-danger btn-delete" data-id="${question.id}" title="Delete">
                            <i class="fa fa-trash"></i>
                        </button>
                    </td>
                </tr>
            `;

            tbody.append(row);
        });
    }

    /**
     * Add answer field
     */
    function addAnswerField(answerText = '') {
        answerCount++;
        const html = `
            <div class="input-group mb-2 answer-field">
                <input type="text" class="form-control" name="answers[${answerCount}][text]" 
                       placeholder="Answer option ${answerCount}" value="${answerText}" required>
                <button class="btn btn-outline-danger" type="button" onclick="$(this).closest('.answer-field').remove()">
                    <i class="fa fa-trash"></i>
                </button>
            </div>
        `;
        $('#answersList').append(html);
    }

    /**
     * Save question
     */
    function saveQuestion() {
        const formData = new FormData($('#questionForm')[0]);
        const url = isEditMode ? 
            `/admin/leads/help-form-questions/${currentQuestionId}` : 
            '/admin/leads/help-form-questions';
        const method = isEditMode ? 'PUT' : 'POST';

        // Convert FormData to JSON for proper handling
        const data = {};
        formData.forEach((value, key) => {
            if (key.startsWith('answers[')) {
                if (!data.answers) data.answers = [];
                const match = key.match(/answers\[(\d+)\]\[(\w+)\]/);
                if (match) {
                    const index = parseInt(match[1]) - 1;
                    const field = match[2];
                    if (!data.answers[index]) data.answers[index] = {};
                    data.answers[index][field] = value;
                }
            } else {
                data[key] = value;
            }
        });

        data.is_required = $('#is_required').is(':checked');

        // Show loading
        $('#btnSaveQuestion .spinner-border').removeClass('d-none');
        $('#btnSaveQuestion .btn-text').text('Saving...');
        $('#btnSaveQuestion').prop('disabled', true);

        $.ajax({
            url: url,
            type: method,
            data: JSON.stringify(data),
            contentType: 'application/json',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    $('#addQuestionModal').modal('hide');
                    loadQuestions();
                    resetForm();
                } else {
                    toastr.error(response.message);
                }
            },
            error: function(xhr) {
                const errors = xhr.responseJSON?.errors;
                if (errors) {
                    Object.keys(errors).forEach(function(field) {
                        toastr.error(errors[field][0]);
                    });
                } else {
                    toastr.error('Error saving question');
                }
            },
            complete: function() {
                $('#btnSaveQuestion .spinner-border').addClass('d-none');
                $('#btnSaveQuestion .btn-text').text('Save');
                $('#btnSaveQuestion').prop('disabled', false);
            }
        });
    }

    /**
     * Edit question
     */
    function editQuestion(id) {
        $.ajax({
            url: `/admin/leads/help-form-questions/${id}`,
            type: 'GET',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    const question = response.data;
                    isEditMode = true;
                    currentQuestionId = id;

                    $('#question_id').val(question.id);
                    $('#question_text').val(question.question_text);
                    $('#question_type').val(question.question_type).trigger('change');
                    $('#is_required').prop('checked', question.is_required);
                    $('#modalTitle').text('Edit Question');

                    // Load answers if exist
                    if (question.answers && question.answers.length > 0) {
                        $('#answersList').empty();
                        answerCount = 0;
                        question.answers.forEach(function(answer) {
                            addAnswerField(answer.answer_text);
                        });
                    }

                    $('#addQuestionModal').modal('show');
                } else {
                    toastr.error('Error loading question');
                }
            },
            error: function() {
                toastr.error('Error loading question');
            }
        });
    }

    /**
     * Delete question
     */
    function deleteQuestion(id) {
        if (!confirm('Are you sure you want to delete this question? This action cannot be undone.')) {
            return;
        }

        $.ajax({
            url: `/admin/leads/help-form-questions/${id}`,
            type: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    loadQuestions();
                } else {
                    toastr.error(response.message);
                }
            },
            error: function() {
                toastr.error('Error deleting question');
            }
        });
    }

    /**
     * Toggle question status
     */
    function toggleStatus(id) {
        $.ajax({
            url: `/admin/leads/help-form-questions/${id}/toggle-status`,
            type: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    loadQuestions();
                } else {
                    toastr.error(response.message);
                }
            },
            error: function() {
                toastr.error('Error updating status');
            }
        });
    }

    /**
     * Reset form
     */
    function resetForm() {
        isEditMode = false;
        currentQuestionId = null;
        answerCount = 0;
        $('#questionForm')[0].reset();
        $('#question_id').val('');
        $('#answersList').empty();
        $('#answersSection').hide();
        $('#modalTitle').text('Add Question');
    }

})();

