<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\ParadigmPayment;
use App\Models\ParadigmAccessToken;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ResetFreeAssessmentGerdaevansSeeder extends Seeder
{
    /**
     * Reset free assessment for Gerdaevans50@gmail.com and remove duplicate payments
     */
    public function run(): void
    {
        $email = 'Gerdaevans50@gmail.com';
        
        DB::beginTransaction();
        try {
            // Find the user
            $user = User::where('email', $email)->first();
            
            if (!$user) {
                $this->command->error("User with email {$email} not found.");
                return;
            }
            
            $this->command->info("Found user: {$user->name} (ID: {$user->id})");
            
            // Find all free assessment payments for this email
            $freePayments = ParadigmPayment::where('client_email', $email)
                ->where('paid_by', 'free')
                ->get();
            
            $this->command->info("Found {$freePayments->count()} free assessment payment(s) to delete.");
            
            // Delete related access tokens first
            foreach ($freePayments as $payment) {
                $accessTokens = ParadigmAccessToken::where('payment_id', $payment->id)->get();
                foreach ($accessTokens as $token) {
                    $token->delete();
                    $this->command->info("Deleted access token ID: {$token->id}");
                }
                
                // Delete the payment
                $payment->delete();
                $this->command->info("Deleted payment ID: {$payment->id} (Payment Link: {$payment->payment_link})");
            }
            
            // Reset user's free assessment flags
            $user->update([
                'free_assessment_used' => false,
                'free_assessment_id' => null
            ]);
            
            $this->command->info("Reset free_assessment_used and free_assessment_id for user ID: {$user->id}");
            
            DB::commit();
            
            $this->command->info("Successfully reset free assessment for {$email} and removed {$freePayments->count()} duplicate payment(s).");
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error("Error: " . $e->getMessage());
            Log::error('ResetFreeAssessmentGerdaevansSeeder failed', [
                'email' => $email,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}

