<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\ParadigmQuestion;
use Illuminate\Support\Facades\DB;

class FixParadigmConnectionStylesGrammarSeeder extends Seeder
{
    /**
     * Fix grammar in Part 5: Connection Styles questions
     * 
     * CRITICAL: This seeder ONLY updates 2 specific grammar issues:
     * 1. Question 4 - Option A: "help" → "helps"
     * 2. Question 20 - Option B: "got" → "get"
     * 
     * NO OTHER QUESTIONS ARE MODIFIED
     */
    public function run(): void
    {
        $this->command->info('🔍 Starting grammar fix for Part 5: Connection Styles...');
        
        try {
            DB::beginTransaction();
            
            // Fix 1: Question 4 - Option A
            $question4 = ParadigmQuestion::where('assessment_module', 'connection_styles')
                ->where('question_type', 'part5')
                ->where('display_order', 4)
                ->where('question_text', 'It means a lot to me when...')
                ->where('option_a_label', "Someone help me with something I'm working on.")
                ->first();
            
            if ($question4) {
                $oldOptionA = $question4->option_a_label;
                $question4->option_a_label = "Someone helps me with something I'm working on.";
                $question4->save();
                
                $this->command->info('✅ Question 4 - Option A updated:');
                $this->command->line("   OLD: {$oldOptionA}");
                $this->command->line("   NEW: {$question4->option_a_label}");
            } else {
                $this->command->warn('⚠️  Question 4 not found or already updated');
            }
            
            // Fix 2: Question 20 - Option B
            $question20 = ParadigmQuestion::where('assessment_module', 'connection_styles')
                ->where('question_type', 'part5')
                ->where('display_order', 20)
                ->where('question_text', 'It means a lot to me when...')
                ->where('option_b_label', 'I got a gift that someone picked out just for me.')
                ->first();
            
            if ($question20) {
                $oldOptionB = $question20->option_b_label;
                $question20->option_b_label = 'I get a gift that someone picked out just for me.';
                $question20->save();
                
                $this->command->info('✅ Question 20 - Option B updated:');
                $this->command->line("   OLD: {$oldOptionB}");
                $this->command->line("   NEW: {$question20->option_b_label}");
            } else {
                $this->command->warn('⚠️  Question 20 not found or already updated');
            }
            
            DB::commit();
            
            $this->command->info('');
            $this->command->info('✅ Grammar fixes completed successfully!');
            $this->command->info('📝 Summary:');
            $this->command->line('   - Question 4, Option A: "help" → "helps"');
            $this->command->line('   - Question 20, Option B: "got" → "get"');
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error('❌ Error: ' . $e->getMessage());
            throw $e;
        }
    }
}

