<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\UserDetail;
use App\Models\Bookings;
use App\Models\Cart;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\ParadigmPayment;
use App\Models\ParadigmAssessmentResult;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DeleteUserAldiserveSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * IMPORTANT: This will permanently delete the user aldiserve@gmail.com and ALL their data.
     */
    public function run(): void
    {
        $email = 'aldiserve@gmail.com';
        
        DB::beginTransaction();
        
        try {
            // Find user including soft deleted ones
            $user = User::withTrashed()->where('email', $email)->first();
            
            if (!$user) {
                echo "❌ User with email {$email} not found.\n";
                DB::rollBack();
                return;
            }
            
            echo "👤 Found user: {$user->name} (ID: {$user->id})\n";
            echo "📋 User Type: {$user->user_type}\n";
            
            // Safety check - only delete normal users (user_type = 3)
            if ($user->user_type != 3) {
                echo "⚠️  WARNING: This user is NOT a normal user (user_type = {$user->user_type}).\n";
                echo "⛔ Deletion cancelled for safety. This seeder only deletes normal users (user_type = 3).\n";
                DB::rollBack();
                return;
            }
            
            $userId = $user->id;
            $deletedCount = [];
            
            // 1. Delete User Details (soft deletes and hard deletes)
            $count = UserDetail::withTrashed()->where('user_id', $userId)->forceDelete();
            $deletedCount['user_details'] = $count;
            echo "🗑️  Deleted {$count} user detail records\n";
            
            // 2. Delete Bookings (soft deletes and hard deletes - if table exists)
            if (DB::getSchemaBuilder()->hasTable('bookings')) {
                $count = DB::table('bookings')->where('user_id', $userId)->delete();
                $deletedCount['bookings'] = $count;
                echo "🗑️  Deleted {$count} booking records\n";
            } else {
                echo "⚠️  Bookings table doesn't exist - skipping\n";
            }
            
            // 3. Delete Cart items (if table exists)
            if (DB::getSchemaBuilder()->hasTable('carts')) {
                $count = DB::table('carts')->where('user_id', $userId)->delete();
                $deletedCount['cart'] = $count;
                echo "🗑️  Deleted {$count} cart records\n";
            } else {
                echo "⚠️  Carts table doesn't exist - skipping\n";
            }
            
            // 4. Delete Wallet History (if table exists)
            if (DB::getSchemaBuilder()->hasTable('wallet_history')) {
                $count = DB::table('wallet_history')->where('user_id', $userId)->delete();
                $deletedCount['wallet_history'] = $count;
                echo "🗑️  Deleted {$count} wallet history records\n";
            } else {
                echo "⚠️  Wallet history table doesn't exist - skipping\n";
            }
            
            // 5. Delete Reviews (as reviewer - if table exists)
            if (DB::getSchemaBuilder()->hasTable('reviews')) {
                $count = DB::table('reviews')->where('user_id', $userId)->delete();
                $deletedCount['reviews'] = $count;
                echo "🗑️  Deleted {$count} review records\n";
            } else {
                echo "⚠️  Reviews table doesn't exist - skipping\n";
            }
            
            // 6. Delete User Notifications (if table exists)
            if (DB::getSchemaBuilder()->hasTable('user_notifications')) {
                $count = DB::table('user_notifications')->where('user_id', $userId)->delete();
                $deletedCount['notifications'] = $count;
                echo "🗑️  Deleted {$count} notification records\n";
            } else {
                echo "⚠️  User notifications table doesn't exist - skipping\n";
            }
            
            // 7. Delete Paradigm Payments (if table exists)
            // User could be client_id (taking assessment) or provider_id (creating assessments)
            if (DB::getSchemaBuilder()->hasTable('paradigm_payments')) {
                $count = DB::table('paradigm_payments')
                    ->where('client_id', $userId)
                    ->orWhere('provider_id', $userId)
                    ->delete();
                $deletedCount['paradigm_payments'] = $count;
                echo "🗑️  Deleted {$count} paradigm payment records\n";
            } else {
                echo "⚠️  Paradigm payments table doesn't exist - skipping\n";
            }
            
            // 8. Delete Paradigm Assessment Results (if table exists)
            // User could be provider_id (created assessments) or email (took assessments)
            if (DB::getSchemaBuilder()->hasTable('paradigm_assessment_results')) {
                $count = DB::table('paradigm_assessment_results')
                    ->where('provider_id', $userId)
                    ->orWhere('email', $email)
                    ->delete();
                $deletedCount['paradigm_results'] = $count;
                echo "🗑️  Deleted {$count} paradigm assessment records\n";
            } else {
                echo "⚠️  Paradigm assessment results table doesn't exist - skipping\n";
            }
            
            // 9. Delete Conversations (where user is participant - if tables exist)
            if (DB::getSchemaBuilder()->hasTable('conversation_user') && DB::getSchemaBuilder()->hasTable('conversations')) {
                $conversationIds = DB::table('conversation_user')
                    ->where('user_id', $userId)
                    ->pluck('conversation_id')
                    ->toArray();
                
                if (!empty($conversationIds)) {
                    // Delete messages in these conversations
                    if (DB::getSchemaBuilder()->hasTable('messages')) {
                        $count = DB::table('messages')->whereIn('conversation_id', $conversationIds)->delete();
                        echo "🗑️  Deleted {$count} message records\n";
                    }
                    
                    // Delete conversation participants
                    $count = DB::table('conversation_user')->where('user_id', $userId)->delete();
                    echo "🗑️  Deleted {$count} conversation participant records\n";
                    
                    // Delete conversations where this user was the only participant
                    foreach ($conversationIds as $convId) {
                        $remainingParticipants = DB::table('conversation_user')
                            ->where('conversation_id', $convId)
                            ->count();
                        
                        if ($remainingParticipants == 0) {
                            DB::table('conversations')->where('id', $convId)->delete();
                        }
                    }
                }
            } else {
                echo "⚠️  Conversation tables don't exist - skipping\n";
            }
            
            // 10. Delete User Documents (if table exists)
            if (DB::getSchemaBuilder()->hasTable('user_documents')) {
                $count = DB::table('user_documents')->where('user_id', $userId)->delete();
                $deletedCount['user_documents'] = $count;
                echo "🗑️  Deleted {$count} user document records\n";
            }
            
            // 11. Delete Lead Unlocks (if user unlocked any leads)
            if (DB::getSchemaBuilder()->hasTable('lead_unlocks')) {
                $count = DB::table('lead_unlocks')->where('provider_id', $userId)->delete();
                $deletedCount['lead_unlocks'] = $count;
                echo "🗑️  Deleted {$count} lead unlock records\n";
            }
            
            // 12. Delete User Form Inputs (leads submitted by this user)
            if (DB::getSchemaBuilder()->hasTable('user_form_inputs')) {
                $count = DB::table('user_form_inputs')->where('user_id', $userId)->delete();
                $deletedCount['user_form_inputs'] = $count;
                echo "🗑️  Deleted {$count} user form input records\n";
            }
            
            // 13. Delete Password Reset Tokens (if table exists)
            if (DB::getSchemaBuilder()->hasTable('password_reset_tokens')) {
                $count = DB::table('password_reset_tokens')->where('email', $email)->delete();
                $deletedCount['password_reset_tokens'] = $count;
                echo "🗑️  Deleted {$count} password reset token records\n";
            } else {
                echo "⚠️  Password reset tokens table doesn't exist - skipping\n";
            }
            
            // 14. Delete Personal Access Tokens (Sanctum)
            if (DB::getSchemaBuilder()->hasTable('personal_access_tokens')) {
                $count = DB::table('personal_access_tokens')
                    ->where('tokenable_type', 'App\\Models\\User')
                    ->where('tokenable_id', $userId)
                    ->delete();
                $deletedCount['personal_access_tokens'] = $count;
                echo "🗑️  Deleted {$count} personal access token records\n";
            }
            
            // 15. Delete Device Tokens (if table exists)
            if (DB::getSchemaBuilder()->hasTable('device_tokens')) {
                $count = DB::table('device_tokens')->where('user_id', $userId)->delete();
                $deletedCount['device_tokens'] = $count;
                echo "🗑️  Deleted {$count} device token records\n";
            }
            
            // FINAL: Force delete the user (including soft deleted)
            $user->forceDelete();
            echo "✅ User {$email} has been permanently deleted!\n";
            
            // Summary
            echo "\n📊 DELETION SUMMARY:\n";
            echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            foreach ($deletedCount as $table => $count) {
                if ($count > 0) {
                    echo "  - {$table}: {$count} records\n";
                }
            }
            echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            echo "✅ All data for {$email} has been permanently deleted!\n\n";
            
            DB::commit();
            
            Log::info("User {$email} (ID: {$userId}) and all related data permanently deleted.", $deletedCount);
            
        } catch (\Exception $e) {
            DB::rollBack();
            echo "❌ ERROR: " . $e->getMessage() . "\n";
            echo "Stack trace: " . $e->getTraceAsString() . "\n";
            Log::error("Failed to delete user {$email}: " . $e->getMessage());
            throw $e;
        }
    }
}

