<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CreateVerificationEmailTemplatesSeeder extends Seeder
{
    /**
     * Create Provider Verification Email Templates
     * 
     * CREATES TWO NEW TEMPLATES:
     * 1. Type 31 - Professional Details Verified (sent when admin verifies professional details badge)
     * 2. Type 6 - Identity Verification Completed (sent when admin verifies ID + Selfie)
     * 
     * SPECIFICATIONS:
     * - Button text: "Login"
     * - Button link: https://app.iseekhelpoffice.com
     * - Removed: "Activated Date: {{date}}"
     * - Removed: "Start building your profile and accepting bookings today!"
     */
    public function run(): void
    {
        $this->command->info('========================================');
        $this->command->info('📧 CREATING VERIFICATION EMAIL TEMPLATES');
        $this->command->info('========================================');
        $this->command->info('');
        
        try {
            DB::beginTransaction();
            
            // ============================================
            // 1. CREATE PROFESSIONAL DETAILS VERIFIED (Type 31)
            // ============================================
            $this->command->info('Creating Type 31 - Professional Details Verified...');
            
            // Check if already exists
            $existing31 = DB::table('templates')
                ->where('notification_type', 31)
                ->where('type', 1)
                ->whereNull('deleted_at')
                ->first();
            
            if ($existing31) {
                $this->command->warn("   ⚠️  Already exists (ID: {$existing31->id}), skipping...");
            } else {
                $id31 = DB::table('templates')->insertGetId([
                    'notification_type' => 31,
                    'type' => 1,
                    'title' => '{{category_name}} Details Verified',
                    'subject' => '{{category_name}} Details Verified - {{company_name}}',
                    'content' => $this->getProfessionalDetailsTemplate(),
                    'status' => 1,
                    'created_by' => 1,
                    'updated_by' => 1,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
                
                $this->command->info("   ✅ Created (ID: {$id31})");
                $this->command->line('      Title: Professional Details Verified');
                $this->command->line('      Subject: Professional Details Verified - {{company_name}}');
                $this->command->line('      Button: Login → https://app.iseekhelpoffice.com');
            }
            
            $this->command->info('');
            
            // ============================================
            // 2. CREATE IDENTITY VERIFICATION COMPLETED (Type 6)
            // ============================================
            $this->command->info('Creating Type 6 - Identity Verification Completed...');
            
            // Check if already exists
            $existing6 = DB::table('templates')
                ->where('notification_type', 6)
                ->where('type', 1)
                ->whereNull('deleted_at')
                ->first();
            
            if ($existing6) {
                $this->command->warn("   ⚠️  Already exists (ID: {$existing6->id}), skipping...");
            } else {
                $id6 = DB::table('templates')->insertGetId([
                    'notification_type' => 6,
                    'type' => 1,
                    'title' => '{{category_name}} Verification Completed',
                    'subject' => '{{category_name}} Verification Completed - {{company_name}}',
                    'content' => $this->getIdentityVerificationTemplate(),
                    'status' => 1,
                    'created_by' => 1,
                    'updated_by' => 1,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
                
                $this->command->info("   ✅ Created (ID: {$id6})");
                $this->command->line('      Title: Identity Verification Completed');
                $this->command->line('      Subject: Identity Verification Completed - {{company_name}}');
                $this->command->line('      Removed: Activated Date, Start building...');
                $this->command->line('      Button: Login → https://app.iseekhelpoffice.com');
            }
            
            DB::commit();
            
            $this->command->info('');
            $this->command->info('========================================');
            $this->command->info('✅ VERIFICATION TEMPLATES CREATED!');
            $this->command->info('========================================');
            $this->command->info('📝 Both templates ready for use:');
            $this->command->line('   • Type 31 - Professional Details Verified');
            $this->command->line('   • Type 6 - Identity Verification Completed');
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error('❌ Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Professional Details Verified Template (Type 31)
     */
    private function getProfessionalDetailsTemplate(): string
    {
        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Professional Details Verified!</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f4f4f4; }
        .container { max-width: 600px; margin: 20px auto; background: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .header { background: linear-gradient(135deg, #044f54 0%, #066a70 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
        .header h1 { margin: 0; font-size: 24px; font-weight: 600; }
        .content { padding: 30px 20px; }
        .content h2 { color: #044f54; margin-top: 0; font-size: 20px; }
        .content p { margin: 15px 0; color: #555; }
        .button-container { text-align: center; margin: 30px 0; }
        .button { display: inline-block; padding: 14px 40px; background: #044f54; color: #ffffff !important; text-decoration: none; border-radius: 5px; font-weight: 600; font-size: 16px; transition: background 0.3s ease; }
        .button:hover { background: #066a70; }
        .info-box { background: #f8f9fa; border-left: 4px solid #044f54; padding: 15px; margin: 20px 0; border-radius: 4px; }
        .info-box p { margin: 5px 0; font-size: 14px; }
        .footer { background: #f8f9fa; padding: 20px; text-align: center; font-size: 13px; color: #6c757d; border-top: 1px solid #e9ecef; }
        .footer a { color: #044f54; text-decoration: none; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>{{category_name}} Details Verified!</h1>
        </div>
        <div class="content">
            <h2>Congratulations {{provider_name}}!</h2>
            
            <p>Your {{category_name}} details have been verified on I Seek Help International.</p>
            
            <div class="info-box">
                <p><strong>Account Status:</strong> ✅ {{category_name}} Details Verified</p>
            </div>
            
            <p>You can now access your {{category_name}} dashboard and complete your identity verification to start offering services as a {{category_name}}.</p>
            
            <div class="button-container">
                <a href="https://app.iseekhelpoffice.com" class="button">Login</a>
            </div>
        </div>
        <div class="footer">
            <p><strong>{{company_name}}</strong></p>
            <p>{{company_email}} | {{company_phone}}</p>
            <p><a href="{{website_link}}">Visit our website</a></p>
            <p style="margin-top: 15px; color: #999;">This is an automated email. Please do not reply to this message.</p>
        </div>
    </div>
</body>
</html>';
    }

    /**
     * Identity Verification Completed Template (Type 6)
     */
    private function getIdentityVerificationTemplate(): string
    {
        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Identity Verification Completed!</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f4f4f4; }
        .container { max-width: 600px; margin: 20px auto; background: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .header { background: linear-gradient(135deg, #044f54 0%, #066a70 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
        .header h1 { margin: 0; font-size: 24px; font-weight: 600; }
        .content { padding: 30px 20px; }
        .content h2 { color: #044f54; margin-top: 0; font-size: 20px; }
        .content p { margin: 15px 0; color: #555; }
        .button-container { text-align: center; margin: 30px 0; }
        .button { display: inline-block; padding: 14px 40px; background: #044f54; color: #ffffff !important; text-decoration: none; border-radius: 5px; font-weight: 600; font-size: 16px; transition: background 0.3s ease; }
        .button:hover { background: #066a70; }
        .info-box { background: #f8f9fa; border-left: 4px solid #044f54; padding: 15px; margin: 20px 0; border-radius: 4px; }
        .info-box p { margin: 5px 0; font-size: 14px; }
        .footer { background: #f8f9fa; padding: 20px; text-align: center; font-size: 13px; color: #6c757d; border-top: 1px solid #e9ecef; }
        .footer a { color: #044f54; text-decoration: none; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>{{category_name}} Verification Completed!</h1>
        </div>
        <div class="content">
            <h2>Congratulations {{provider_name}}!</h2>
            
            <p>Your {{category_name}} identity verification has been completed successfully. You can now start offering your services as a {{category_name}} on I Seek Help International.</p>
            
            <div class="info-box">
                <p><strong>Account Status:</strong> ✅ {{category_name}} Verified</p>
            </div>
            
            <div class="button-container">
                <a href="https://app.iseekhelpoffice.com" class="button">Login</a>
            </div>
        </div>
        <div class="footer">
            <p><strong>{{company_name}}</strong></p>
            <p>{{company_email}} | {{company_phone}}</p>
            <p><a href="{{website_link}}">Visit our website</a></p>
            <p style="margin-top: 15px; color: #999;">This is an automated email. Please do not reply to this message.</p>
        </div>
    </div>
</body>
</html>';
    }
}

