<?php

namespace App\Services;

use App\Repositories\Eloquent\WalletRepository;
use App\Models\WalletHistory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

/**
 * ParadigmWalletService
 * 
 * Wrapper service for Paradigm assessment wallet operations.
 * Uses existing WalletRepository without modifying it.
 * 
 * IMPORTANT: This service does NOT modify any existing wallet functionality.
 * It only provides a safe interface for Paradigm assessments.
 */
class ParadigmWalletService
{
    protected WalletRepository $walletRepo;

    public function __construct()
    {
        $this->walletRepo = new WalletRepository();
    }

    /**
     * Get user's wallet balance
     * Uses existing WalletRepository method
     * 
     * @param int $userId
     * @return float
     */
    public function getBalance(int $userId): float
    {
        $balanceData = $this->walletRepo->getWalletBalance($userId);
        return (float) $balanceData['balance'];
    }

    /**
     * Get detailed wallet balance (total credits, debits, balance)
     * Uses existing WalletRepository method
     * 
     * @param int $userId
     * @return array ['total' => float, 'debit' => float, 'balance' => float]
     */
    public function getDetailedBalance(int $userId): array
    {
        return $this->walletRepo->getWalletBalance($userId);
    }

    /**
     * Check if user has sufficient balance for Paradigm assessment
     * 
     * @param int $userId
     * @param float $amount
     * @return bool
     */
    public function hasSufficientBalance(int $userId, float $amount): bool
    {
        $balance = $this->getBalance($userId);
        return $balance >= $amount;
    }

    /**
     * Deduct amount from user wallet for Paradigm assessment (PROVIDER)
     * Uses existing WalletRepository::processWalletPayment() method
     * 
     * @param int $providerId
     * @param float $amount
     * @param int $paradigmPaymentId - Reference to paradigm_payments.id
     * @param string $description
     * @return WalletHistory|false
     */
    public function deductProviderWallet(int $providerId, float $amount, int $paradigmPaymentId, string $description)
    {
        try {
            // Double-check balance before deducting
            if (!$this->hasSufficientBalance($providerId, $amount)) {
                Log::warning('Insufficient balance for provider', [
                    'provider_id' => $providerId,
                    'required_amount' => $amount,
                    'balance' => $this->getBalance($providerId)
                ]);
                return false;
            }

            // Use existing wallet repository method (same as bookings/leads)
            // NOTE: payment_type must be 'Others' (enum constraint)
            // We use transaction_id to identify Paradigm transactions
            $walletTransaction = $this->walletRepo->processWalletPayment([
                'user_id' => $providerId,
                'amount' => $amount,
                'payment_type' => 'Others', // Using enum value
                'reference_id' => $paradigmPaymentId, // Links to paradigm_payments table
                'transaction_id' => 'PARADIGM-PROVIDER-' . $paradigmPaymentId // Unique identifier
            ]);

            if ($walletTransaction) {
                Log::info('Provider wallet deducted for Paradigm assessment', [
                    'provider_id' => $providerId,
                    'amount' => $amount,
                    'wallet_history_id' => $walletTransaction->id,
                    'paradigm_payment_id' => $paradigmPaymentId
                ]);
            }

            return $walletTransaction;

        } catch (\Exception $e) {
            Log::error('Failed to deduct provider wallet', [
                'provider_id' => $providerId,
                'amount' => $amount,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Deduct amount from client wallet for Paradigm assessment
     * Uses existing WalletRepository::processWalletPayment() method
     * 
     * @param int $clientId
     * @param float $amount
     * @param int $paradigmPaymentId - Reference to paradigm_payments.id
     * @param string $description
     * @return WalletHistory|false
     */
    public function deductClientWallet(int $clientId, float $amount, int $paradigmPaymentId, string $description)
    {
        try {
            // Double-check balance before deducting
            if (!$this->hasSufficientBalance($clientId, $amount)) {
                Log::warning('Insufficient balance for client', [
                    'client_id' => $clientId,
                    'required_amount' => $amount,
                    'balance' => $this->getBalance($clientId)
                ]);
                return false;
            }

            // Use existing wallet repository method (same as bookings/leads)
            // NOTE: payment_type must be 'Others' (enum constraint)
            // We use transaction_id to identify Paradigm transactions
            $walletTransaction = $this->walletRepo->processWalletPayment([
                'user_id' => $clientId,
                'amount' => $amount,
                'payment_type' => 'Others', // Using enum value
                'reference_id' => $paradigmPaymentId, // Links to paradigm_payments table
                'transaction_id' => 'PARADIGM-CLIENT-' . $paradigmPaymentId // Unique identifier
            ]);

            if ($walletTransaction) {
                Log::info('Client wallet deducted for Paradigm assessment', [
                    'client_id' => $clientId,
                    'amount' => $amount,
                    'wallet_history_id' => $walletTransaction->id,
                    'paradigm_payment_id' => $paradigmPaymentId
                ]);
            }

            return $walletTransaction;

        } catch (\Exception $e) {
            Log::error('Failed to deduct client wallet', [
                'client_id' => $clientId,
                'amount' => $amount,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get wallet transaction history for user (Paradigm assessments only)
     * Identifies Paradigm transactions by transaction_id prefix
     * 
     * @param int $userId
     * @param int $limit
     * @return \Illuminate\Support\Collection
     */
    public function getParadigmTransactions(int $userId, int $limit = 50)
    {
        return WalletHistory::where('user_id', $userId)
            ->where('status', 'completed')
            ->where(function ($query) {
                $query->where('transaction_id', 'like', 'PARADIGM-PROVIDER-%')
                      ->orWhere('transaction_id', 'like', 'PARADIGM-CLIENT-%');
            })
            ->orderBy('transaction_date', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Get all wallet transactions for user (including non-Paradigm)
     * Uses direct query to avoid modifying existing repository
     * 
     * @param int $userId
     * @param int $limit
     * @return \Illuminate\Support\Collection
     */
    public function getAllTransactions(int $userId, int $limit = 100)
    {
        $transactions = WalletHistory::where('user_id', $userId)
            ->where('status', 'completed')
            ->orderBy('transaction_date', 'desc')
            ->limit($limit)
            ->get();

        // Calculate running balance for each transaction
        // Since transactions are ordered DESC (newest first), we need to reverse for balance calculation
        $reversedTransactions = $transactions->reverse();
        $runningBalance = 0;
        
        $transactionsWithBalance = $reversedTransactions->map(function ($transaction) use (&$runningBalance) {
            // Calculate balance based on transaction type
            if ($transaction->type == 1) { // Credit
                $runningBalance += $transaction->amount;
            } else { // Debit
                $runningBalance -= $transaction->amount;
            }
            
            // Add balance to transaction object
            $transaction->balance = $runningBalance;
            return $transaction;
        });
        
        // Reverse back to show newest first
        $transactionsWithBalance = $transactionsWithBalance->reverse();

        return $transactionsWithBalance;
    }

    /**
     * Refund Paradigm assessment (if needed)
     * Adds credit back to wallet
     * 
     * @param int $userId
     * @param float $amount
     * @param int $paradigmPaymentId
     * @param string $reason
     * @return WalletHistory|false
     */
    public function refundAssessment(int $userId, float $amount, int $paradigmPaymentId, string $reason = 'Refund')
    {
        try {
            // Use existing addWalletAmount method to credit back
            // NOTE: payment_method becomes payment_type, must use enum value
            $refundTransaction = $this->walletRepo->addWalletAmount([
                'user_id' => $userId,
                'amount' => $amount,
                'payment_method' => 'Others', // Using enum value (becomes payment_type)
                'status' => 'completed',
                'transaction_id' => 'PARADIGM-REFUND-' . $paradigmPaymentId, // Unique identifier
                'type' => 1, // Credit (type 1 = add money)
            ]);

            Log::info('Paradigm assessment refunded', [
                'user_id' => $userId,
                'amount' => $amount,
                'paradigm_payment_id' => $paradigmPaymentId,
                'reason' => $reason
            ]);

            return $refundTransaction;

        } catch (\Exception $e) {
            Log::error('Failed to refund Paradigm assessment', [
                'user_id' => $userId,
                'amount' => $amount,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get currency symbol from existing wallet system
     * 
     * @return string
     */
    public function getCurrencySymbol(): string
    {
        $currencyData = $this->walletRepo->getCurrencySymbol();
        return $currencyData->symbol ?? '$';
    }
}
