<?php

namespace App\Services;

// use Modules\Communication\app\Models\Notifications; // Disabled - Firebase removed
use Modules\Communication\app\Models\Templates;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

/**
 * MarketplaceNotificationService
 * 
 * Service for handling marketplace-specific notifications.
 * Uses existing notification system without modifying it.
 * 
 * IMPORTANT: This service does NOT modify any existing notification functionality.
 * It only provides marketplace-specific notification methods.
 */
class MarketplaceNotificationService
{
    /**
     * Send notification when a lead is unlocked by a provider
     * 
     * @param int $clientId - Client who submitted the lead
     * @param int $providerId - Provider who unlocked the lead
     * @param int $leadId - Lead that was unlocked
     * @param string $categoryName - Category name for context
     * @return bool
     */
    public function notifyLeadUnlocked(int $clientId, int $providerId, int $leadId, string $categoryName): bool
    {
        try {
            // Validate input parameters
            if ($clientId <= 0) {
                throw new \InvalidArgumentException('Invalid client ID');
            }
            
            if ($providerId <= 0) {
                throw new \InvalidArgumentException('Invalid provider ID');
            }
            
            if ($leadId <= 0) {
                throw new \InvalidArgumentException('Invalid lead ID');
            }
            
            if (empty(trim($categoryName))) {
                throw new \InvalidArgumentException('Category name is required');
            }
            
            // Get provider details
            $provider = User::find($providerId);
            if (!$provider) {
                Log::error('Provider not found for lead unlock notification', ['provider_id' => $providerId]);
                return false;
            }

            // Get client details
            $client = User::find($clientId);
            if (!$client) {
                Log::error('Client not found for lead unlock notification', ['client_id' => $clientId]);
                return false;
            }

            // Notification database record disabled - Firebase removed

            // Push notifications disabled - Firebase removed

            // Send email notification to client
            $this->sendEmailNotification($client, 'lead_unlocked', [
                'provider_name' => $provider->name ?? 'A provider',
                'category_name' => $categoryName,
                'lead_id' => $leadId
            ]);

            Log::info('Lead unlock notification sent successfully', [
                'client_id' => $clientId,
                'provider_id' => $providerId,
                'lead_id' => $leadId
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send lead unlock notification', [
                'client_id' => $clientId,
                'provider_id' => $providerId,
                'lead_id' => $leadId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send notification when a connection request is sent
     * 
     * @param int $clientId - Client receiving the connection request
     * @param int $providerId - Provider sending the connection request
     * @param int $leadId - Related lead
     * @param string $message - Optional message from provider
     * @return bool
     */
    public function notifyConnectionRequest(int $clientId, int $providerId, int $leadId, string $message = ''): bool
    {
        try {
            // Get provider details
            $provider = User::find($providerId);
            if (!$provider) {
                Log::error('Provider not found for connection request notification', ['provider_id' => $providerId]);
                return false;
            }

            // Get client details
            $client = User::find($clientId);
            if (!$client) {
                Log::error('Client not found for connection request notification', ['client_id' => $clientId]);
                return false;
            }

            $messageText = $message ? "Message: {$message}" : '';

            // Notification database record disabled - Firebase removed

            // Push notifications disabled - Firebase removed

            // Send email notification to client
            $this->sendEmailNotification($client, 'connection_request', [
                'provider_name' => $provider->name ?? 'A provider',
                'message' => $message,
                'lead_id' => $leadId
            ]);

            Log::info('Connection request notification sent successfully', [
                'client_id' => $clientId,
                'provider_id' => $providerId,
                'lead_id' => $leadId
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send connection request notification', [
                'client_id' => $clientId,
                'provider_id' => $providerId,
                'lead_id' => $leadId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send notification when a connection request is accepted
     * 
     * @param int $providerId - Provider who receives the acceptance
     * @param int $clientId - Client who accepted
     * @param int $leadId - Related lead
     * @return bool
     */
    public function notifyConnectionAccepted(int $providerId, int $clientId, int $leadId): bool
    {
        try {
            // Get client details
            $client = User::find($clientId);
            if (!$client) {
                Log::error('Client not found for connection accepted notification', ['client_id' => $clientId]);
                return false;
            }

            // Get provider details
            $provider = User::find($providerId);
            if (!$provider) {
                Log::error('Provider not found for connection accepted notification', ['provider_id' => $providerId]);
                return false;
            }

            // Notification database record disabled - Firebase removed

            // Push notifications disabled - Firebase removed

            // Send email notification to provider
            $this->sendEmailNotification($provider, 'connection_accepted', [
                'client_name' => $client->name ?? 'A client',
                'lead_id' => $leadId
            ]);

            Log::info('Connection accepted notification sent successfully', [
                'provider_id' => $providerId,
                'client_id' => $clientId,
                'lead_id' => $leadId
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send connection accepted notification', [
                'provider_id' => $providerId,
                'client_id' => $clientId,
                'lead_id' => $leadId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send notification when a request is cancelled
     * 
     * @param int $leadId - Cancelled lead
     * @param array $providerIds - Providers who had unlocked this lead
     * @param string $reason - Reason for cancellation
     * @return bool
     */
    public function notifyRequestCancelled(int $leadId, array $providerIds, string $reason = ''): bool
    {
        try {
            if (empty($providerIds)) {
                return true; // No providers to notify
            }

            $reasonText = $reason ? " Reason: {$reason}" : '';

            foreach ($providerIds as $providerId) {
                $provider = User::find($providerId);
                if (!$provider) continue;

                // Notification database record disabled - Firebase removed

                // Push notifications disabled - Firebase removed
            }

            Log::info('Request cancelled notifications sent successfully', [
                'lead_id' => $leadId,
                'provider_ids' => $providerIds
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send request cancelled notifications', [
                'lead_id' => $leadId,
                'provider_ids' => $providerIds,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    // sendPushNotification method removed - Firebase/push notifications disabled

    /**
     * Send email notification using existing email system
     * 
     * @param User $user - User to send email to
     * @param string $templateKey - Email template key
     * @param array $data - Template data
     * @return bool
     */
    private function sendEmailNotification(User $user, string $templateKey, array $data): bool
    {
        try {
            // Map template keys to notification types
            $notificationTypeMap = [
                'lead_unlocked' => 19, // Accept Leads Provider Notification
                'connection_request_sent' => 18, // New Leads Notification
                'connection_request_accepted' => 21, // Accept Leads User Notification
                'connection_request_rejected' => 22, // Reject Leads User Notification
                'request_cancelled' => 20, // Reject Leads Provider
            ];
            
            $notificationType = $notificationTypeMap[$templateKey] ?? null;
            
            if (!$notificationType) {
                Log::warning('Unknown template key', ['template_key' => $templateKey]);
                return false;
            }
            
            // Get email template
            $template = Templates::where('notification_type', $notificationType)
                ->where('type', 1) // Email type
                ->where('status', 1) // Active only
                ->first();
                
            if (!$template) {
                Log::warning('Email template not found', [
                    'template_key' => $templateKey,
                    'notification_type' => $notificationType
                ]);
                return false;
            }

            // Replace template variables
            $subject = $template->subject;
            $body = $template->content; // Fixed: use 'content' not 'body'
            
            foreach ($data as $key => $value) {
                $subject = str_replace("{{{$key}}}", $value, $subject);
                $body = str_replace("{{{$key}}}", $value, $body);
            }

            // Send email using Laravel's mail system
            Mail::send([], [], function ($message) use ($user, $subject, $body) {
                $message->to($user->email, $user->name)
                        ->subject($subject)
                        ->setBody($body, 'text/html');
            });

            Log::info('Email notification sent successfully', [
                'user_id' => $user->id,
                'template_key' => $templateKey,
                'notification_type' => $notificationType
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send email notification', [
                'user_id' => $user->id,
                'template_key' => $templateKey,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}
