<?php

namespace App\Repositories\Eloquent;

use App\Models\WalletHistory;
use App\Repositories\Contracts\WalletInterface;
use Modules\GlobalSetting\app\Models\Currency;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Leads\app\Models\Payments;
use Illuminate\Support\Facades\Cache;
use Stripe\Stripe;
use Stripe\PaymentIntent;

class WalletRepository implements WalletInterface
{
    public function addWalletAmount(array $data)
    {
        return WalletHistory::create([
            'user_id' => $data['user_id'],
            'amount' => $data['amount'], // Price paid in ZAR
            'credits' => $data['credits'] ?? $data['amount'], // Number of credits (fallback for backward compatibility)
            'payment_type' => $data['payment_method'],
            'status' => $data['status'] ?? 'pending',
            'transaction_id' => $data['transaction_id'] ?? null,
            'transaction_date' => now(),
            'type' => $data['type'] ?? 1,
        ]);
    }

    public function getWalletHistory(int $userId)
    {
        $walletHistory = WalletHistory::where('user_id', $userId)
            ->where('type', '1')
            ->orderBy('transaction_date', 'desc')
            ->get();

        $dateFormatSetting = GlobalSetting::where('key', 'date_format_view')->first();

        return $walletHistory->map(function ($record) use ($dateFormatSetting) {
            $record->transaction_date = \Carbon\Carbon::parse($record->transaction_date)
                ->format($dateFormatSetting->value);
            return $record;
        });
    }

    public function getAllWalletHistory(int $userId)
    {
        // Get all wallet history (credits and debits)
        $walletHistory = WalletHistory::where('user_id', $userId)
            ->orderBy('created_at', 'desc')
            ->get();

        // Add running balance (calculate using credits)
        $balance = 0;
        $historyWithBalance = $walletHistory->reverse()->map(function ($transaction) use (&$balance) {
            if ($transaction->status === 'completed') {
                if ($transaction->type == 1) { // Credit
                    // Use credits column, fallback to amount for old records
                    $balance += $transaction->credits ?? $transaction->amount;
                } else { // Debit
                    $balance -= $transaction->amount;
                }
            }
            $transaction->balance = $balance;
            return $transaction;
        })->reverse()->values();

        return $historyWithBalance;
    }

    public function getWalletBalance(int $userId)
    {
        // Sum credits for Type 1 (Credit/Top-ups) - use 'credits' column, fallback to 'amount' for old records
        $totalCredits = WalletHistory::where('user_id', $userId)
            ->where('status', 'Completed')
            ->where('type', '1')
            ->get()
            ->sum(function($record) {
                return $record->credits ?? $record->amount;
            });

        // Sum credits for Type 2 (Debits/Spending) - use 'amount' for debits as they represent credits spent
        $totalDebits = WalletHistory::where('user_id', $userId)
            ->where('status', 'Completed')
            ->where('type', '2')
            ->sum('amount');

        return [
            'total' => $totalCredits,
            'debit' => $totalDebits,
            'balance' => $totalCredits - $totalDebits
        ];
    }

    public function processWalletPayment(array $data)
    {
        $balance = $this->getWalletBalance($data['user_id'])['balance'];
        
        if ($balance < $data['amount']) {
            return false;
        }

        return WalletHistory::create([
            'user_id' => $data['user_id'],
            'amount' => $data['amount'],
            'payment_type' => $data['payment_type'],
            'status' => 'Completed',
            'reference_id' => $data['reference_id'],
            'transaction_id' => $data['transaction_id'],
            'transaction_date' => now(),
            'type' => 2,
        ]);
    }

    public function processLeadPayment(array $data)
    {
        Payments::where('id', $data['transaction_id'])->update([
            'status' => 2,
            'payment_type' => $data['payment_type']
        ]);

        return $this->addWalletAmount([
            'user_id' => $data['user_id'],
            'amount' => $data['amount'],
            'payment_method' => $data['payment_type'],
            'status' => 'Completed',
            'transaction_id' => $data['transaction_id'],
            'type' => 2,
        ]);
    }

    public function confirmTransaction(string $transactionId, string $status)
    {
        $model = WalletHistory::where('transaction_id', $transactionId)->first();
        
        if ($model) {
            $normalizedStatus = strtolower($status);
            $model->update(['status' => $normalizedStatus]);
            
            \Log::info('Transaction status updated', [
                'transaction_id' => $transactionId,
                'status' => $normalizedStatus,
                'user_id' => $model->user_id,
                'amount' => $model->amount,
                'type' => $model->type
            ]);
            
            // Credits are automatically available because wallet balance is calculated from wallet_history
            // Type 1 (credit) + status 'completed' = Credits are available
            // No need to update users table - balance is calculated dynamically
            
            // Generate invoice for wallet top-up (Type 4) when completed
            if ($normalizedStatus === 'completed' && $model->type == 1) {
                try {
                    $pdfPath = \App\Helpers\InvoiceHelper::generateInvoice(
                        $model->id,
                        $model->amount,
                        4, // Type 4 = Wallet Top-up
                        $model->user_id
                    );
                    \Log::info('Wallet top-up invoice generated', [
                        'wallet_history_id' => $model->id,
                        'pdf_path' => $pdfPath
                    ]);
                } catch (\Exception $e) {
                    \Log::error('Failed to generate invoice: ' . $e->getMessage());
                }
                
                // Send wallet top-up success email (Template 103)
                try {
                    $this->sendWalletTopUpEmail($model);
                } catch (\Exception $e) {
                    \Log::error('Failed to send wallet top-up email: ' . $e->getMessage());
                }
            }
            
            return $model->id;
        }

        $payment = Payments::where('transaction_id', $transactionId)->first();
        
        if ($payment) {
            $payment->update(['status' => 2]);
            return $payment->id;
        }

        return false;
    }

    public function createStripePaymentIntent(float $amount)
    {
        Stripe::setApiKey(config('stripe.test.sk'));
        
        return PaymentIntent::create([
            'amount' => $amount * 100,
            'currency' => 'usd',
            'payment_method_types' => ['card'],
        ]);
    }

    public function getCurrencySymbol()
    {
        return Cache::remember('currency_details', 86400, function () {
            return Currency::select('symbol')->orderBy('id', 'DESC')->where('is_default', 1)->first();
        });
    }
    
    /**
     * Send wallet top-up success email (Template 103)
     */
    private function sendWalletTopUpEmail($walletHistory)
    {
        try {
            // Configure mail
            \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
            
            // Get template 103 (Wallet Top-up Success)
            $template = \DB::table('templates')
                ->where('title', 'Wallet Top-up Successful') // Match by exact title
                ->where('type', 1) // Email type
                ->where('status', 1) // Active
                ->first();
            
            if (!$template) {
                \Log::warning('Wallet top-up email template (103) not found');
                return;
            }
            
            // Get user details
            $user = \App\Models\User::with('userDetails')->find($walletHistory->user_id);
            if (!$user) {
                \Log::warning('User not found for wallet top-up email', ['user_id' => $walletHistory->user_id]);
                return;
            }
            
            // Get company settings
            $settings = \DB::table('general_settings')
                ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no'])
                ->pluck('value', 'key');
            
            $companyName = $settings['company_name'] ?? config('app.name');
            $companyWebsite = $settings['website'] ?? config('app.url');
            $companyEmail = $settings['site_email'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            
            // Get currency
            $currency = $this->getCurrencySymbol();
            $currencySymbol = $currency ? $currency->symbol : 'R';
            
            // Prepare replacements
            $userName = $user->name ?? '';
            $firstName = $user->userDetails->first_name ?? '';
            $lastName = $user->userDetails->last_name ?? '';
            $credits = $walletHistory->credits ?? $walletHistory->amount;
            $amountPaid = $walletHistory->amount;
            $balance = $this->getWalletBalance($user->id)['balance'];
            $transactionId = $walletHistory->transaction_id;
            $transactionDate = $walletHistory->transaction_date ? date('Y-m-d H:i:s', strtotime($walletHistory->transaction_date)) : now()->format('Y-m-d H:i:s');
            
            // Replace placeholders in subject
            $subject = str_replace(
                ['{{user_name}}', '{{first_name}}', '{{last_name}}', '{{amount}}', '{{credits}}', '{{balance}}', '{{transaction_id}}', '{{transaction_date}}', '{{company_name}}', '{{company_email}}', '{{company_phone}}'],
                [$userName, $firstName, $lastName, $currencySymbol . $amountPaid, $credits, $balance, $transactionId, $transactionDate, $companyName, $companyEmail, $companyPhone],
                $template->subject
            );
            
            // Replace placeholders in content
            $content = str_replace(
                ['{{user_name}}', '{{first_name}}', '{{last_name}}', '{{amount}}', '{{credits}}', '{{balance}}', '{{transaction_id}}', '{{transaction_date}}', '{{company_name}}', '{{website_link}}', '{{company_email}}', '{{company_phone}}'],
                [$userName, $firstName, $lastName, $currencySymbol . $amountPaid, $credits, $balance, $transactionId, $transactionDate, $companyName, $companyWebsite, $companyEmail, $companyPhone],
                $template->content
            );
            
            // Send email
            $emailData = [
                'to_email' => $user->email,
                'subject' => $subject,
                'content' => $content
            ];
            
            \Log::info('Sending wallet top-up success email', [
                'to_email' => $user->email,
                'subject' => $subject,
                'credits' => $credits,
                'amount' => $amountPaid
            ]);
            
            $emailRequest = new \Illuminate\Http\Request($emailData);
            $emailController = new \Modules\Communication\app\Http\Controllers\EmailController();
            $emailController->sendEmail($emailRequest);
            
            \Log::info('Wallet top-up success email sent successfully');
            
            // Send wallet top-up WhatsApp notification
            try {
                $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
                $phoneNumber = $user->phone_number ?? null;
                
                if (!empty($phoneNumber)) {
                    $smsRepository->sendWalletTopUpWhatsApp(
                        $phoneNumber,
                        $userName,
                        $currencySymbol . ' ' . number_format($amountPaid, 2),
                        number_format($credits, 0) . ' credits'
                    );
                }
            } catch (\Exception $whatsappEx) {
                \Log::warning('Failed to send wallet top-up WhatsApp: ' . $whatsappEx->getMessage());
            }
            
            // Send admin WhatsApp notification for credits purchase
            try {
                $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
                
                // Get admin WhatsApp number from settings
                $adminWhatsApp = \DB::table('communication_settings')
                    ->where('key', 'admin_whatsapp_number')
                    ->value('value');
                
                if (!empty($adminWhatsApp)) {
                    // Check if user is provider (user_type = 2) or regular user (user_type = 0)
                    if ($user->user_type == 2) {
                        // Provider - get category name
                        $categoryName = 'Provider'; // Default
                        if ($user->userDetails && !empty($user->userDetails->category_id)) {
                            $category = \DB::table('categories')->where('id', $user->userDetails->category_id)->first();
                            $categoryName = $category->name ?? 'Provider';
                        }
                        
                        $smsRepository->sendAdminProviderCreditsWhatsApp(
                            $adminWhatsApp,
                            $userName,
                            $categoryName,
                            $currencySymbol . ' ' . number_format($amountPaid, 2),
                            number_format($credits, 0) . ' credits'
                        );
                    } else {
                        // Regular user
                        $smsRepository->sendAdminUserCreditsWhatsApp(
                            $adminWhatsApp,
                            $userName,
                            $currencySymbol . ' ' . number_format($amountPaid, 2),
                            number_format($credits, 0) . ' credits'
                        );
                    }
                }
            } catch (\Exception $adminWhatsappEx) {
                \Log::warning('Failed to send admin credits WhatsApp: ' . $adminWhatsappEx->getMessage());
            }
            
        } catch (\Exception $e) {
            \Log::error('Failed to send wallet top-up success email: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}