<?php

namespace App\Repositories\Eloquent;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Repositories\Eloquent\WalletRepository;

class PayFastRepository
{
    private $merchantId;
    private $merchantKey;
    private $passphrase;
    private $mode;
    private $baseUrl;
    
    public function __construct()
    {
        $this->merchantId = trim(config('payfast.merchant_id'));
        $this->merchantKey = trim(config('payfast.merchant_key'));
        $this->passphrase = trim(config('payfast.passphrase'));
        $this->mode = config('payfast.mode', 'sandbox');
        $this->baseUrl = config('payfast.hosts.' . $this->mode);
        
        Log::info('PayFast initialized', [
            'mode' => $this->mode,
            'has_merchant_id' => !empty($this->merchantId),
            'has_merchant_key' => !empty($this->merchantKey),
            'has_passphrase' => !empty($this->passphrase),
            'passphrase_length' => strlen($this->passphrase)
        ]);
    }
    
    /**
     * Generate payment data for PayFast
     * Follows Stripe pattern exactly
     * NOTE: PayFast only accepts ZAR (South African Rand)
     * Conversion: 1 Credit = $1 USD = R 17.50 ZAR
     */
    public function createPayment($amount, $userId, $trxId, $type = 'wallet', $userName = '', $userEmail = '')
    {
        // Amount is already in ZAR from the frontend (from credit_packages table)
        // NO conversion needed - just format it
        $amountInZAR = $amount;
        
        // Use ngrok URL for sandbox testing, or standard route for production
        $notifyUrl = env('PAYFAST_NOTIFY_URL') 
            ? env('PAYFAST_NOTIFY_URL') 
            : route('payfast.notify');
        
        $data = [
            'merchant_id' => $this->merchantId,
            'merchant_key' => $this->merchantKey,
            'return_url' => $type === 'provider' ? route('payfast.success.provider') : route('payfast.success.user'),
            'cancel_url' => $type === 'provider' ? route('payfast.cancel.provider') : route('payfast.cancel.user'),
            'notify_url' => $notifyUrl,
            'name_first' => $userName ?: 'Customer',
            'name_last' => 'User',
            'email_address' => $userEmail ?: '',
            'm_payment_id' => $trxId,
            'amount' => number_format($amountInZAR, 2, '.', ''),
            'item_name' => 'Wallet Top-up',
            'item_description' => 'Wallet top-up for user ID: ' . $userId,
        ];
        
        // Generate signature
        $signature = $this->generateSignature($data);
        $data['signature'] = $signature;
        
        Log::info('PayFast payment created', [
            'trx_id' => $trxId,
            'amount' => $amount,
            'user_id' => $userId,
            'type' => $type,
            'merchant_id' => $this->merchantId,
            'has_passphrase' => !empty($this->passphrase),
            'signature' => $signature,
            'all_fields' => array_keys($data)
        ]);
        
        return [
            'action_url' => $this->baseUrl . '/eng/process',
            'data' => $data
        ];
    }
    
    /**
     * Generate MD5 signature using EXACT PayFast SDK method
     * This is the EXACT same method that worked in our test
     */
    public function generateSignature($data)
    {
        // Remove signature field if it exists
        unset($data['signature']);
        
        // Use EXACT PayFast SDK method (no sorting!)
        $pfOutput = '';
        foreach ($data as $key => $val) {
            if ($val !== '') {
                $pfOutput .= $key . '=' . urlencode(trim($val)) . '&';
            }
        }
        
        // Remove last ampersand
        $getString = substr($pfOutput, 0, -1);
        
        // Add passphrase if set
        if (!empty($this->passphrase)) {
            $getString .= '&passphrase=' . urlencode(trim($this->passphrase));
        }
        
        Log::info('PayFast signature generation (SDK method)', [
            'param_string' => $getString,
            'signature' => md5($getString)
        ]);
        
        return md5($getString);
    }
    
    /**
     * Validate ITN (Instant Transaction Notification)
     * Implements all 6 security checks as per PayFast documentation
     */
    public function validateITN($postData)
    {
        Log::info('PayFast ITN: Validation started', [
            'ip' => request()->ip(),
            'has_data' => !empty($postData),
            'm_payment_id' => $postData['m_payment_id'] ?? '',
            'payment_status' => $postData['payment_status'] ?? ''
        ]);
        
        // PayFast Official 4 Security Checks
        
        // Check 1: Verify signature (as per PayFast docs)
        $check1 = $this->pfValidSignature($postData);
        Log::info('PayFast ITN: Check 1 - Signature', ['passed' => $check1]);
        
        // Check 2: Verify source IP (as per PayFast docs)
        $check2 = $this->pfValidIP();
        Log::info('PayFast ITN: Check 2 - IP', ['passed' => $check2]);
        
        // Check 3: Verify payment data (amount match)
        $check3 = $this->pfValidPaymentData($postData);
        Log::info('PayFast ITN: Check 3 - Payment Data', ['passed' => $check3]);
        
        // Check 4: Server confirmation (as per PayFast docs)
        $check4 = $this->pfValidServerConfirmation($postData);
        Log::info('PayFast ITN: Check 4 - Server Confirmation', ['passed' => $check4]);
        
        $allPassed = $check1 && $check2 && $check3 && $check4;
        
        Log::info('PayFast ITN: All checks result', [
            'check1_signature' => $check1,
            'check2_ip' => $check2,
            'check3_payment' => $check3,
            'check4_server' => $check4,
            'all_passed' => $allPassed
        ]);
        
        return $allPassed;
    }
    
    /**
     * PayFast Official Check 1: Verify signature
     */
    private function pfValidSignature($pfData)
    {
        $signature = $pfData['signature'] ?? '';
        if (empty($signature)) {
            Log::warning('PayFast ITN: No signature provided');
            return false;
        }
        
        // Create parameter string (exclude signature)
        $pfParamString = '';
        foreach ($pfData as $key => $val) {
            if ($key !== 'signature' && $val !== '') {
                $pfParamString .= $key . '=' . urlencode(trim($val)) . '&';
            }
        }
        $pfParamString = substr($pfParamString, 0, -1);
        
        // Add passphrase
        $tempParamString = $pfParamString . '&passphrase=' . urlencode(trim($this->passphrase));
        
        $calculatedSignature = md5($tempParamString);
        
        $isValid = ($signature === $calculatedSignature);
        
        Log::info('PayFast ITN: Signature validation', [
            'received' => $signature,
            'calculated' => $calculatedSignature,
            'param_string' => $pfParamString,
            'with_passphrase' => $tempParamString,
            'valid' => $isValid
        ]);
        
        return $isValid;
    }
    
    /**
     * PayFast Official Check 2: Verify source IP
     */
    private function pfValidIP()
    {
        $validHosts = [
            'www.payfast.co.za',
            'sandbox.payfast.co.za',
            'w1w.payfast.co.za',
            'w2w.payfast.co.za',
        ];
        
        $validIps = [];
        foreach ($validHosts as $pfHostname) {
            $ips = gethostbynamel($pfHostname);
            if ($ips !== false) {
                $validIps = array_merge($validIps, $ips);
            }
        }
        
        // Add PayFast's new IP range from their support emails (197.97.148.0/24)
        for ($i = 0; $i <= 255; $i++) {
            $validIps[] = '197.97.148.' . $i;
        }
        
        $validIps = array_unique($validIps);
        $requestIp = request()->ip();
        
        $isValid = in_array($requestIp, $validIps);
        
        Log::info('PayFast ITN: IP validation', [
            'request_ip' => $requestIp,
            'valid' => $isValid
        ]);
        
        return $isValid;
    }
    
    /**
     * PayFast Official Check 3: Verify payment data
     */
    private function pfValidPaymentData($pfData)
    {
        // Get expected amount from database
        $transactionId = $pfData['m_payment_id'] ?? '';
        if (empty($transactionId)) {
            return false;
        }
        
        $transaction = \App\Models\WalletHistory::where('transaction_id', $transactionId)->first();
        if (!$transaction) {
            Log::error('PayFast ITN: Transaction not found', ['transaction_id' => $transactionId]);
            return false;
        }
        
        $expectedAmount = (float) $transaction->amount;
        $receivedAmount = (float) ($pfData['amount_gross'] ?? 0);
        
        $isValid = !(abs($expectedAmount - $receivedAmount) > 0.01);
        
        Log::info('PayFast ITN: Payment data validation', [
            'expected' => $expectedAmount,
            'received' => $receivedAmount,
            'difference' => abs($expectedAmount - $receivedAmount),
            'valid' => $isValid
        ]);
        
        return $isValid;
    }
    
    /**
     * PayFast Official Check 4: Server confirmation
     */
    private function pfValidServerConfirmation($pfData)
    {
        // Create parameter string for server confirmation
        $pfParamString = '';
        foreach ($pfData as $key => $val) {
            if ($key !== 'signature' && $val !== '') {
                $pfParamString .= $key . '=' . urlencode(trim($val)) . '&';
            }
        }
        $pfParamString = substr($pfParamString, 0, -1);
        
        $pfHost = $this->mode === 'sandbox' ? 'sandbox.payfast.co.za' : 'www.payfast.co.za';
        $url = 'https://' . $pfHost . '/eng/query/validate';
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_USERAGENT, NULL);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $pfParamString);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $isValid = ($response === 'VALID');
        
        Log::info('PayFast ITN: Server confirmation', [
            'url' => $url,
            'param_string' => $pfParamString,
            'response' => $response,
            'valid' => $isValid
        ]);
        
        return $isValid;
    }
    
    /**
     * Verify source IP address
     * Security measure as per PayFast documentation
     */
    private function verifySourceIP()
    {
        $validHosts = config('payfast.valid_hosts');
        
        $validIps = [];
        foreach ($validHosts as $host) {
            $ips = gethostbynamel($host);
            if ($ips !== false) {
                $validIps = array_merge($validIps, $ips);
            }
        }
        
        $requestIp = request()->ip();
        
        // For sandbox testing, also allow localhost
        if ($this->mode === 'sandbox' && in_array($requestIp, ['127.0.0.1', '::1'])) {
            return true;
        }
        
        return in_array($requestIp, $validIps);
    }
    
    /**
     * Server confirmation with PayFast
     * Final validation step as per PayFast documentation
     */
    private function serverConfirmation($postData)
    {
        try {
            $response = Http::asForm()->post($this->baseUrl . '/eng/query/validate', $postData);
            
            $result = $response->body() === 'VALID';
            
            Log::info('PayFast server confirmation', [
                'response' => $response->body(),
                'valid' => $result
            ]);
            
            return $result;
        } catch (\Exception $e) {
            Log::error('PayFast server confirmation error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Process ITN and update transaction
     * This is where credits are allocated - following Stripe pattern exactly
     */
    public function processITN($postData)
    {
        try {
            // Validate ITN first
            if (!$this->validateITN($postData)) {
                return false;
            }
            
            $paymentStatus = $postData['payment_status'] ?? '';
            $trxId = $postData['m_payment_id'] ?? '';
            $amount = $postData['amount_gross'] ?? 0;
            
            Log::info('PayFast ITN processing', [
                'payment_status' => $paymentStatus,
                'trx_id' => $trxId,
                'amount' => $amount
            ]);
            
            // Map PayFast status to our status
            $status = 'pending';
            if ($paymentStatus === 'COMPLETE') {
                $status = 'Completed';
            } elseif ($paymentStatus === 'FAILED') {
                $status = 'Failed';
            } elseif ($paymentStatus === 'CANCELLED') {
                $status = 'Cancelled';
            }
            
            // Use existing WalletRepository to confirm transaction
            // This is the EXACT SAME method that Stripe uses to allocate credits
            $walletRepo = new WalletRepository();
            $result = $walletRepo->confirmTransaction($trxId, $status);
            
            if ($result) {
                Log::info('PayFast transaction confirmed - Credits allocated', [
                    'trx_id' => $trxId,
                    'status' => $status,
                    'wallet_history_id' => $result
                ]);
            } else {
                Log::error('PayFast transaction confirmation failed', [
                    'trx_id' => $trxId,
                    'status' => $status
                ]);
            }
            
            return $result;
            
        } catch (\Exception $e) {
            Log::error('PayFast ITN processing error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }
}

