<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ParadigmQuestion extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'question_text',
        'option_a_label',
        'option_b_label',
        'option_c_label',
        'option_d_label',
        'option_e_label',
        'assessment_module',
        'question_type',
        'rating_scale_min',
        'rating_scale_max',
        'is_required',
        'display_order',
        'question_settings'
    ];

    protected $casts = [
        'is_required' => 'boolean',
        'rating_scale_min' => 'integer',
        'rating_scale_max' => 'integer',
        'question_settings' => 'array'
    ];

    /**
     * Get the template that owns this question
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Scope for questions by assessment module
     */
    public function scopeByModule($query, string $module)
    {
        return $query->where('assessment_module', $module);
    }

    /**
     * Scope for questions by type
     */
    public function scopeByType($query, string $type)
    {
        return $query->where('question_type', $type);
    }

    /**
     * Scope for required questions
     */
    public function scopeRequired($query)
    {
        return $query->where('is_required', true);
    }

    /**
     * Get the valid question types for Paradigm Profile
     */
    public static function getValidQuestionTypes(): array
    {
        return [
            'rating_scale',      // 1-4 scale for Balance Profile
            'multiple_choice',   // A/B choices for Conflict/Connection Styles
            'ranking',          // 1st, 2nd, 3rd, 4th preferences for Brain Profile
            'binary_choice'     // a/b choices for Personality Blueprint
        ];
    }

    /**
     * Get the valid assessment modules
     */
    public static function getValidAssessmentModules(): array
    {
        return [
            'brain_profile_4frame',     // 4 Frame Brain Profile (LU/LL/RU/RL)
            'brain_profile_8field',     // 8 Field Brain Profile (detailed)
            'conflict_styles',          // 5 Conflict Styles
            'connection_styles',        // 5 Connection Styles  
            'personality_blueprint',    // E/I Personality Blueprint
            'social_intelligence',      // Social Intelligence (SI)
            'emotional_intelligence',   // Emotional Intelligence (EQ)
            'relevancy_quotient',       // Relevancy Quotient (RQ)
            'balance_profile',          // Legacy support
            'serq',                     // Legacy support
            'brain_frames',             // Legacy support
            'brain_fields',             // Legacy support
            'type_classification'       // Legacy support
        ];
    }

    /**
     * Validate that question type is valid
     */
    public function validateQuestionType(): bool
    {
        return in_array($this->question_type, self::getValidQuestionTypes());
    }

    /**
     * Validate that assessment module is valid
     */
    public function validateAssessmentModule(): bool
    {
        return in_array($this->assessment_module, self::getValidAssessmentModules());
    }

    /**
     * Get question options as array based on question type
     */
    public function getOptions(): array
    {
        switch ($this->question_type) {
            case 'rating_scale':
                return [
                    '1' => 'Rarely',
                    '2' => 'Sometimes', 
                    '3' => 'Most of the time',
                    '4' => 'Always'
                ];
                
            case 'multiple_choice':
                $options = [];
                if ($this->option_a_label) $options['A'] = $this->option_a_label;
                if ($this->option_b_label) $options['B'] = $this->option_b_label;
                if ($this->option_c_label) $options['C'] = $this->option_c_label;
                if ($this->option_d_label) $options['D'] = $this->option_d_label;
                if ($this->option_e_label) $options['E'] = $this->option_e_label;
                return $options;
                
            case 'binary_choice':
                return [
                    'a' => $this->option_a_label,
                    'b' => $this->option_b_label
                ];
                
            case 'ranking':
                return [
                    '1st' => '1st Preference',
                    '2nd' => '2nd Preference',
                    '3rd' => '3rd Preference',
                    '4th' => '4th Preference'
                ];
                
            default:
                return [];
        }
    }

    /**
     * Get options based on assessment module
     */
    public function getModuleOptions(): array
    {
        switch ($this->assessment_module) {
            case 'brain_profile_4frame':
                // 4 Frame Brain Profile - ranking questions
                return [
                    '1st' => '1st Preference',
                    '2nd' => '2nd Preference', 
                    '3rd' => '3rd Preference',
                    '4th' => '4th Preference'
                ];
                
            case 'conflict_styles':
                // Conflict Styles - A/B choices
                return [
                    'A' => $this->option_a_label,
                    'B' => $this->option_b_label
                ];
                
            case 'connection_styles':
                // Connection Styles - A/B/C/D/E choices
                return [
                    'A' => $this->option_a_label,
                    'B' => $this->option_b_label,
                    'C' => $this->option_c_label,
                    'D' => $this->option_d_label,
                    'E' => $this->option_e_label
                ];
                
            case 'personality_blueprint':
                // Personality Blueprint - A/B choices
                return [
                    'a' => $this->option_a_label,
                    'b' => $this->option_b_label
                ];
                
            case 'social_intelligence':
            case 'emotional_intelligence':
            case 'relevancy_quotient':
                // SI/EQ/RQ - A/B/C choices (3/2/1 points)
                return [
                    'A' => $this->option_a_label,
                    'B' => $this->option_b_label,
                    'C' => $this->option_c_label
                ];
                
            default:
                return $this->getOptions();
        }
    }

    /**
     * Get rating scale range
     */
    public function getRatingScaleRange(): array
    {
        if ($this->question_type !== 'rating_scale') {
            return [];
        }
        
        $min = $this->rating_scale_min ?? 1;
        $max = $this->rating_scale_max ?? 4;
        
        return range($min, $max);
    }

    /**
     * Check if question has multiple options
     */
    public function hasMultipleOptions(): bool
    {
        return in_array($this->question_type, ['multiple_choice', 'ranking']);
    }

    /**
     * Get question settings
     */
    public function getQuestionSettings(): array
    {
        return $this->question_settings ?? [];
    }
}
