<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ParadigmPdfPage extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'page_number',
        'page_title',
        'background_image_path',
        'is_content_page',
        'page_settings',
        'display_order'
    ];

    protected $casts = [
        'is_content_page' => 'boolean',
        'page_settings' => 'array'
    ];

    /**
     * Get the template that owns this PDF page
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Get the PDF elements for this page
     */
    public function pdfElements(): HasMany
    {
        return $this->hasMany(ParadigmPdfElement::class, 'page_id')->orderBy('display_order');
    }

    /**
     * Scope for content pages (with elements)
     */
    public function scopeContentPages($query)
    {
        return $query->where('is_content_page', true);
    }

    /**
     * Scope for image-only pages
     */
    public function scopeImageOnlyPages($query)
    {
        return $query->where('is_content_page', false);
    }

    /**
     * Scope for specific page number
     */
    public function scopeByPageNumber($query, int $pageNumber)
    {
        return $query->where('page_number', $pageNumber);
    }

    /**
     * Check if page has background image
     */
    public function hasBackgroundImage(): bool
    {
        return !empty($this->background_image_path);
    }

    /**
     * Get background image URL
     */
    public function getBackgroundImageUrl(): ?string
    {
        return $this->hasBackgroundImage() ? asset('storage/' . $this->background_image_path) : null;
    }

    /**
     * Get page settings as array
     */
    public function getPageSettings(): array
    {
        return $this->page_settings ?? [];
    }
}
