<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Str;

class ParadigmPayment extends Model
{
    use HasFactory;

    protected $table = 'paradigm_payments';

    protected $fillable = [
        'template_id',
        'provider_id',
        'client_id',
        'client_email',
        'client_name',
        'client_surname',
        'client_gender',
        'client_dob',
        'client_whatsapp',
        'amount',
        'currency',
        'status',
        'paid_by',
        'wallet_transaction_id',
        'payment_link',
        'payment_reference',
        'payment_method',
        'payment_data',
        'expires_at',
        'paid_at',
        'self_purchased', // NEW: For user self-purchase feature
        'progress_data'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'payment_data' => 'array',
        'progress_data' => 'array',
        'expires_at' => 'datetime',
        'paid_at' => 'datetime',
        'client_dob' => 'date'
    ];

    /**
     * Get the template that owns the payment
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Get the provider who generated the payment link
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(User::class, 'provider_id');
    }

    /**
     * Get the client who made the payment
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(User::class, 'client_id');
    }

    /**
     * Get the access token for this payment
     */
    public function accessToken(): HasOne
    {
        return $this->hasOne(ParadigmAccessToken::class, 'payment_id');
    }

    /**
     * Get the wallet transaction for this payment (if paid from wallet)
     */
    public function walletTransaction(): BelongsTo
    {
        return $this->belongsTo(WalletHistory::class, 'wallet_transaction_id');
    }

    /**
     * Check if this payment is free
     */
    public function isFree(): bool
    {
        return $this->amount == 0.00 || $this->status === 'free';
    }

    /**
     * Check if payment is completed
     */
    public function isCompleted(): bool
    {
        return in_array($this->status, ['paid', 'free']);
    }

    /**
     * Generate a unique payment link
     */
    public static function generatePaymentLink(): string
    {
        do {
            $link = 'paradigm-' . Str::random(32);
        } while (self::where('payment_link', $link)->exists());
        
        return $link;
    }

    /**
     * Create a free payment (bypass payment system)
     */
    public static function createFreePayment(int $templateId, int $providerId, ?int $clientId = null): self
    {
        return self::create([
            'template_id' => $templateId,
            'provider_id' => $providerId,
            'client_id' => $clientId,
            'amount' => 0.00,
            'currency' => 'USD',
            'status' => 'free',
            'payment_link' => self::generatePaymentLink(),
            'paid_at' => now()
        ]);
    }
    
    /**
     * Get access tokens relationship
     */
    public function accessTokens()
    {
        return $this->hasMany(ParadigmAccessToken::class, 'payment_id');
    }
}