<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class ParadigmFormTemplate extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'shortcode_id',
        'description',
        'is_active',
        'settings',
        'type_groups'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'settings' => 'array',
        'type_groups' => 'array'
    ];

    /**
     * Get the personal fields for this template
     */
    public function personalFields(): HasMany
    {
        return $this->hasMany(ParadigmPersonalField::class, 'template_id')->orderBy('display_order');
    }

    /**
     * Get the questions for this template
     */
    public function questions(): HasMany
    {
        return $this->hasMany(ParadigmQuestion::class, 'template_id')->orderBy('display_order');
    }

    /**
     * Get the badge titles for this template
     */
    public function badgeTitles(): HasMany
    {
        return $this->hasMany(ParadigmBadgeTitle::class, 'template_id');
    }

    /**
     * Get the pricing for this template
     */
    public function pricing(): HasOne
    {
        return $this->hasOne(ParadigmPricing::class, 'template_id')->where('is_active', true);
    }

    /**
     * Get the PDF pages for this template
     */
    public function pdfPages(): HasMany
    {
        return $this->hasMany(\App\Models\ParadigmPdfPage::class, 'template_id')->orderBy('page_number');
    }

    /**
     * Get the PDF elements for this template
     */
    public function pdfElements(): HasMany
    {
        return $this->hasMany(\App\Models\ParadigmPdfElement::class, 'template_id');
    }

    /**
     * Get the assessment results for this template
     */
    public function assessmentResults(): HasMany
    {
        return $this->hasMany(ParadigmAssessmentResult::class, 'template_id');
    }

    /**
     * Get questions by assessment module
     */
    public function getQuestionsByModule(string $module): HasMany
    {
        return $this->questions()->where('assessment_module', $module);
    }

    /**
     * Get badge title by type code
     */
    public function getBadgeTitleByType(string $typeCode): ?ParadigmBadgeTitle
    {
        return $this->badgeTitles()->where('type_code', $typeCode)->first();
    }

    /**
     * Scope for active templates
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get shortcode for this template
     */
    public function getShortcode(): string
    {
        return "[paradigm_form id=\"{$this->id}\"]";
    }
    
    /**
     * Get the type groups configuration for Paradigm Profile
     */
    public function getTypeGroups(): array
    {
        return $this->type_groups ?? [
            'balance_profile' => [
                'name' => 'Balance Profile',
                'description' => '8 dimensions of wellness and balance',
                'elements' => [
                    'emotional_balance_percentage',
                    'spiritual_balance_percentage', 
                    'physical_balance_percentage',
                    'social_balance_percentage',
                    'financial_balance_percentage',
                    'occupational_balance_percentage',
                    'intellectual_balance_percentage',
                    'environmental_balance_percentage',
                    'overall_wellness_score'
                ]
            ],
            'conflict_styles' => [
                'name' => 'Conflict Styles',
                'description' => '5 approaches to handling conflict',
                'elements' => [
                    'conflict_style_assertive_challenger_score',
                    'conflict_style_integrative_bridge_builder_score',
                    'conflict_style_fair_negotiator_score',
                    'conflict_style_quiet_protector_score',
                    'conflict_style_peacekeeper_score'
                ]
            ],
            'connection_styles' => [
                'name' => 'Connection Styles',
                'description' => '5 ways of connecting with others',
                'elements' => [
                    'connection_style_verbal_validator_score',
                    'connection_style_presence_seeker_score',
                    'connection_style_symbolic_appreciator_score',
                    'connection_style_practical_supporter_score',
                    'connection_style_grounded_connector_score'
                ]
            ],
            'personality_blueprint' => [
                'name' => 'Personality Blueprint',
                'description' => 'Extroversion vs Introversion preferences',
                'elements' => [
                    'extroversion_percentage',
                    'introversion_percentage'
                ]
            ],
            'serq' => [
                'name' => 'SERQ Assessment',
                'description' => 'Social, Emotional, and Relevancy Quotients',
                'elements' => [
                    'sq_level',
                    'eq_level',
                    'rq_level'
                ]
            ],
            'brain_frames' => [
                'name' => 'Brain Profile - FRAMES',
                'description' => '4 brain quadrants and thinking preferences',
                'elements' => [
                    'lu_percentage',
                    'll_percentage',
                    'ru_percentage',
                    'rl_percentage'
                ]
            ],
            'brain_fields' => [
                'name' => 'Brain Profile - FIELDS',
                'description' => '8 specific thinking and behavioral fields',
                'elements' => [
                    'fact_finder_percentage',
                    'question_asker_percentage',
                    'rule_keeper_percentage',
                    'task_manager_percentage',
                    'creator_percentage',
                    'big_thinker_percentage',
                    'empath_percentage',
                    'friend_maker_percentage'
                ]
            ],
            'type_classification' => [
                'name' => 'Paradigm Type Classification',
                'description' => '15 distinct Paradigm personality types',
                'elements' => [
                    'paradigm_type',
                    'dominance_category_short',
                    'lu_preference_level',
                    'll_preference_level',
                    'ru_preference_level',
                    'rl_preference_level'
                ]
            ]
        ];
    }

    /**
     * Get shortcode with custom attributes
     */
    public function getShortcodeWithAttributes(array $attributes = []): string
    {
        $shortcode = "[paradigm_form id=\"{$this->id}\"";
        
        foreach ($attributes as $key => $value) {
            $shortcode .= " {$key}=\"{$value}\"";
        }
        
        $shortcode .= "]";
        
        return $shortcode;
    }

    /**
     * Get template by shortcode ID
     */
    public static function findByShortcodeId(int $id): ?self
    {
        return self::active()->where('id', $id)->first();
    }
}
