<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ParadigmAssessmentResult extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'provider_id',
        'session_id',
        'name',
        'surname',
        'gender',
        'age_range',
        'dob',
        'phone',
        'email',
        'scores_json',
        'paradigm_type',
        'dominance_category_short',
        'answers_json',
        'personal_data',
        'completed_at',
        'is_available_in_marketplace',
        'marketplace_listed_at',
        'marketplace_purchase_count',
        // All 74 Paradigm Profile elements
        'lu_percentage', 'll_percentage', 'ru_percentage', 'rl_percentage',
        'fact_finder_percentage', 'question_asker_percentage', 'rule_keeper_percentage', 'task_manager_percentage',
        'creator_percentage', 'big_thinker_percentage', 'empath_percentage', 'friend_maker_percentage',
        'extroversion_percentage', 'introversion_percentage',
        'emotional_balance_percentage', 'spiritual_balance_percentage', 'physical_balance_percentage',
        'social_balance_percentage', 'financial_balance_percentage', 'occupational_balance_percentage',
        'intellectual_balance_percentage', 'environmental_balance_percentage', 'overall_wellness_score',
        'connection_style_verbal_validator_score', 'connection_style_presence_seeker_score', 'connection_style_symbolic_appreciator_score', 'connection_style_practical_supporter_score', 'connection_style_grounded_connector_score',
        'conflict_style_assertive_challenger_score', 'conflict_style_integrative_bridge_builder_score', 'conflict_style_fair_negotiator_score', 'conflict_style_quiet_protector_score', 'conflict_style_peacekeeper_score',
        'connection_style_verbal_validator_rank', 'connection_style_presence_seeker_rank', 'connection_style_symbolic_appreciator_rank', 'connection_style_practical_supporter_rank', 'connection_style_grounded_connector_rank',
        'conflict_style_assertive_challenger_rank', 'conflict_style_integrative_bridge_builder_rank', 'conflict_style_fair_negotiator_rank', 'conflict_style_quiet_protector_rank', 'conflict_style_peacekeeper_rank',
        'user_full_name', 'report_date', 'user_gender', 'user_age_range',
        'practitioner_name', 'practitioner_email', 'practitioner_phone', 'practitioner_url',
        'lu_preference_level', 'll_preference_level', 'ru_preference_level', 'rl_preference_level',
        'connection_style_rank_1_name', 'connection_style_rank_2_name', 'connection_style_rank_3_name', 'connection_style_rank_4_name', 'connection_style_rank_5_name',
        'conflict_style_rank_1_name', 'conflict_style_rank_2_name', 'conflict_style_rank_3_name', 'conflict_style_rank_4_name', 'conflict_style_rank_5_name',
        'social_intelligence_level', 'emotional_intelligence_level', 'relevancy_quotient_level',
        'social_intelligence_score', 'emotional_intelligence_score', 'relevancy_quotient_score',
        'is_single_frame', 'is_dual_frame', 'is_triple_frame', 'is_all_frame'
    ];

    protected $casts = [
        'scores_json' => 'array',
        'answers_json' => 'array',
        'personal_data' => 'array',
        'completed_at' => 'datetime',
        'marketplace_listed_at' => 'datetime',
        'is_available_in_marketplace' => 'boolean'
    ];

    /**
     * Get the template that owns this assessment result
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Get the provider (user) who sent this assessment
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(User::class, 'provider_id');
    }

    /**
     * Scope for completed assessments
     */
    public function scopeCompleted($query)
    {
        return $query->whereNotNull('completed_at');
    }

    /**
     * Scope for specific paradigm type
     */
    public function scopeByType($query, string $paradigmType)
    {
        return $query->where('paradigm_type', $paradigmType);
    }

    /**
     * Scope for specific dominance category
     */
    public function scopeByDominanceCategory($query, string $category)
    {
        return $query->where('dominance_category_short', $category);
    }

    /**
     * Scope for specific session
     */
    public function scopeBySession($query, string $sessionId)
    {
        return $query->where('session_id', $sessionId);
    }

    /**
     * Get full name
     */
    public function getFullNameAttribute(): string
    {
        return trim($this->name . ' ' . $this->surname);
    }

    /**
     * Get scores as a flattened key-value array for PDF generation.
     */
    public function getFlattenedData(): array
    {
        $scores = $this->getScores();
        $flatData = [];

        // Part 1: Brain Frames (LU, LL, RU, RL)
        $part1_percentages = $scores['paradigm_main_part1']['percentages'] ?? [];
        foreach ($part1_percentages as $key => $value) {
            $flatData[strtolower($key) . '_percentage'] = round($value, 2);
        }

        // Part 2: Brain Fields (8 fields)
        $part2_percentages = $scores['paradigm_main_part2']['percentages'] ?? [];
        foreach ($part2_percentages as $key => $value) {
            // "LU – The Question Asker" -> "question_asker_percentage"
            $parts = explode('– The ', $key);
            if (isset($parts[1])) {
                $fieldKey = strtolower(str_replace(' ', '_', trim($parts[1])));
                $flatData[$fieldKey . '_percentage'] = round($value, 2);
            }
        }
        
        // Text Elements
        $flatData['paradigm_type'] = $this->paradigm_type ?? 'Unknown';
        $flatData['dominance_category'] = $scores['paradigm_main_part1']['dominance'] ?? 'Unknown';

        // Personal Details
        $personal_data = $this->personal_data ?? [];
        $flatData['name'] = $this->name ?? $personal_data['name'] ?? 'N/A';
        $flatData['surname'] = $this->surname ?? $personal_data['surname'] ?? 'N/A';
        $flatData['email'] = $this->email ?? $personal_data['email'] ?? 'N/A';
        $flatData['age_range'] = $this->age_range ?? $personal_data['age'] ?? 'N/A';
        $flatData['gender'] = $this->gender ?? $personal_data['gender'] ?? 'N/A';
        $flatData['phone'] = $this->phone ?? $personal_data['phone'] ?? 'N/A';
        $flatData['date_completed'] = $this->completed_at ? $this->completed_at->format('Y-m-d H:i') : date('Y-m-d H:i');

        return $flatData;
    }

    /**
     * Get scores as array
     */
    public function getScores(): array
    {
        if (is_string($this->scores_json)) {
            return json_decode($this->scores_json, true) ?? [];
        }
        return $this->scores_json ?? [];
    }

    /**
     * Get specific score
     */
    public function getScore(string $key): mixed
    {
        // Handle Paradigm PDF builder element keys (paradigm_score_*, etc.)
        if (str_starts_with($key, 'paradigm_score_')) {
            $elementKey = str_replace('paradigm_score_', '', $key);
            return $this->getScores()[$elementKey] ?? 0;
        }
        
        return $this->getScores()[$key] ?? 0;
    }

    /**
     * Get all Balance Profile scores
     */
    public function getBalanceProfileScores(): array
    {
        $scores = $this->getScores();
        return [
            'emotional_balance_percentage' => $scores['emotional_balance_percentage'] ?? 0,
            'spiritual_balance_percentage' => $scores['spiritual_balance_percentage'] ?? 0,
            'physical_balance_percentage' => $scores['physical_balance_percentage'] ?? 0,
            'social_balance_percentage' => $scores['social_balance_percentage'] ?? 0,
            'financial_balance_percentage' => $scores['financial_balance_percentage'] ?? 0,
            'occupational_balance_percentage' => $scores['occupational_balance_percentage'] ?? 0,
            'intellectual_balance_percentage' => $scores['intellectual_balance_percentage'] ?? 0,
            'environmental_balance_percentage' => $scores['environmental_balance_percentage'] ?? 0,
            'overall_wellness_score' => $scores['overall_wellness_score'] ?? 0,
        ];
    }

    /**
     * Get all Conflict Styles scores
     */
    public function getConflictStylesScores(): array
    {
        $scores = $this->getScores();
        return [
            'conflict_style_assertive_challenger_score' => $scores['conflict_style_assertive_challenger_score'] ?? 0,
            'conflict_style_integrative_bridge_builder_score' => $scores['conflict_style_integrative_bridge_builder_score'] ?? 0,
            'conflict_style_fair_negotiator_score' => $scores['conflict_style_fair_negotiator_score'] ?? 0,
            'conflict_style_quiet_protector_score' => $scores['conflict_style_quiet_protector_score'] ?? 0,
            'conflict_style_peacekeeper_score' => $scores['conflict_style_peacekeeper_score'] ?? 0,
        ];
    }

    /**
     * Get all Connection Styles scores
     */
    public function getConnectionStylesScores(): array
    {
        $scores = $this->getScores();
        return [
            'connection_style_verbal_validator_score' => $scores['connection_style_verbal_validator_score'] ?? 0,
            'connection_style_presence_seeker_score' => $scores['connection_style_presence_seeker_score'] ?? 0,
            'connection_style_symbolic_appreciator_score' => $scores['connection_style_symbolic_appreciator_score'] ?? 0,
            'connection_style_practical_supporter_score' => $scores['connection_style_practical_supporter_score'] ?? 0,
            'connection_style_grounded_connector_score' => $scores['connection_style_grounded_connector_score'] ?? 0,
        ];
    }

    /**
     * Get Personality Blueprint scores
     */
    public function getPersonalityBlueprintScores(): array
    {
        $scores = $this->getScores();
        return [
            'extroversion_percentage' => $scores['extroversion_percentage'] ?? 0,
            'introversion_percentage' => $scores['introversion_percentage'] ?? 0,
        ];
    }

    /**
     * Get SERQ levels
     */
    public function getSerqLevels(): array
    {
        $scores = $this->getScores();
        return [
            'sq_level' => $scores['sq_level'] ?? 'Low',
            'eq_level' => $scores['eq_level'] ?? 'Low',
            'rq_level' => $scores['rq_level'] ?? 'Low',
        ];
    }

    /**
     * Get Brain Profile - FRAMES scores
     */
    public function getBrainFramesScores(): array
    {
        $scores = $this->getScores();
        return [
            'lu_percentage' => $scores['lu_percentage'] ?? 0,
            'll_percentage' => $scores['ll_percentage'] ?? 0,
            'ru_percentage' => $scores['ru_percentage'] ?? 0,
            'rl_percentage' => $scores['rl_percentage'] ?? 0,
        ];
    }

    /**
     * Get Brain Profile - FIELDS scores
     */
    public function getBrainFieldsScores(): array
    {
        $scores = $this->getScores();
        return [
            'fact_finder_percentage' => $scores['fact_finder_percentage'] ?? 0,
            'question_asker_percentage' => $scores['question_asker_percentage'] ?? 0,
            'rule_keeper_percentage' => $scores['rule_keeper_percentage'] ?? 0,
            'task_manager_percentage' => $scores['task_manager_percentage'] ?? 0,
            'creator_percentage' => $scores['creator_percentage'] ?? 0,
            'big_thinker_percentage' => $scores['big_thinker_percentage'] ?? 0,
            'empath_percentage' => $scores['empath_percentage'] ?? 0,
            'friend_maker_percentage' => $scores['friend_maker_percentage'] ?? 0,
        ];
    }

    /**
     * Get answers as array
     */
    public function getAnswers(): array
    {
        return $this->answers_json ?? [];
    }

    /**
     * Get specific answer
     */
    public function getAnswer(int $questionId): ?string
    {
        return $this->getAnswers()[$questionId] ?? null;
    }

    /**
     * Check if assessment is completed
     */
    public function isCompleted(): bool
    {
        return !is_null($this->completed_at);
    }

    /**
     * Get completion date
     */
    public function getCompletionDate(): ?string
    {
        return $this->completed_at?->format('Y-m-d H:i:s');
    }

    /**
     * Get personal data as array
     */
    public function getPersonalData(): array
    {
        // First try to get from individual columns (for regular assessments)
        if ($this->name || $this->surname || $this->email) {
            return [
                'name' => $this->name,
                'surname' => $this->surname,
                'gender' => $this->gender,
                'age_range' => $this->age_range,
                'phone' => $this->phone,
                'email' => $this->email,
            ];
        }
        
        // Fallback to personal_data JSON field (for previews)
        $personalData = json_decode($this->personal_data ?? '{}', true);
        return $personalData ?: [
            'name' => 'John',
            'surname' => 'Doe',
            'gender' => 'Male',
            'age_range' => '25-34',
            'phone' => '+1 (555) 123-4567',
            'email' => 'john.doe@example.com',
        ];
    }

    /**
     * Get Paradigm type information
     */
    public function getParadigmTypeInfo(): array
    {
        return [
            'paradigm_type' => $this->paradigm_type ?? 'The Whole-Brain Thinker',
            'dominance_category_short' => $this->dominance_category_short ?? 'All Frame',
        ];
    }

    /**
     * Get marketplace purchases for this result
     */
    public function marketplacePurchases(): HasMany
    {
        return $this->hasMany(ParadigmMarketplacePurchase::class, 'result_id');
    }

    /**
     * Check if result is available in marketplace
     */
    public function isAvailableInMarketplace(): bool
    {
        return $this->is_available_in_marketplace && $this->isCompleted();
    }

    /**
     * Get marketplace price based on pricing rules
     */
    public function getMarketplacePrice(): float
    {
        $paradigmType = $this->paradigm_type;
        $scores = $this->getScores();
        
        $rule = ParadigmMarketplacePricingRule::where('is_active', true)
            ->where(function($query) use ($paradigmType, $scores) {
                $query->where(function($q) use ($paradigmType) {
                    $q->whereNull('paradigm_type')
                      ->orWhere('paradigm_type', $paradigmType);
                });
                
                if ($scores && isset($scores['total_score'])) {
                    $totalScore = $scores['total_score'];
                    $query->where(function($q) use ($totalScore) {
                        $q->whereNull('min_score')
                          ->orWhere('min_score', '<=', $totalScore);
                    })
                    ->where(function($q) use ($totalScore) {
                        $q->whereNull('max_score')
                          ->orWhere('max_score', '>=', $totalScore);
                    });
                }
            })
            ->orderBy('sort_order')
            ->first();
        
        return $rule ? (float) $rule->price_credits : 0.0;
    }
}
