<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class ParadigmAccessToken extends Model
{
    use HasFactory;

    protected $table = 'paradigm_access_tokens';

    protected $fillable = [
        'token',
        'template_id',
        'provider_id',
        'client_id',
        'payment_id',
        'status',
        'used_at',
        'expires_at'
    ];

    protected $casts = [
        'used_at' => 'datetime',
        'expires_at' => 'datetime'
    ];

    /**
     * Get the template that owns the access token
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(ParadigmFormTemplate::class, 'template_id');
    }

    /**
     * Get the provider who owns the access token
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(User::class, 'provider_id');
    }

    /**
     * Get the client who owns the access token
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(User::class, 'client_id');
    }

    /**
     * Get the payment that owns the access token
     */
    public function payment(): BelongsTo
    {
        return $this->belongsTo(ParadigmPayment::class, 'payment_id');
    }

    /**
     * Check if token is active and not expired
     */
    public function isActive(): bool
    {
        return $this->status === 'active' && 
               (!$this->expires_at || $this->expires_at->isFuture());
    }

    /**
     * Check if token has been used
     */
    public function isUsed(): bool
    {
        return $this->status === 'used' || $this->used_at !== null;
    }

    /**
     * Generate a unique access token
     */
    public static function generateToken(): string
    {
        do {
            $token = Str::random(64);
        } while (self::where('token', $token)->exists());
        
        return $token;
    }

    /**
     * Create access token for a payment
     */
    public static function createForPayment(ParadigmPayment $payment, int $expirationHours = 24): self
    {
        return self::create([
            'token' => self::generateToken(),
            'template_id' => $payment->template_id,
            'provider_id' => $payment->provider_id,
            'client_id' => $payment->client_id,
            'payment_id' => $payment->id,
            'status' => 'active',
            'expires_at' => now()->addHours($expirationHours)
        ]);
    }
}