<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\ParadigmLinkRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ParadigmLinkRequestController extends Controller
{
    /**
     * List link requests for user
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        if (!$user) {
            abort(403, 'Unauthorized');
        }
        
        // Start with basic query
        $query = ParadigmLinkRequest::where('client_id', $user->id);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Get requests without eager loading first to test
        $requests = $query->orderBy('created_at', 'desc')->paginate(20);
        
        // Eager load relationships after pagination
        $requests->load(['result.template', 'practitioner.userDetail']);

        return view('user.paradigm.link-requests.index', compact('requests'));
    }

    /**
     * Accept link request
     */
    public function accept($requestId)
    {
        $user = Auth::user();
        $linkRequest = ParadigmLinkRequest::findOrFail($requestId);

        if ($linkRequest->client_id != $user->id) {
            abort(403, 'Unauthorized');
        }

        if ($linkRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'This request has already been responded to.');
        }

        $linkRequest->accept();

        return redirect()->back()->with('success', 'Link request accepted successfully.');
    }

    /**
     * Decline link request
     */
    public function decline($requestId)
    {
        $user = Auth::user();
        $linkRequest = ParadigmLinkRequest::findOrFail($requestId);

        if ($linkRequest->client_id != $user->id) {
            abort(403, 'Unauthorized');
        }

        if ($linkRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'This request has already been responded to.');
        }

        $linkRequest->decline();

        return redirect()->back()->with('success', 'Link request declined.');
    }
}

