<?php

namespace App\Http\Controllers;

use App\Models\StaffDocument;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;

class StaffDocumentController extends Controller
{
    /**
     * Upload staff ID document
     */
    public function uploadIdDocument(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'staff_id' => 'required|exists:users,id',
            'file' => 'required|file|mimes:jpg,jpeg,png,gif,pdf|max:5120', // 5MB
        ], [
            'file.required' => 'ID document is required.',
            'file.mimes' => 'ID document must be JPG, PNG, GIF, or PDF.',
            'file.max' => 'File must not exceed 5MB.'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        try {
            $staffId = $request->staff_id;
            $file = $request->file('file');
            
            // Delete old ID document if exists
            StaffDocument::where('staff_id', $staffId)
                ->where('document_type', 'id_document')
                ->each(function($doc) {
                    if (Storage::disk('public')->exists($doc->file_path)) {
                        Storage::disk('public')->delete($doc->file_path);
                    }
                    $doc->delete();
                });
            
            // Store the file
            $filename = Str::uuid() . '_' . time() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('staff-documents', $filename, 'public');
            
            // Create document record
            $document = StaffDocument::create([
                'staff_id' => $staffId,
                'document_type' => 'id_document',
                'file_name' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_size' => $file->getSize(),
                'uploaded_by' => Auth::id(),
            ]);
            
            // Update staff verification status to pending
            User::where('id', $staffId)->update(['staff_verification_status' => 'pending']);
            
            return response()->json([
                'success' => true,
                'message' => 'ID document uploaded successfully',
                'data' => [
                    'filename' => $filename,
                    'url' => asset('storage/' . $path)
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload document: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Upload staff selfie
     */
    public function uploadSelfie(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'staff_id' => 'required|exists:users,id',
            'file' => 'required|image|mimes:jpg,jpeg,png,gif|max:5120', // 5MB
        ], [
            'file.required' => 'Selfie photo is required.',
            'file.image' => 'File must be an image.',
            'file.mimes' => 'Selfie must be JPG, PNG, or GIF.',
            'file.max' => 'Image must not exceed 5MB.'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        try {
            $staffId = $request->staff_id;
            $file = $request->file('file');
            
            // Delete old selfie if exists
            StaffDocument::where('staff_id', $staffId)
                ->where('document_type', 'selfie')
                ->each(function($doc) {
                    if (Storage::disk('public')->exists($doc->file_path)) {
                        Storage::disk('public')->delete($doc->file_path);
                    }
                    $doc->delete();
                });
            
            // Store the file
            $filename = Str::uuid() . '_' . time() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('staff-documents', $filename, 'public');
            
            // Create document record
            $document = StaffDocument::create([
                'staff_id' => $staffId,
                'document_type' => 'selfie',
                'file_name' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_size' => $file->getSize(),
                'uploaded_by' => Auth::id(),
            ]);
            
            // Update staff verification status to pending
            User::where('id', $staffId)->update(['staff_verification_status' => 'pending']);
            
            return response()->json([
                'success' => true,
                'message' => 'Selfie uploaded successfully',
                'data' => [
                    'filename' => $filename,
                    'url' => asset('storage/' . $path)
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload selfie: ' . $e->getMessage()
            ], 500);
        }
    }
}

