<?php

namespace App\Http\Controllers\Public;

use App\Http\Controllers\Controller;
use App\Models\ParadigmFormTemplate;
use App\Models\ParadigmAssessmentResult;
use App\Services\ParadigmScoringService;
use App\Services\ParadigmPdfServiceDomPDF;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class ParadigmAssessmentController extends Controller
{
    protected ParadigmScoringService $scoringService;
    protected ParadigmPdfServiceDomPDF $pdfService;
    
    public function __construct(ParadigmScoringService $scoringService, ParadigmPdfServiceDomPDF $pdfService)
    {
        $this->scoringService = $scoringService;
        $this->pdfService = $pdfService;
    }
    
    /**
     * Display assessment form
     */
    public function show(Request $request, int $templateId): View
    {
        $template = ParadigmFormTemplate::findByShortcodeId($templateId);
        
        if (!$template) {
            abort(404, 'Assessment not found.');
        }
        
        $template->load(['questions', 'personalFields']);

        // Get user profile data for auto-population
        $userProfile = null;
        if (auth()->check()) {
            $user = auth()->user();
            $userProfile = $user->userDetails;
        }

        // Get saved progress
        $savedProgress = null;
        if ($userProfile) {
             $payment = \App\Models\ParadigmPayment::where('template_id', $templateId)
                ->where(function($query) use ($user) {
                    $query->where('client_id', $user->id)
                          ->orWhere('client_email', $user->email);
                })
                ->where('status', '!=', 'expired')
                ->orderBy('created_at', 'desc')
                ->first();
            
            if ($payment && $payment->progress_data) {
                $savedProgress = $payment->progress_data;
            }
        }

        return view('paradigm.assessment', [
            'template' => $template,
            'personal_fields' => $template->personalFields->sortBy('display_order'),
            'user_profile' => $userProfile,
            'saved_progress' => $savedProgress,
        ]);
    }
    
    /**
     * Display assessment form via direct payment link (no login required)
     */
    public function showDirect(Request $request, string $paymentLink): View
    {
        try {
            $payment = \App\Models\ParadigmPayment::where('payment_link', $paymentLink)
                ->with(['template'])
                ->first();
            
            if (!$payment || !$payment->client_id) {
                abort(404, 'Assessment link not found or invalid.');
            }
            
            // Check if assessment is already completed
            // Check if there's a result for this payment/client/template combination
            $existingResult = ParadigmAssessmentResult::where('template_id', $payment->template_id)
                ->where('provider_id', $payment->provider_id)
                ->where(function($query) use ($payment) {
                    $query->where('email', $payment->client_email);
                    if ($payment->client_name && $payment->client_surname) {
                        $query->orWhere(function($q) use ($payment) {
                            $q->where('name', $payment->client_name)
                              ->where('surname', $payment->client_surname);
                        });
                    }
                })
                ->where('created_at', '>=', $payment->created_at)
                ->orderBy('created_at', 'desc')
                ->first();
            
            // Auto-login the user for this session
            \Auth::loginUsingId($payment->client_id);

            if ($existingResult || $payment->status === 'completed') {
            // Assessment already completed - show message instead of error
            // Pass the result ID if available so they can download the report
            $resultId = $existingResult ? $existingResult->id : null;
            
            // If we don't have the result but payment is completed, try to find the result
            if (!$resultId && $payment->status === 'completed') {
                 $latestResult = ParadigmAssessmentResult::where('template_id', $payment->template_id)
                    ->where('provider_id', $payment->provider_id)
                    ->where('email', $payment->client_email)
                    ->orderBy('created_at', 'desc')
                    ->first();
                 if ($latestResult) {
                     $resultId = $latestResult->id;
                 }
            }
            
            $downloadUrl = null;
            if ($resultId) {
                 $downloadUrl = \URL::signedRoute('paradigm.results.download-pdf', ['resultId' => $resultId]);
            }
            
            return view('paradigm.assessment-already-completed', ['resultId' => $resultId, 'downloadUrl' => $downloadUrl]);
        }

            $template = $payment->template;
            if (!$template) {
                abort(404, 'Assessment template not found.');
            }
            
            $template->load(['questions', 'personalFields']);
            
            $userProfile = \Auth::user()->userDetails;

            return view('paradigm.assessment', [
                'template' => $template,
                'personal_fields' => $template->personalFields->sortBy('display_order'),
                'user_profile' => $userProfile,
                'access_token' => null, // Not needed as user is now logged in
                'direct_access' => false, // No longer direct access, it's an authenticated session
                'saved_progress' => $payment->progress_data,
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error in showDirect method', [
                'payment_link' => $paymentLink,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Re-throw with more context
            abort(500, 'An error occurred while loading the assessment. Please try again later.');
        }
    }
    
    /**
     * Submit assessment
     */
    public function submit(Request $request)
    {
        // Set memory limit to 2GB for PDF generation with high-quality images
        ini_set('memory_limit', '2048M');
        
        $validated = $request->validate([
            'template_id' => 'required|exists:paradigm_form_templates,id',
            'personal' => 'required|array',
            'questions' => 'required|array',
            'access_token' => 'nullable|string', // Optional access token to link provider
        ]);

        try {
            $template = ParadigmFormTemplate::findOrFail($validated['template_id']);
            $personalData = $validated['personal'];
            $answers = $validated['questions'];

            $resultData = $this->scoringService->processAssessment($template, $answers, $personalData);

            // Try to get provider_id and user_id
            $providerId = null;
            $userId = auth()->id(); // Get authenticated user ID
            
            // Method 1: Try access token (backwards compatibility)
            $payment = null;
            if (!empty($validated['access_token'])) {
                // SECURITY FIX: Check token is active and not expired
                $accessToken = \App\Models\ParadigmAccessToken::where('token', $validated['access_token'])
                    ->where('status', 'active')
                    ->where(function($q) {
                        $q->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                    })
                    ->with('payment')
                    ->first();
                
                if ($accessToken && $accessToken->payment) {
                    $payment = $accessToken->payment;
                    $providerId = $payment->provider_id;
                    
                    // If no user logged in but payment has client details, use payment client details
                    if (!$userId && $payment->client_name && $payment->client_surname) {
                        $personalData['name'] = $payment->client_name ?? $personalData['name'] ?? '';
                        $personalData['surname'] = $payment->client_surname ?? $personalData['surname'] ?? '';
                        $personalData['gender'] = $payment->client_gender ?? $personalData['gender'] ?? null;
                        // Calculate age_range from DOB if provided
                        if ($payment->client_dob) {
                            $age = now()->diffInYears($payment->client_dob);
                            $personalData['age_range'] = $this->calculateAgeRange($age);
                            $personalData['dob'] = $payment->client_dob->format('Y-m-d');
                        }
                        $personalData['email'] = $payment->client_email ?? $personalData['email'] ?? null;
                    }
                }
            }
            
            // Method 2: If no access token, find payment by user + template (new dashboard flow)
            if (!$providerId && $userId) {
                $userEmail = auth()->user()->email;
                $payment = \App\Models\ParadigmPayment::where('template_id', $template->id)
                    ->where(function($query) use ($userId, $userEmail) {
                        $query->where('client_id', $userId)
                              ->orWhere('client_email', $userEmail);
                    })
                    ->where('status', '!=', 'expired')
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($payment) {
                    $providerId = $payment->provider_id;
                    
                    // Update payment with client_id if not set
                    if (!$payment->client_id) {
                        $payment->update(['client_id' => $userId]);
                    }
                }
            }

            // Get DOB from personal data or payment
            $dob = null;
            if (isset($personalData['dob'])) {
                $dob = is_string($personalData['dob']) ? $personalData['dob'] : $personalData['dob'];
            } elseif ($payment && $payment->client_dob) {
                $dob = $payment->client_dob;
            }

            // Check for existing result to prevent duplicates (within last 5 minutes or identical session)
            $existingResult = ParadigmAssessmentResult::where('template_id', $template->id)
                ->where('provider_id', $providerId)
                ->where('email', $personalData['email'] ?? null)
                ->where('created_at', '>=', now()->subMinutes(5))
                ->first();

            if ($existingResult) {
                // Return JSON response for existing result
                // Use signed URL for download to allow access without login if needed
                $downloadUrl = \URL::signedRoute('paradigm.results.download-pdf', ['resultId' => $existingResult->id]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Assessment already completed.',
                    'download_url' => $downloadUrl,
                    'redirect_url' => route('paradigm.assessment.completion', ['resultId' => $existingResult->id])
                ]);
            }
            
            $assessmentResult = ParadigmAssessmentResult::create([
                'template_id' => $template->id,
                'provider_id' => $providerId, // Store provider_id for PDF generation
                'session_id' => $resultData['session_id'],
                'name' => $personalData['name'] ?? '',
                'surname' => $personalData['surname'] ?? '',
                'gender' => $personalData['gender'] ?? null,
                'age_range' => $personalData['age'] ?? $personalData['age_range'] ?? null,
                'dob' => $dob,
                'phone' => $personalData['phone_visible'] ?? $personalData['phone'] ?? null,
                'email' => $personalData['email'] ?? null,
                'scores_json' => $resultData['scores'],
                'paradigm_type' => $resultData['overall_type'],
                'answers_json' => $answers,
                'completed_at' => $resultData['completed_at'],
            ]);
            
            // Auto-list in marketplace if self-purchased
            try {
                $payment = \App\Models\ParadigmPayment::where('template_id', $template->id)
                    ->where('provider_id', $providerId)
                    ->where(function($query) use ($personalData) {
                        $query->where('client_email', $personalData['email'] ?? null);
                        if (!empty($personalData['name']) && !empty($personalData['surname'])) {
                            $query->orWhere(function($q) use ($personalData) {
                                $q->where('client_name', $personalData['name'])
                                  ->where('client_surname', $personalData['surname']);
                            });
                        }
                    })
                    ->where('created_at', '<=', $assessmentResult->created_at)
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($payment && $payment->self_purchased == 1) {
                    $assessmentResult->update([
                        'is_available_in_marketplace' => true,
                        'marketplace_listed_at' => now(),
                    ]);
                }
            } catch (\Exception $e) {
                \Log::warning('Failed to auto-list in marketplace: ' . $e->getMessage());
            }
            
            // Send assessment completion emails (Templates 107 & 108)
            try {
                $this->sendAssessmentCompletionEmails($assessmentResult, $providerId);
            } catch (\Exception $e) {
                \Log::error('Failed to send assessment completion emails: ' . $e->getMessage());
            }
            
            // Send assessment completion WhatsApp notifications
            try {
                $this->sendAssessmentCompletionWhatsApp($assessmentResult, $providerId);
            } catch (\Exception $e) {
                \Log::error('Failed to send assessment completion WhatsApp: ' . $e->getMessage());
            }

            // Generate PDF and redirect to dashboard after download
            // $pdfResponse = $this->pdfService->generatePdf($assessmentResult);
            
            // Return JSON response with redirect URL for the PDF download
            // This fixes the "blank page" issue on mobile browsers which struggle with AJAX Blob downloads
            // Use signed URL for download to allow access without login if needed
            $downloadUrl = \URL::signedRoute('paradigm.results.download-pdf', ['resultId' => $assessmentResult->id]);

            return response()->json([
                'success' => true,
                'message' => 'Assessment completed successfully!',
                'download_url' => $downloadUrl,
                'redirect_url' => route('paradigm.assessment.completion', ['resultId' => $assessmentResult->id])
            ]);

        } catch (\Exception $e) {
            Log::error('Error processing assessment', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'message' => 'Error processing assessment: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Display assessment results as PDF
     * SAFE: Read-only method that regenerates PDF for download
     */
    public function results(int $resultId)
    {
        // Set memory limit to 2GB for PDF generation with high-quality images
        ini_set('memory_limit', '2048M');
        
        $result = ParadigmAssessmentResult::with('template.pdfPages.pdfElements')->findOrFail($resultId);
        
        // SECURITY FIX: Authorization check - only client, provider, or admin can view
        if (!auth()->check()) {
            abort(403, 'Please log in to view assessment results');
        }
        
        $userId = auth()->id();
        $userEmail = auth()->user()->email;
        $userType = auth()->user()->user_type;
        
        // Check if user is admin (user_type 1), the client (by email), OR the provider (by provider_id)
        $isAdmin = ($userType == 1);
        $isClient = ($result->email === $userEmail);
        $isProvider = ($result->provider_id === $userId);
        
        if (!$isAdmin && !$isClient && !$isProvider) {
            abort(403, 'Unauthorized access to this assessment result');
        }
        
        try {
            // Generate and return PDF response directly
            // generatePdf() returns a Response object, not a file path
            return $this->pdfService->generatePdf($result);
            
        } catch (\Exception $e) {
            // Return error response instead of digital report page
            return response()->json([
                'success' => false,
                'message' => 'PDF generation failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Download PDF results
     */
    public function downloadPdf(Request $request, int $resultId)
    {
        // Set memory limit to 2GB for PDF generation with high-quality images
        ini_set('memory_limit', '2048M');
        
        $result = ParadigmAssessmentResult::with('template.pdfPages.pdfElements')->findOrFail($resultId);
        
        // SECURITY FIX: Authorization check - only client or provider can download
        // Allow if request has valid signature (generated by submit/showDirect) OR if user is logged in
        if (!$request->hasValidSignature()) {
            if (!auth()->check()) {
                abort(403, 'Please log in to download assessment results');
            }
            
            $userId = auth()->id();
            $userEmail = auth()->user()->email;
            
            // Check if user is the client (by email) OR the provider (by provider_id)
            $isClient = ($result->email === $userEmail);
            $isProvider = ($result->provider_id === $userId);
            
            if (!$isClient && !$isProvider) {
                abort(403, 'Unauthorized access to this assessment result');
            }
        }
        
        try {
            return $this->pdfService->generatePdf($result);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'PDF generation failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get assessment by session ID (for returning users)
     */
    public function getBySession(string $sessionId): JsonResponse
    {
        $result = ParadigmAssessmentResult::where('session_id', $sessionId)->first();
        
        if (!$result) {
            return response()->json([
                'success' => false,
                'message' => 'Assessment not found'
            ], 404);
        }
        
        return response()->json([
            'success' => true,
            'result' => $result
        ]);
    }
    
    /**
     * Send assessment completion emails (Templates 107 & 108)
     */
    private function sendAssessmentCompletionEmails($assessmentResult, $providerId)
    {
        try {
            // Configure mail
            \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
            
            // Get company settings
            $settings = \DB::table('general_settings')
                ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no'])
                ->pluck('value', 'key');
            
            $companyName = $settings['company_name'] ?? config('app.name');
            $companyWebsite = $settings['website'] ?? config('app.url');
            $companyEmail = $settings['site_email'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            
            $clientName = ($assessmentResult->name ?? '') . ' ' . ($assessmentResult->surname ?? '');
            $paradigmType = $assessmentResult->paradigm_type ?? '';
            
            // Send email to Provider (Template 107 - Assessment Completed)
            if ($providerId) {
                $provider = \App\Models\User::find($providerId);
                if ($provider) {
                    $template107 = \DB::table('templates')
                        ->where('notification_type', 54) // Assessment Completed notification type
                        ->where('type', 1) // Email type
                        ->where('status', 1) // Active
                        ->first();
                    
                    if ($template107) {
                        $providerName = $provider->name ?? '';
                        $viewUrl = url('/provider/paradigm/results/' . $assessmentResult->id);
                        
                        $subject = str_replace(
                            ['{{provider_name}}', '{{client_name}}', '{{paradigm_type}}', '{{company_name}}'],
                            [$providerName, $clientName, $paradigmType, $companyName],
                            $template107->subject
                        );
                        
                        $content = str_replace(
                            ['{{provider_name}}', '{{client_name}}', '{{paradigm_type}}', '{{company_name}}', '{{website_link}}', '{{view_url}}', '{{company_email}}', '{{company_phone}}'],
                            [$providerName, $clientName, $paradigmType, $companyName, $companyWebsite, $viewUrl, $companyEmail, $companyPhone],
                            $template107->content
                        );
                        
                        $emailData = [
                            'to_email' => $provider->email,
                            'subject' => $subject,
                            'content' => $content
                        ];
                        
                        \Log::info('Sending assessment completed email to provider', ['to_email' => $provider->email]);
                        
                        $emailRequest = new \Illuminate\Http\Request($emailData);
                        $emailController = new \Modules\Communication\app\Http\Controllers\EmailController();
                        $emailController->sendEmail($emailRequest);
                    }
                }
            }
            
            // Email to Client (Template 108) removed as per request to stop "Ticket Assigned" email
            // The template with notification_type 24 was incorrect.

            
        } catch (\Exception $e) {
            \Log::error('Failed to send assessment completion emails: ' . $e->getMessage());
        }
    }
    
    /**
     * Calculate age range from age
     * Must match the exact format used in dropdown options: '65 or over' not '65+'
     */
    private function calculateAgeRange(int $age): string
    {
        if ($age < 18) {
            return 'Under 18';
        } elseif ($age < 25) {
            return '18-24';
        } elseif ($age < 35) {
            return '25-34';
        } elseif ($age < 45) {
            return '35-44';
        } elseif ($age < 55) {
            return '45-54';
        } elseif ($age < 65) {
            return '55-64';
        } else {
            return '65 or over'; // Fixed: Must match dropdown options exactly
        }
    }
    
    /**
     * Send assessment completion WhatsApp notifications
     */
    private function sendAssessmentCompletionWhatsApp($assessmentResult, $providerId)
    {
        try {
            $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
            
            // Generate PDF download link for completed assessment
            $assessmentLink = url('/paradigm/results/' . $assessmentResult->id . '/download-pdf');
            
            // Send to client
            $clientPhone = $assessmentResult->phone ?? null;
            $clientName = trim(($assessmentResult->name ?? '') . ' ' . ($assessmentResult->surname ?? ''));
            
            if (!empty($clientPhone) && !empty($clientName)) {
                $smsRepository->sendProfileCompletedClientWhatsApp(
                    $clientPhone,
                    $clientName,
                    $assessmentLink
                );
            }
            
            // Send to provider
            if ($providerId) {
                $provider = \App\Models\User::with('userDetails')->find($providerId);
                if ($provider && !empty($provider->phone_number)) {
                    $providerName = $provider->name ?? 'Provider';
                    if ($provider->userDetails) {
                        $firstName = $provider->userDetails->first_name ?? '';
                        $lastName = $provider->userDetails->last_name ?? '';
                        $providerName = trim($firstName . ' ' . $lastName) ?: $provider->name;
                    }
                    
                    $smsRepository->sendProfileCompletedProviderWhatsApp(
                        $provider->phone_number,
                        $providerName,
                        $clientName
                    );
                }
            }
            
        } catch (\Exception $e) {
            \Log::error('Failed to send assessment completion WhatsApp: ' . $e->getMessage());
        }
    }
    /**
     * Save assessment progress
     */
    public function saveProgress(Request $request)
    {
        try {
            \Log::info('saveProgress called', ['data' => $request->all()]);

            $validated = $request->validate([
                'template_id' => 'required|exists:paradigm_form_templates,id',
                'progress_data' => 'required|array',
                'access_token' => 'nullable|string',
            ]);

            $userId = auth()->id();
            $payment = null;

            // Method 1: Try access token
            if (!empty($validated['access_token'])) {
                $accessToken = \App\Models\ParadigmAccessToken::where('token', $validated['access_token'])
                    ->where('status', 'active')
                    ->with('payment')
                    ->first();
                
                if ($accessToken && $accessToken->payment) {
                    $payment = $accessToken->payment;
                    \Log::info('Payment found via access token', ['payment_id' => $payment->id]);
                }
            }
            
            // Method 2: If no access token, find payment by user + template
            if (!$payment && $userId) {
                $userEmail = auth()->user()->email;
                $payment = \App\Models\ParadigmPayment::where('template_id', $validated['template_id'])
                    ->where(function($query) use ($userId, $userEmail) {
                        $query->where('client_id', $userId)
                              ->orWhere('client_email', $userEmail);
                    })
                    ->where('status', '!=', 'expired')
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($payment) {
                    \Log::info('Payment found via user/email', ['payment_id' => $payment->id]);
                }
            }

            if ($payment) {
                $payment->progress_data = $validated['progress_data'];
                $payment->save();
                \Log::info('Progress saved successfully');
                
                return response()->json(['success' => true, 'message' => 'Progress saved']);
            }

            \Log::warning('Payment record not found for saveProgress', ['user_id' => $userId, 'template_id' => $validated['template_id']]);
            return response()->json(['success' => false, 'message' => 'Payment record not found'], 404);

        } catch (\Exception $e) {
            Log::error('Error saving progress', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['success' => false, 'message' => 'Error saving progress'], 500);
        }
    }

    /**
     * Show assessment completion page
     */
    public function completion($resultId = null)
    {
        $downloadUrl = null;
        if ($resultId) {
            $downloadUrl = \URL::signedRoute('paradigm.results.download-pdf', ['resultId' => $resultId]);
        }
        return view('paradigm.assessment-completion', ['resultId' => $resultId, 'downloadUrl' => $downloadUrl]);
    }
}
