<?php

namespace App\Http\Controllers;

use App\Models\ProviderIdentityVerification;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;

class ProviderIdentityVerificationController extends Controller
{
    /**
     * Show the identity verification upload page
     */
    public function showUploadPage()
    {
        $userId = Auth::id();
        
        // Check if ID verification is enabled
        if (!providerIdVerificationEnabled()) {
            return redirect()->route('provider.dashboard')->with('error', 'ID verification is not enabled.');
        }
        
        // Get or create verification record
        $verification = ProviderIdentityVerification::firstOrCreate(
            ['user_id' => $userId],
            ['verification_status' => 'pending']
        );
        
        return view('provider.identity-verification', compact('verification'));
    }
    
    /**
     * Upload ID document (front or back)
     */
    public function uploadIdDocument(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'document_type' => 'required|in:id_document',
            'file' => 'required|image|mimes:jpg,jpeg,png,gif|max:2048', // Images only, NO PDFs
        ], [
            'file.required' => 'ID document image is required.',
            'file.image' => 'File must be an image.',
            'file.mimes' => 'ID document must be a JPEG, PNG, or GIF image file. PDF and document files are NOT accepted.',
            'file.max' => 'Image must not exceed 2MB.'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        try {
            $userId = Auth::id();
            $documentType = $request->document_type;
            $file = $request->file('file');
            
            // Get or create verification record
            $verification = ProviderIdentityVerification::firstOrCreate(
                ['user_id' => $userId],
                ['verification_status' => 'pending']
            );
            
            // Store the file
            $filename = $this->storeDocument($file, $userId);
            
            // Update verification record - using id_document_front column for single ID document
            if ($documentType === 'id_document') {
                // Delete old file if exists
                if ($verification->id_document_front) {
                    $this->deleteDocument($verification->id_document_front, $userId);
                }
                $verification->id_document_front = $filename;
            }
            
            $verification->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Document uploaded successfully',
                'data' => [
                    'filename' => $filename,
                    'url' => $verification->getIdFrontUrl() ?? $verification->getIdBackUrl()
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload document: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Upload selfie photo
     */
    public function uploadSelfie(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|image|mimes:jpg,jpeg,png,gif|max:2048', // Images only
        ], [
            'file.required' => 'Selfie photo is required.',
            'file.image' => 'File must be an image.',
            'file.mimes' => 'Selfie must be a JPEG, PNG, or GIF image file.',
            'file.max' => 'Image must not exceed 2MB.'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        try {
            $userId = Auth::id();
            $file = $request->file('file');
            
            // Get or create verification record
            $verification = ProviderIdentityVerification::firstOrCreate(
                ['user_id' => $userId],
                ['verification_status' => 'pending']
            );
            
            // Delete old selfie if exists
            if ($verification->selfie_photo) {
                $this->deleteDocument($verification->selfie_photo, $userId);
            }
            
            // Store the file
            $filename = $this->storeDocument($file, $userId, 'selfie');
            
            // Update verification record
            $verification->selfie_photo = $filename;
            
            // If all documents are uploaded, mark as submitted
            if ($verification->hasAllDocuments()) {
                $verification->markAsSubmitted();
                
                // Set provider verification status to 0 (pending) when documents are submitted
                $user = Auth::user();
                if ($user && $user->provider_verified_status !== 0) {
                    $user->provider_verified_status = 0;
                    $user->save();
                }
                
                // Log out the user after submission so they see proper verification status on next login
                Auth::logout();
                request()->session()->invalidate();
                request()->session()->regenerateToken();
            } else {
                $verification->save();
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Selfie uploaded successfully',
                'data' => [
                    'filename' => $filename,
                    'url' => $verification->getSelfieUrl(),
                    'all_uploaded' => $verification->hasAllDocuments()
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload selfie: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get verification status
     */
    public function getStatus(): JsonResponse
    {
        try {
            $userId = Auth::id();
            $verification = ProviderIdentityVerification::where('user_id', $userId)->first();
            
            if (!$verification) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'status' => 'not_started',
                        'message' => 'No verification submitted yet'
                    ]
                ]);
            }
            
            $data = [
                'status' => $verification->verification_status,
                'has_id_front' => !empty($verification->id_document_front),
                'has_id_back' => !empty($verification->id_document_back),
                'has_selfie' => !empty($verification->selfie_photo),
                'all_uploaded' => $verification->hasAllDocuments(),
                'submitted_at' => $verification->submitted_at?->format('Y-m-d H:i:s'),
                'verified_at' => $verification->verified_at?->format('Y-m-d H:i:s'),
                'rejection_reason' => $verification->rejection_reason
            ];
            
            // Add message based on status
            switch ($verification->verification_status) {
                case 'pending':
                    $data['message'] = 'Your documents are being reviewed by admin.';
                    break;
                case 'verified':
                    $data['message'] = 'Your identity has been verified!';
                    break;
                case 'rejected':
                    $data['message'] = 'Your verification was rejected. Please upload new documents.';
                    break;
            }
            
            return response()->json([
                'success' => true,
                'data' => $data
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get status: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Store document file
     */
    private function storeDocument(UploadedFile $file, int $userId, string $prefix = 'id'): string
    {
        $extension = $file->getClientOriginalExtension();
        $filename = $prefix . '-' . Str::uuid() . '_' . time() . '.' . $extension;
        
        // Create user directory if not exists
        $directory = 'provider-verification/' . $userId;
        if (!Storage::disk('public')->exists($directory)) {
            Storage::disk('public')->makeDirectory($directory);
        }
        
        // Store file
        $file->storeAs($directory, $filename, 'public');
        
        return $filename;
    }
    
    /**
     * Delete document file
     */
    private function deleteDocument(string $filename, int $userId): void
    {
        $path = 'provider-verification/' . $userId . '/' . $filename;
        if (Storage::disk('public')->exists($path)) {
            Storage::disk('public')->delete($path);
        }
    }
    
    /**
     * Admin: Verify provider identity
     */
    public function adminVerify(Request $request): JsonResponse
    {
        try {
            $verificationId = $request->verification_id;
            $userId = $request->user_id;
            
            $verification = ProviderIdentityVerification::findOrFail($verificationId);
            
            $verification->verification_status = 'verified';
            $verification->verified_at = now();
            $verification->verified_by = Auth::id();
            $verification->rejection_reason = null;
            $verification->save();
            
            // Also verify the provider account
            $provider = User::findOrFail($userId);
            
            // Always send verification email and WhatsApp, even if already verified
            $wasAlreadyVerified = $provider->provider_verified_status === 1;
            
            if (!$wasAlreadyVerified) {
                $provider->provider_verified_status = 1;
                $provider->save();
            }
            
            // Send provider approval email (Template 102) and WhatsApp
            try {
                $this->sendProviderApprovalEmail($provider);
            } catch (\Exception $e) {
                \Log::error('Failed to send provider approval email: ' . $e->getMessage());
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Provider identity verified successfully!'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to verify: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Admin: Reject provider identity
     */
    public function adminReject(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'verification_id' => 'required|exists:provider_identity_verifications,id',
                'rejection_reason' => 'required|string|min:10'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $verificationId = $request->verification_id;
            $verification = ProviderIdentityVerification::findOrFail($verificationId);
            
            $verification->verification_status = 'rejected';
            $verification->rejection_reason = $request->rejection_reason;
            $verification->verified_at = null;
            $verification->verified_by = null;
            $verification->save();
            
            // Reset provider verification status to 0 (pending) when rejected
            $userId = $request->user_id;
            $provider = User::findOrFail($userId);
            $provider->provider_verified_status = 0;
            $provider->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Provider identity rejected successfully!'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to reject: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Send provider approval email (Template 102)
     */
    private function sendProviderApprovalEmail($provider)
    {
        \Log::info('=== sendProviderApprovalEmail called ===', [
            'provider_id' => $provider->id,
            'provider_email' => $provider->email
        ]);
        
        try {
            // Configure mail
            \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
            
            // Get template 112 (Provider Verified/Approved)
            $template = \DB::table('templates')
                ->where('notification_type', 6) // Provider Verified notification type
                ->where('type', 1) // Email type
                ->where('status', 1) // Active
                ->first();
            
            if (!$template) {
                \Log::error('Provider approval email template NOT FOUND', [
                    'notification_type' => 6,
                    'type' => 1,
                    'status' => 1
                ]);
                return;
            }
            
            \Log::info('Template found', [
                'template_id' => $template->id,
                'subject' => $template->subject
            ]);
            
            // Get provider details
            $provider->load('userDetails');
            
            // Get company settings
            $settings = \DB::table('general_settings')
                ->whereIn('key', ['company_name', 'website', 'site_email', 'phone_no', 'address'])
                ->pluck('value', 'key');
            
            $companyName = $settings['company_name'] ?? config('app.name');
            $companyWebsite = $settings['website'] ?? config('app.url');
            $companyEmail = $settings['site_email'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyAddress = $settings['address'] ?? '';
            
            // Prepare replacements
            $firstName = $provider->userDetails->first_name ?? '';
            $lastName = $provider->userDetails->last_name ?? '';
            $providerName = trim($firstName . ' ' . $lastName) ?: ($provider->name ?? '');
            $email = $provider->email ?? '';
            $dashboardUrl = url('/provider/dashboard');
            $loginUrl = url('/provider/login');
            $verifiedDate = now()->format('d/m/Y');
            
            // Get category name
            $categoryName = 'Provider'; // Default
            if ($provider->userDetails && !empty($provider->userDetails->category_id)) {
                $category = \DB::table('categories')->where('id', $provider->userDetails->category_id)->first();
                $categoryName = $category->name ?? 'Provider';
            }
            
            // Replace placeholders in subject
            $subject = str_replace(
                ['{{provider_name}}', '{{first_name}}', '{{last_name}}', '{{user_name}}', '{{category_name}}', '{{email}}', '{{email_id}}', '{{date}}', '{{company_name}}', '{{website_link}}', '{{dashboard_url}}', '{{login_url}}', '{{company_email}}', '{{company_phone}}', '{{company_address}}'],
                [$providerName, $firstName, $lastName, $providerName, $categoryName, $email, $email, $verifiedDate, $companyName, $companyWebsite, $dashboardUrl, $loginUrl, $companyEmail, $companyPhone, $companyAddress],
                $template->subject
            );
            
            // Replace placeholders in content
            $content = str_replace(
                ['{{provider_name}}', '{{first_name}}', '{{last_name}}', '{{user_name}}', '{{category_name}}', '{{email}}', '{{email_id}}', '{{date}}', '{{company_name}}', '{{website_link}}', '{{company_email}}', '{{company_phone}}', '{{company_address}}', '{{dashboard_url}}', '{{login_url}}'],
                [$providerName, $firstName, $lastName, $providerName, $categoryName, $email, $email, $verifiedDate, $companyName, $companyWebsite, $companyEmail, $companyPhone, $companyAddress, $dashboardUrl, $loginUrl],
                $template->content
            );
            
            // Send email
            $emailData = [
                'to_email' => $provider->email,
                'subject' => $subject,
                'content' => $content
            ];
            
            \Log::info('Sending provider approval email', [
                'to_email' => $provider->email,
                'subject' => $subject
            ]);
            
            $emailRequest = new \Illuminate\Http\Request($emailData);
            $emailController = new \Modules\Communication\app\Http\Controllers\EmailController();
            $emailController->sendEmail($emailRequest);
            
            \Log::info('Provider approval email sent successfully');
            
            // Also send WhatsApp notification to provider if enabled
            try {
                $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
                $phoneNumber = $provider->phone_number ?? '';
                
                // Category name was already retrieved above for email placeholders
                
                if (!empty($phoneNumber)) {
                    \Log::info('Attempting to send provider verified WhatsApp', [
                        'phone' => $phoneNumber,
                        'provider_name' => $providerName,
                        'category' => $categoryName
                    ]);
                    
                    $smsRepository->sendProviderVerifiedWhatsApp(
                        $phoneNumber,
                        $providerName,
                        $categoryName,
                        $loginUrl
                    );
                    \Log::info('Provider verified WhatsApp sent successfully');
                } else {
                    \Log::warning('Provider has no phone number for WhatsApp', [
                        'provider_id' => $provider->id,
                        'provider_email' => $provider->email
                    ]);
                }
            } catch (\Exception $e) {
                \Log::warning('Failed to send provider verified WhatsApp: ' . $e->getMessage());
            }
            
        } catch (\Exception $e) {
            \Log::error('Failed to send provider approval email: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}

