<?php

namespace App\Http\Controllers;

use App\Services\PaddleService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class PaddleSubscriptionController extends Controller
{
    protected $paddleService;

    public function __construct(PaddleService $paddleService)
    {
        $this->paddleService = $paddleService;
    }

    /**
     * Create Paddle subscription checkout
     */
    public function createCheckout(Request $request)
    {
        $request->validate([
            'price_id' => 'required|string',
            'userId' => 'required|integer'
        ]);

        try {
            // Check if Paddle is enabled
            if (!$this->paddleService->isEnabled()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Paddle payment is not configured. Please contact administrator.'
                ], 400);
            }

            // Create subscription record first
            $subscriptionId = DB::table('subscriptions')->insertGetId([
                'user_id' => $request->userId,
                'price_id' => $request->price_id,
                'status' => 'pending',
                'payment_method' => 'paddle',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create Paddle subscription
            $subscription = $this->paddleService->createSubscription(
                $request->price_id,
                $request->userId
            );

            if ($subscription['success']) {
                // Update subscription record with Paddle subscription ID
                DB::table('subscriptions')
                    ->where('id', $subscriptionId)
                    ->update([
                        'paddle_subscription_id' => $subscription['subscription_id'],
                        'updated_at' => now(),
                    ]);

                return response()->json([
                    'success' => true,
                    'checkout_url' => $subscription['checkout_url'],
                    'subscription_id' => $subscription['subscription_id'],
                    'local_subscription_id' => $subscriptionId,
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => $subscription['error'] ?? 'Failed to create Paddle subscription'
            ], 400);

        } catch (\Exception $e) {
            Log::error('Paddle subscription checkout error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error processing subscription request'
            ], 500);
        }
    }

    /**
     * Handle successful subscription payment
     */
    public function success(Request $request)
    {
        try {
            $subscriptionId = $request->get('subscription_id');
            
            if (!$subscriptionId) {
                return redirect()->route('provider.subscription')
                    ->with('error', 'Invalid subscription response');
            }

            // Get subscription details from Paddle
            $subscription = $this->paddleService->getSubscription($subscriptionId);
            
            if ($subscription['success']) {
                $data = $subscription['data']['data'];
                
                // Update subscription record
                DB::table('subscriptions')
                    ->where('paddle_subscription_id', $subscriptionId)
                    ->update([
                        'status' => 'active',
                        'updated_at' => now(),
                    ]);
                
                return view('provider.subscription.payment_success', [
                    'subscription' => $data,
                    'subscription_id' => $subscriptionId,
                ]);
            }

            return redirect()->route('provider.subscription')
                ->with('error', 'Subscription verification failed');

        } catch (\Exception $e) {
            Log::error('Paddle subscription success error: ' . $e->getMessage());
            return redirect()->route('provider.subscription')
                ->with('error', 'Error processing subscription');
        }
    }

    /**
     * Handle cancelled subscription payment
     */
    public function cancel(Request $request)
    {
        return redirect()->route('provider.subscription')
            ->with('error', 'Subscription was cancelled');
    }

    /**
     * Cancel a subscription
     */
    public function cancelSubscription(Request $request)
    {
        $request->validate([
            'subscription_id' => 'required|string'
        ]);

        try {
            $result = $this->paddleService->cancelSubscription($request->subscription_id);
            
            if ($result['success']) {
                // Update local subscription record
                DB::table('subscriptions')
                    ->where('paddle_subscription_id', $request->subscription_id)
                    ->update([
                        'status' => 'cancelled',
                        'updated_at' => now(),
                    ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Subscription cancelled successfully'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => $result['error'] ?? 'Failed to cancel subscription'
            ], 400);

        } catch (\Exception $e) {
            Log::error('Paddle cancel subscription error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error cancelling subscription'
            ], 500);
        }
    }

    /**
     * Handle Paddle webhook for subscriptions
     */
    public function webhook(Request $request)
    {
        try {
            $payload = $request->getContent();
            $signature = $request->header('Paddle-Signature');
            
            // Verify webhook signature
            if (!$this->paddleService->verifyWebhook($payload, $signature)) {
                Log::warning('Invalid Paddle webhook signature');
                return response()->json(['error' => 'Invalid signature'], 400);
            }

            $data = json_decode($payload, true);
            $eventType = $data['event_type'] ?? '';

            Log::info('Paddle subscription webhook received', [
                'event_type' => $eventType,
                'data' => $data
            ]);

            switch ($eventType) {
                case 'subscription.created':
                    $this->handleSubscriptionCreated($data);
                    break;
                    
                case 'subscription.updated':
                    $this->handleSubscriptionUpdated($data);
                    break;
                    
                case 'subscription.canceled':
                    $this->handleSubscriptionCancelled($data);
                    break;
                    
                case 'subscription.paused':
                    $this->handleSubscriptionPaused($data);
                    break;
                    
                case 'subscription.resumed':
                    $this->handleSubscriptionResumed($data);
                    break;
                    
                default:
                    Log::info('Unhandled Paddle webhook event', ['event_type' => $eventType]);
            }

            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            Log::error('Paddle subscription webhook error: ' . $e->getMessage());
            return response()->json(['error' => 'Webhook processing failed'], 500);
        }
    }

    /**
     * Handle subscription created webhook
     */
    private function handleSubscriptionCreated($data)
    {
        $subscription = $data['data'];
        $paddleSubscriptionId = $subscription['id'];
        
        // Update local subscription record
        DB::table('subscriptions')
            ->where('paddle_subscription_id', $paddleSubscriptionId)
            ->update([
                'status' => 'active',
                'updated_at' => now(),
            ]);
        
        Log::info('Paddle subscription created', [
            'subscription_id' => $paddleSubscriptionId
        ]);
    }

    /**
     * Handle subscription updated webhook
     */
    private function handleSubscriptionUpdated($data)
    {
        $subscription = $data['data'];
        $paddleSubscriptionId = $subscription['id'];
        $status = $subscription['status'] ?? '';
        
        // Map Paddle status to local status
        $localStatus = $this->mapPaddleStatusToLocal($status);
        
        if ($localStatus) {
            DB::table('subscriptions')
                ->where('paddle_subscription_id', $paddleSubscriptionId)
                ->update([
                    'status' => $localStatus,
                    'updated_at' => now(),
                ]);
        }
    }

    /**
     * Handle subscription cancelled webhook
     */
    private function handleSubscriptionCancelled($data)
    {
        $subscription = $data['data'];
        $paddleSubscriptionId = $subscription['id'];
        
        DB::table('subscriptions')
            ->where('paddle_subscription_id', $paddleSubscriptionId)
            ->update([
                'status' => 'cancelled',
                'updated_at' => now(),
            ]);
        
        Log::info('Paddle subscription cancelled', [
            'subscription_id' => $paddleSubscriptionId
        ]);
    }

    /**
     * Handle subscription paused webhook
     */
    private function handleSubscriptionPaused($data)
    {
        $subscription = $data['data'];
        $paddleSubscriptionId = $subscription['id'];
        
        DB::table('subscriptions')
            ->where('paddle_subscription_id', $paddleSubscriptionId)
            ->update([
                'status' => 'paused',
                'updated_at' => now(),
            ]);
    }

    /**
     * Handle subscription resumed webhook
     */
    private function handleSubscriptionResumed($data)
    {
        $subscription = $data['data'];
        $paddleSubscriptionId = $subscription['id'];
        
        DB::table('subscriptions')
            ->where('paddle_subscription_id', $paddleSubscriptionId)
            ->update([
                'status' => 'active',
                'updated_at' => now(),
            ]);
    }

    /**
     * Map Paddle subscription status to local status
     */
    private function mapPaddleStatusToLocal($paddleStatus)
    {
        $statusMap = [
            'active' => 'active',
            'canceled' => 'cancelled',
            'paused' => 'paused',
            'past_due' => 'past_due',
            'trialing' => 'trialing',
        ];

        return $statusMap[$paddleStatus] ?? null;
    }
}
