<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ParadigmFormTemplate;
use App\Models\ParadigmPdfPage;
use App\Models\ParadigmPdfElement;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class ParadigmPdfBuilderController extends Controller
{
    /**
     * Display PDF Builder index page
     */
    public function index(): View
    {
        $templates = ParadigmFormTemplate::with(['pdfPages', 'questions'])->get();
        
        return view('admin.paradigm.pdf-builder.index', compact('templates'));
    }
    
    /**
     * Display PDF Builder for specific template
     */
    public function builder(Request $request, int $templateId = null): View
    {
        // Handle both route parameter and query parameter
        $templateId = $templateId ?? $request->get('template', 1);
        
        $template = ParadigmFormTemplate::with(['pdfPages.pdfElements'])->find($templateId);
        
        if (!$template) {
            abort(404, 'Template not found');
        }
        
        $available_elements = $this->getAvailableElements();
        
        return view('admin.paradigm.pdf-builder.builder', compact('template', 'available_elements'));
    }

    public function saveElements(Request $request): JsonResponse
    {
        Log::info('Save elements request received', $request->all());

        try {
            $validated = $request->validate([
                'template_id' => 'required|exists:paradigm_form_templates,id',
                'page_number' => 'required|integer|min:1',
                'elements' => 'present|array',
                'background_image' => 'nullable|string',
            ]);

            $page = ParadigmPdfPage::firstOrCreate(
                [
                    'template_id' => $validated['template_id'],
                    'page_number' => $validated['page_number'],
                ],
                [
                    'page_title' => 'Page ' . $validated['page_number'],
                    'is_content_page' => true,
                    'display_order' => $validated['page_number'],
                ]
            );
            
            ParadigmPdfElement::where('page_id', $page->id)->delete();

            if (!empty($validated['elements'])) {
                foreach ($validated['elements'] as $index => $elementData) {
                    ParadigmPdfElement::create([
                        'template_id' => $validated['template_id'],
                        'page_id' => $page->id,
                        'page_number' => $validated['page_number'],
                        'element_type' => $elementData['element_type'],
                        'element_key' => $elementData['element_key'],
                        'text_template' => $elementData['text_template'],
                        'x_pct' => $elementData['x_pct'],
                        'y_pct' => $elementData['y_pct'],
                        'w_pct' => $elementData['w_pct'],
                        'h_pct' => $elementData['h_pct'],
                        'font_px' => $elementData['font_px'],
                        'color' => $elementData['color'],
                        'text_align' => $elementData['text_align'],
                        'display_order' => $index + 1,
                    ]);
                }
            }

            if (isset($validated['background_image'])) {
                if ($validated['background_image'] === 'DELETE') {
                    if ($page->background_image_path) {
                        Storage::disk('public')->delete($page->background_image_path);
                    }
                    $page->update(['background_image_path' => null]);
                } elseif (strpos($validated['background_image'], 'data:image/') === 0) {
                    $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $validated['background_image']));
                    $filename = 'paradigm-backgrounds/template-' . $validated['template_id'] . '-page-' . $validated['page_number'] . '-' . time() . '.png';
                    
                    if ($page->background_image_path) {
                        Storage::disk('public')->delete($page->background_image_path);
                    }
                    
                    Storage::disk('public')->put($filename, $imageData);
                    $page->update(['background_image_path' => $filename]);
                }
            }

            return response()->json(['success' => true, 'message' => 'Elements saved successfully.']);

        } catch (ValidationException $e) {
            Log::error('Validation error during saveElements', ['errors' => $e->errors()]);
            return response()->json(['success' => false, 'message' => 'Validation failed: ' . $e->getMessage(), 'errors' => $e->errors()], 422);
        } catch (\Exception $e) {
            Log::error('Error during saveElements', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['success' => false, 'message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }
    
    /**
     * Get PDF elements for template
     */
    public function getElements(int $templateId): JsonResponse
    {
        $template = ParadigmFormTemplate::with(['pdfPages.pdfElements'])->find($templateId);
        
        if (!$template) {
            return response()->json(['error' => 'Template not found'], 404);
        }
        
        return response()->json([
            'pages' => $template->pdfPages->map(function ($page) {
                return [
                    'id' => $page->id,
                    'page_number' => $page->page_number,
                    'background_image_path' => $page->background_image_path,
                    'elements' => $page->pdfElements->map(function ($element) {
                        return [
                            'id' => $element->id,
                            'element_type' => $element->element_type,
                            'element_key' => $element->element_key,
                            'text_template' => $element->text_template,
                            'x_pct' => $element->x_pct,
                            'y_pct' => $element->y_pct,
                            'w_pct' => $element->w_pct,
                            'font_px' => $element->font_px,
                            'text_align' => $element->text_align,
                            'font_weight' => $element->font_weight,
                            'color' => $element->color,
                            'bg_color' => $element->bg_color,
                            'fg_color' => $element->fg_color,
                        ];
                    })
                ];
            })
        ]);
    }

    /**
     * Define all available PDF elements for the builder
     */
    private function getAvailableElements(): array
    {
        return [
            'Personal Details' => [
                ['key' => 'name', 'label' => 'First Name'],
                ['key' => 'surname', 'label' => 'Last Name'],
                ['key' => 'email', 'label' => 'Email'],
                ['key' => 'phone', 'label' => 'Phone'],
                ['key' => 'age_range', 'label' => 'Age Group'],
                ['key' => 'gender', 'label' => 'Gender'],
                ['key' => 'date_completed', 'label' => 'Date Completed'],
            ],
            'Brain Frames (Part 1)' => [
                ['key' => 'lu_percentage', 'label' => 'LU Score (%)'],
                ['key' => 'll_percentage', 'label' => 'LL Score (%)'],
                ['key' => 'ru_percentage', 'label' => 'RU Score (%)'],
                ['key' => 'rl_percentage', 'label' => 'RL Score (%)'],
            ],
            'Brain Fields (Part 2)' => [
                ['key' => 'question_asker_percentage', 'label' => 'Question Asker (%)'],
                ['key' => 'fact_finder_percentage', 'label' => 'Fact Finder (%)'],
                ['key' => 'rule_keeper_percentage', 'label' => 'Rule Keeper (%)'],
                ['key' => 'task_manager_percentage', 'label' => 'Task Manager (%)'],
                ['key' => 'creator_percentage', 'label' => 'Creator (%)'],
                ['key' => 'big_thinker_percentage', 'label' => 'Big Thinker (%)'],
                ['key' => 'friend_maker_percentage', 'label' => 'Friend Maker (%)'],
                ['key' => 'empath_percentage', 'label' => 'Empath (%)'],
            ],
            'Text Elements' => [
                ['key' => 'paradigm_type', 'label' => 'Paradigm Type'],
                ['key' => 'dominance_category', 'label' => 'Dominance Category'],
            ],
            'New Elements (Auto-sizing)' => [
                ['key' => 'full_name', 'label' => 'Full Name (Auto-size)'],
                ['key' => 'user_details_block', 'label' => 'User Details Block (Auto-size)'],
                ['key' => 'provider_details_block', 'label' => 'Provider Details Block (Auto-size)'],
            ],
        ];
    }
}