<?php

namespace Modules\Leads\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Modules\Leads\app\Models\UserFormInput;
use Modules\Leads\app\Models\LeadPricingRule;
use Modules\Categories\app\Models\Categories;
use Modules\Product\app\Models\Product;
use App\Models\User;
use App\Models\UserDetail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Modules\Leads\app\Models\UserHelpFormResponse;

class MarketplaceLeadController extends Controller
{
    /**
     * Get active categories with provider count
     */
    public function getCategories(): JsonResponse
    {
        try {
            // Get top-level categories (parent_id is NULL or 0)
            $categories = Categories::where(function($query) {
                    $query->whereNull('parent_id')->orWhere('parent_id', 0);
                })
                ->where('status', 1)
                ->get(['id', 'name', 'image']);

            return response()->json([
                'success' => true,
                'categories' => $categories
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching categories: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch categories'
            ], 500);
        }
    }

    /**
     * Get services for a category
     */
    public function getServices(Request $request): JsonResponse
    {
        try {
            $categoryId = $request->category_id;

            if (!$categoryId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Category ID is required'
                ], 422);
            }

            $services = Product::where('source_category', $categoryId)
                ->where('source_type', 'service')
                ->where('status', 1)
                ->get(['id', 'source_name as name', 'source_description as description']);

            return response()->json([
                'success' => true,
                'services' => $services
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching services: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch services'
            ], 500);
        }
    }

    /**
     * Calculate unlock price in real-time
     */
    public function calculatePrice(Request $request): JsonResponse
    {
        try {
            $categoryId = $request->category_id;
            $fieldsCount = (int)($request->fields_filled ?? 0);
            $charCount = (int)($request->message_chars ?? 0);

            if (!$categoryId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Category ID is required'
                ], 422);
            }

            $pricingRule = LeadPricingRule::where('category_id', $categoryId)
                ->where('active', true)
                ->first();

            if (!$pricingRule) {
                // Get default pricing from configuration
                $defaultPrice = config('leads.default_unlock_price', 15.00);
                
                return response()->json([
                    'success' => true,
                    'unlock_price' => number_format($defaultPrice, 2, '.', ''),
                    'breakdown' => [
                        'base_price' => number_format($defaultPrice, 2, '.', ''),
                        'field_price' => '0.00',
                        'char_price' => '0.00',
                        'total' => number_format($defaultPrice, 2, '.', '')
                    ],
                    'note' => 'Using default pricing - no rule found'
                ]);
            }

            $base = (float)$pricingRule->base_price;
            $fieldPrice = $fieldsCount * (float)$pricingRule->price_per_field;
            $charPrice = ceil($charCount / 100) * (float)$pricingRule->price_per_100_chars;
            $total = $base + $fieldPrice + $charPrice;

            // Apply min/max caps
            $finalPrice = max(
                (float)$pricingRule->min_price,
                min($total, (float)$pricingRule->max_price)
            );

            return response()->json([
                'success' => true,
                'unlock_price' => number_format($finalPrice, 2, '.', ''),
                'breakdown' => [
                    'base_price' => number_format($base, 2, '.', ''),
                    'field_price' => number_format($fieldPrice, 2, '.', ''),
                    'char_price' => number_format($charPrice, 2, '.', ''),
                    'total' => number_format($finalPrice, 2, '.', '')
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error calculating price: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to calculate price'
            ], 500);
        }
    }

    /**
     * Submit marketplace lead
     */
    public function submit(Request $request): JsonResponse
    {
        // Check if user is logged in
        $isLoggedIn = Auth::check();
        
        // Base validation rules (always required)
        $rules = [
            'category_id' => 'required|exists:categories,id',
            'service_id' => 'nullable|exists:products,id',
            'message_text' => 'required|string|min:50|max:2000',
        ];
        
        // Add registration fields validation only if user is NOT logged in
        if (!$isLoggedIn) {
            $rules = array_merge($rules, [
                'first_name' => 'required|string|max:50',
                'last_name' => 'required|string|max:50',
                'name' => 'required|string|max:100',
                'email' => 'required|email|unique:users,email',
                'phone_number' => 'required|string',
                'password' => 'required|string|min:8',
                'country' => 'required|string',
                'state' => 'required|string',
                'city' => 'required|string',
                'address' => 'nullable|string|max:150',
                'postal_code' => 'nullable|string|max:20',
                'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
                'dob' => 'nullable|date|before:today',
            ]);
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // For non-logged in users, check if OTP is enabled and send SMS OTP
        if (!$isLoggedIn) {
            $regStatus = DB::table('general_settings')->where('key', 'register')->value('value');
            
            if ($regStatus === "1") {
                // OTP is enabled - Use EXACT same approach as Join Us form
                // Store form data in session
                session([
                    'help_form_data' => $request->all(),
                    'help_form_email' => $request->email,
                    'help_form_phone' => $request->phone_number,
                ]);

                // Get OTP settings (same as Join Us)
                $settings = \Modules\GlobalSetting\Entities\GlobalSetting::whereIn('key', ['otp_digit_limit', 'otp_expire_time', 'otp_type'])
                    ->pluck('value', 'key');

                // Generate OTP using UserRepository (EXACT SAME as Join Us form)
                $userRepository = app(\App\Repositories\Contracts\UserInterface::class);
                $otpData = $userRepository->generateOtp($request->email, (int) $settings['otp_digit_limit']);
                
                // Get OTP template (same as Join Us)
                $notificationType = 2;
                $type = $settings['otp_type'] === 'email' ? 1 : 2;
                $template = \Modules\GlobalSetting\app\Models\Templates::select('subject', 'content')
                    ->where('type', $type)
                    ->where('notification_type', $notificationType)
                    ->first();

                // For SMS, return template WITHOUT replacing variables (frontend will send it)
                $templateContent = $template->content ?? '';
                if ($settings['otp_type'] !== 'sms') {
                    // For email, replace variables in backend
                    $templateContent = str_replace(
                        ['{{user_name}}', '{{otp}}'],
                        [$request->name, $otpData['otp']],
                        $templateContent
                    );
                }

                Log::info("Help Form OTP generated", ['email' => $request->email, 'phone' => $request->phone_number]);

                // Return EXACT SAME format as Join Us form
                return response()->json([
                    'success' => true,
                    'requires_otp' => true,
                    'code' => 200,
                    'register_status' => '1',
                    'first_name' => $request->first_name,
                    'last_name' => $request->last_name,
                    'name' => $request->name,
                    'phone_number' => $request->phone_number,
                    'email' => $request->email,
                    'password' => $request->password,
                    'gender' => $request->gender,
                    'dob' => $request->dob,
                    'address' => $request->address,
                    'country' => $request->country,
                    'state' => $request->state,
                    'city' => $request->city,
                    'postal_code' => $request->postal_code,
                    'otp_digit_limit' => $settings['otp_digit_limit'],
                    'otp_expire_time' => $settings['otp_expire_time'],
                    'otp_type' => $settings['otp_type'],
                    'otp' => $otpData['otp'],
                    'expires_at' => $otpData['expires_at'],
                    'email_subject' => $template->subject ?? '',
                    'email_content' => $templateContent,
                ]);
            }
        }

        DB::beginTransaction();
        try {
            // 1. Get or create user
            if ($isLoggedIn) {
                // User is logged in - use existing user
                $user = Auth::user();
                Log::info('Logged in user submitting marketplace lead', ['user_id' => $user->id]);
            } else {
                // User is not logged in - create new user
                $user = User::where('email', $request->email)->first();
                
                if (!$user) {
                    // Create user with proper name
                    $fullName = trim(($request->first_name ?? '') . ' ' . ($request->last_name ?? ''));
                    if (empty($fullName)) {
                        $fullName = $request->name ?? 'User';
                    }
                    
                    $user = User::create([
                        'name' => $fullName,
                        'email' => $request->email,
                        'phone_number' => $request->phone_number,
                        'password' => Hash::make($request->password ?? str()->random(12)),
                        'user_type' => 3, // Customer
                        'status' => 1,
                    ]);

                    // Create user details record with all form data
                    UserDetail::create([
                        'user_id' => $user->id,
                        'first_name' => $request->first_name ?? '',
                        'last_name' => $request->last_name ?? '',
                        'gender' => $request->gender ?? null,
                        'dob' => $request->dob ?? null,
                        'address' => $request->address ?? null,
                        'city' => $request->city ?? null,
                        'state' => $request->state ?? null,
                        'country' => $request->country ?? null,
                        'postal_code' => $request->postal_code ?? null,
                    ]);

                    // Auto-login the new user
                    Auth::login($user);

                    Log::info('New user created for marketplace lead', ['user_id' => $user->id, 'email' => $user->email, 'name' => $fullName]);
                } else {
                    // User exists - auto-login
                    Auth::login($user);
                }
            }

            // 2. Calculate pricing
            $fieldsFilledCount = $this->countFieldsFilled($request);
            $messageCharCount = strlen($request->message_text);
            
            $pricingRule = LeadPricingRule::where('category_id', $request->category_id)
                ->where('active', true)
                ->first();
                
            $calculatedPrice = $pricingRule 
                ? $this->calculateFinalPrice($pricingRule, $fieldsFilledCount, $messageCharCount) 
                : config('leads.default_unlock_price', 15.00);

            // 3. Calculate quality score
            $qualityScore = $this->calculateQualityScore($request, $fieldsFilledCount, $messageCharCount);

            // 4. Create lead
            // Prepare form inputs - use user details if logged in, otherwise use form data
            $formInputs = [];
            if ($isLoggedIn) {
                // User is logged in - use their existing details
                $userDetails = $user->userDetails;
                $formInputs = [
                    'first_name' => $userDetails->first_name ?? '',
                    'last_name' => $userDetails->last_name ?? '',
                    'email' => $user->email,
                    'phone' => $user->phone_number,
                    'country' => $userDetails->country ?? '',
                    'state' => $userDetails->state ?? '',
                    'city' => $userDetails->city ?? '',
                    'address' => $userDetails->address ?? '',
                    'postal_code' => $userDetails->postal_code ?? '',
                    'gender' => $userDetails->gender ?? '',
                    'dob' => $userDetails->dob ?? '',
                ];
            } else {
                // User is not logged in - use form data
                $formInputs = [
                    'first_name' => $request->first_name,
                    'last_name' => $request->last_name,
                    'email' => $request->email,
                    'phone' => $request->phone_number,
                    'country' => $request->country,
                    'state' => $request->state,
                    'city' => $request->city,
                    'address' => $request->address,
                    'postal_code' => $request->postal_code,
                    'gender' => $request->gender,
                    'dob' => $request->dob,
                ];
            }
            
            $lead = UserFormInput::create([
                'user_id' => $user->id,
                'category_id' => $request->category_id,
                'service_id' => $request->service_id,
                'lead_source' => 'marketplace',
                'message_text' => $request->message_text,
                'message_char_count' => $messageCharCount,
                'form_fields_filled_count' => $fieldsFilledCount,
                'calculated_unlock_price' => $calculatedPrice,
                'quality_score' => $qualityScore,
                'is_unlocked' => false,
                'blur_details' => true,
                'marketplace_status' => 'pending',
                'expires_at' => now()->addDays(30),
                'form_inputs' => $formInputs
            ]);

            // 5. Save question responses
            if ($request->has('question_responses') && is_array($request->question_responses)) {
                foreach ($request->question_responses as $response) {
                    if (isset($response['answer_ids'])) {
                        // Multiple answers (checkbox)
                        foreach ($response['answer_ids'] as $answerId) {
                            UserHelpFormResponse::create([
                                'lead_id' => $lead->id,
                                'question_id' => $response['question_id'],
                                'answer_id' => $answerId,
                            ]);
                        }
                    } else {
                        // Single answer or text
                        UserHelpFormResponse::create([
                            'lead_id' => $lead->id,
                            'question_id' => $response['question_id'],
                            'answer_id' => $response['answer_id'] ?? null,
                            'answer_text' => $response['answer_text'] ?? null,
                        ]);
                    }
                }
            }

            DB::commit();

            Log::info('Marketplace lead created successfully', [
                'lead_id' => $lead->id,
                'user_id' => $user->id,
                'category_id' => $request->category_id,
                'unlock_price' => $calculatedPrice
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Your request has been submitted successfully! Providers will contact you soon.',
                'lead_id' => $lead->id,
                'unlock_price' => number_format($calculatedPrice, 2, '.', '')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Marketplace lead submission error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while submitting your request. Please try again.'
            ], 500);
        }
    }

    /**
     * Count how many fields were filled
     */
    private function countFieldsFilled(Request $request): int
    {
        $fields = [
            'service_id', 'first_name', 'last_name', 'email', 'phone_number',
            'country', 'state', 'city', 'address', 'postal_code', 'gender', 'dob'
        ];

        $count = 0;
        foreach ($fields as $field) {
            if ($request->filled($field) && !empty($request->$field)) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Calculate final price based on pricing rule
     */
    private function calculateFinalPrice(LeadPricingRule $pricingRule, int $fieldsCount, int $charCount): float
    {
        $base = (float)$pricingRule->base_price;
        $fieldPrice = $fieldsCount * (float)$pricingRule->price_per_field;
        $charPrice = ceil($charCount / 100) * (float)$pricingRule->price_per_100_chars;
        $total = $base + $fieldPrice + $charPrice;

        // Apply min/max caps
        return max(
            (float)$pricingRule->min_price,
            min($total, (float)$pricingRule->max_price)
        );
    }

    /**
     * Calculate quality score (0-100)
     */
    private function calculateQualityScore(Request $request, int $fieldsFilledCount, int $messageLength): int
    {
        $score = 0;

        // Completeness (30 points) - based on 12 total fields
        $score += min(30, ($fieldsFilledCount / 12) * 30);

        // Message detail (20 points) - based on 500 chars being "good"
        $score += min(20, ($messageLength / 500) * 20);

        // Has service selected (15 points)
        if ($request->filled('service_id')) {
            $score += 15;
        }

        // Has full address (15 points)
        if ($request->filled('address') && $request->filled('postal_code')) {
            $score += 15;
        }

        // Has additional details (20 points)
        if ($request->filled('gender')) $score += 10;
        if ($request->filled('dob')) $score += 10;

        return min(100, (int)round($score));
    }

    /**
     * Get user's submitted requests
     */
    public function getUserRequests(Request $request): JsonResponse
    {
        try {
            $userId = Auth::id();
            
            if (!$userId) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            $requests = UserFormInput::with(['category:id,name', 'service:id,source_name'])
                ->where('user_id', $userId)
                ->where('lead_source', 'marketplace')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $requests
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching user requests: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch requests'
            ], 500);
        }
    }

    /**
     * Generate OTP for verification
     */
    private function generateOtp(string $email, int $digitLimit): array
    {
        $otp = '';
        for ($i = 0; $i < $digitLimit; $i++) {
            $otp .= random_int(0, 9);
        }

        $otpExpireTime = DB::table('general_settings')->where('key', 'otp_expire_time')->value('value');
        $expiresAt = now()->addMinutes((int) explode(' ', $otpExpireTime)[0]);

        // Store OTP in otp_settings table (same as Join Us form)
        $existingOtp = DB::table('otp_settings')->where('email', $email)->first();
        
        if ($existingOtp) {
            DB::table('otp_settings')
                ->where('email', $email)
                ->update([
                    'otp' => $otp,
                    'expires_at' => $expiresAt,
                ]);
        } else {
            DB::table('otp_settings')->insert([
                'email' => $email,
                'otp' => $otp,
                'expires_at' => $expiresAt,
            ]);
        }

        return [
            'otp' => $otp,
            'expires_at' => $expiresAt
        ];
    }

    /**
     * Get OTP template for email/SMS
     */
    private function getOtpTemplate(string $otpType): array
    {
        $notificationType = 2;
        $type = $otpType === 'email' ? 1 : 2;

        $template = DB::table('templates')
            ->select('subject', 'content')
            ->where('type', $type)
            ->where('notification_type', $notificationType)
            ->first();

        return $template ? (array) $template : [];
    }

    /**
     * Verify OTP and complete lead submission
     */
    public function verifyOtpAndSubmit(Request $request): JsonResponse
    {
        // Validate only OTP
        $validator = Validator::make($request->all(), [
            'otp' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Retrieve form data from session
        $formData = session('help_form_data');
        $email = session('help_form_email');
        $phoneNumber = session('help_form_phone');

        if (!$formData || !$email) {
            return response()->json([
                'success' => false,
                'message' => 'Session expired. Please submit the form again.'
            ], 400);
        }

        // Verify OTP from otp_settings table (same as Join Us form)
        $otpSetting = DB::table('otp_settings')->where('email', $email)->first();

        if (!$otpSetting) {
            return response()->json([
                'success' => false,
                'message' => 'OTP not found. Please submit the form again.'
            ], 400);
        }

        // Check if OTP has expired
        $currentDateTime = now();
        $expiresAt = \Carbon\Carbon::parse($otpSetting->expires_at);
        if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
            return response()->json([
                'success' => false,
                'message' => 'OTP has expired'
            ], 400);
        }

        // Verify OTP matches
        if ($request->otp !== $otpSetting->otp) {
            return response()->json([
                'success' => false,
                'message' => 'The OTP you entered is invalid. Please check and try again.'
            ], 400);
        }

        // Delete OTP after successful verification (same as Join Us form)
        DB::table('otp_settings')->where('email', $email)->delete();

        // Clear session data
        session()->forget(['help_form_data', 'help_form_email', 'help_form_phone']);

        // OTP verified - now create the user and lead using session data
        DB::beginTransaction();
        try {
            // 1. Create or get user (using session form data)
            $user = User::where('email', $email)->first();
            
            if (!$user) {
                // Create user with proper name - ensure it's never null
                $firstName = $formData['first_name'] ?? '';
                $lastName = $formData['last_name'] ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
                
                // If fullName is empty or just spaces, use the 'name' field or default
                if (empty($fullName) || $fullName === ' ') {
                    $fullName = $formData['name'] ?? null;
                }
                
                // Final fallback - use email username if still empty
                if (empty($fullName)) {
                    $fullName = explode('@', $email)[0];
                }
                
                Log::info('Creating user from Help Form', [
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'name_field' => $formData['name'] ?? 'not set',
                    'final_name' => $fullName,
                    'email' => $email
                ]);
                
                $user = User::create([
                    'name' => $fullName,
                    'email' => $email,
                    'phone_number' => $phoneNumber,
                    'password' => Hash::make($formData['password']),
                    'user_type' => 3, // Customer
                    'status' => 1,
                ]);
                
                // Create user details record with all form data
                UserDetail::create([
                    'user_id' => $user->id,
                    'first_name' => $formData['first_name'] ?? '',
                    'last_name' => $formData['last_name'] ?? '',
                    'gender' => $formData['gender'] ?? null,
                    'dob' => $formData['dob'] ?? null,
                    'address' => $formData['address'] ?? null,
                    'city' => $formData['city'] ?? null,
                    'state' => $formData['state'] ?? null,
                    'country' => $formData['country'] ?? null,
                    'postal_code' => $formData['postal_code'] ?? null,
                ]);
                
                Log::info('New user created for marketplace lead after OTP verification', ['user_id' => $user->id, 'email' => $user->email, 'name' => $fullName]);
            }

            // 2. Calculate pricing
            $fieldsFilledCount = count(array_filter($formData));
            $messageCharCount = strlen($formData['message_text'] ?? '');
            
            $pricingRule = LeadPricingRule::where('category_id', $formData['category_id'])
                ->where('active', true)
                ->first();
                
            $calculatedPrice = $pricingRule 
                ? $this->calculateFinalPrice($pricingRule, $fieldsFilledCount, $messageCharCount) 
                : config('leads.default_unlock_price', 15.00);

            // 3. Calculate quality score (simple version for OTP flow)
            $qualityScore = min(100, (int)round(($fieldsFilledCount / 15) * 100));

            // 4. Create lead
            $lead = UserFormInput::create([
                'user_id' => $user->id,
                'category_id' => $formData['category_id'],
                'service_id' => $formData['service_id'] ?? null,
                'lead_source' => 'marketplace',
                'message_text' => $formData['message_text'],
                'message_char_count' => $messageCharCount,
                'form_fields_filled_count' => $fieldsFilledCount,
                'calculated_unlock_price' => $calculatedPrice,
                'quality_score' => $qualityScore,
                'is_unlocked' => false,
                'blur_details' => true,
                'marketplace_status' => 'pending',
                'expires_at' => now()->addDays(30),
                'form_inputs' => [
                    'first_name' => $formData['first_name'],
                    'last_name' => $formData['last_name'],
                    'email' => $email,
                    'phone' => $phoneNumber,
                    'country' => $formData['country'],
                    'state' => $formData['state'],
                    'city' => $formData['city'],
                    'address' => $formData['address'],
                    'postal_code' => $formData['postal_code'],
                    'gender' => $formData['gender'],
                    'dob' => $formData['dob'],
                ]
            ]);

            // 5. Save question responses (if any)
            if (!empty($formData['question_responses'])) {
                foreach ($formData['question_responses'] as $response) {
                    HelpFormAnswer::create([
                        'lead_id' => $lead->id,
                        'question_id' => $response['question_id'],
                        'answer_value' => $response['value'],
                    ]);
                }
            }

            // 6. Auto-login the new user
            Auth::login($user);

            DB::commit();

            Log::info('Lead created successfully after OTP verification', [
                'lead_id' => $lead->id,
                'user_id' => $user->id,
                'category_id' => $formData['category_id']
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Your request has been submitted successfully!',
                'lead_id' => $lead->id
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create lead after OTP verification', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to submit request. Please try again.'
            ], 500);
        }
    }

    /**
     * Count fields filled from data array
     */
    private function countFieldsFilledFromData(array $data): int
    {
        $fields = ['first_name', 'last_name', 'email', 'phone_number', 'gender', 'dob', 'address', 'country', 'state', 'city', 'postal_code', 'service_id'];
        $count = 0;
        foreach ($fields as $field) {
            if (!empty($data[$field])) {
                $count++;
            }
        }
        return $count;
    }

    /**
     * Calculate quality score from data array
     */
    private function calculateQualityScoreFromData(array $data, int $fieldsFilledCount, int $messageCharCount): int
    {
        $score = 0;

        // Message quality (40 points)
        if ($messageCharCount >= 200) $score += 40;
        elseif ($messageCharCount >= 100) $score += 30;
        elseif ($messageCharCount >= 50) $score += 20;

        // Completeness (40 points)
        $completenessPercent = ($fieldsFilledCount / 12) * 100;
        $score += (int)($completenessPercent * 0.4);

        // Has service selected (20 points)
        if (!empty($data['service_id'])) $score += 20;

        return min(100, (int)round($score));
    }

    /**
     * Resend OTP for help form registration
     */
    public function resendOtp(Request $request): JsonResponse
    {
        // Get the data from session (stored during submit)
        $formData = session('help_form_data');
        $email = session('help_form_email');
        $phoneNumber = session('help_form_phone');

        if (!$email || !$phoneNumber || !$formData) {
            return response()->json([
                'success' => false,
                'message' => 'Session expired. Please submit the form again.'
            ], 400);
        }

        // Get OTP settings (same as Join Us)
        $settings = \Modules\GlobalSetting\Entities\GlobalSetting::whereIn('key', ['otp_digit_limit', 'otp_expire_time', 'otp_type'])
            ->pluck('value', 'key');

        // Generate OTP using UserRepository (EXACT SAME as Join Us form)
        $userRepository = app(\App\Repositories\Contracts\UserInterface::class);
        $otpData = $userRepository->generateOtp($email, (int) $settings['otp_digit_limit']);
        
        // Get OTP template (same as Join Us)
        $notificationType = 2;
        $type = $settings['otp_type'] === 'email' ? 1 : 2;
        $template = \Modules\GlobalSetting\app\Models\Templates::select('subject', 'content')
            ->where('type', $type)
            ->where('notification_type', $notificationType)
            ->first();

        \Log::info("Help Form OTP resent", ['email' => $email, 'phone' => $phoneNumber]);

        // Return EXACT SAME format as Join Us resend
        return response()->json([
            'success' => true,
            'name' => $formData['name'],
            'phone_number' => $phoneNumber,
            'otp' => $otpData['otp'],
            'otp_expire_time' => $settings['otp_expire_time'],
            'email_subject' => $template->subject ?? '',
            'email_content' => $template->content ?? '',
        ]);
    }
}




