<?php

namespace Modules\Leads\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Modules\Leads\app\Models\HelpFormQuestion;
use Modules\Leads\app\Models\HelpFormAnswer;
use Modules\Leads\app\Models\UserHelpFormResponse;
use Modules\Leads\app\Models\UserFormInput;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class HelpFormQuestionsController extends Controller
{
    /**
     * Display the questions management page
     */
    public function index()
    {
        return view('leads::admin.help-form-questions.index');
    }

    /**
     * Get all questions with answers
     */
    public function list(Request $request): JsonResponse
    {
        try {
            $questions = HelpFormQuestion::with(['answers' => function($query) {
                $query->where('is_active', true)->orderBy('order');
            }])
            ->when($request->search, function($query, $search) {
                $query->where('question_text', 'like', "%{$search}%");
            })
            ->when($request->filled('status'), function($query) use ($request) {
                $query->where('is_active', $request->status);
            })
            ->orderBy('order')
            ->get();

            return response()->json([
                'success' => true,
                'data' => $questions
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading questions'
            ], 500);
        }
    }

    /**
     * Get a single question
     */
    public function show($id): JsonResponse
    {
        try {
            $question = HelpFormQuestion::with('answers')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $question
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Question not found'
            ], 404);
        }
    }

    /**
     * Store a new question
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'question_text' => 'required|string|max:500',
            'question_type' => 'required|in:radio,checkbox,select,text',
            'is_required' => 'boolean',
            'answers' => 'required_if:question_type,radio,checkbox,select|array|min:1',
            'answers.*.text' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $maxOrder = HelpFormQuestion::max('order') ?? 0;

            $question = HelpFormQuestion::create([
                'question_text' => $request->question_text,
                'question_type' => $request->question_type,
                'is_required' => $request->is_required ?? true,
                'is_active' => true,
                'order' => $maxOrder + 1,
            ]);

            // Create answers if provided
            if ($request->has('answers') && is_array($request->answers)) {
                foreach ($request->answers as $index => $answer) {
                    HelpFormAnswer::create([
                        'question_id' => $question->id,
                        'answer_text' => $answer['text'],
                        'order' => $index + 1,
                        'is_active' => true,
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Question created successfully',
                'data' => $question->load('answers')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error creating question'
            ], 500);
        }
    }

    /**
     * Update a question
     */
    public function update(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'question_text' => 'required|string|max:500',
            'question_type' => 'required|in:radio,checkbox,select,text',
            'is_required' => 'boolean',
            'answers' => 'required_if:question_type,radio,checkbox,select|array|min:1',
            'answers.*.text' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $question = HelpFormQuestion::findOrFail($id);
            
            $question->update([
                'question_text' => $request->question_text,
                'question_type' => $request->question_type,
                'is_required' => $request->is_required ?? true,
            ]);

            // Delete old answers and create new ones
            $question->answers()->delete();
            
            if ($request->has('answers') && is_array($request->answers)) {
                foreach ($request->answers as $index => $answer) {
                    HelpFormAnswer::create([
                        'question_id' => $question->id,
                        'answer_text' => $answer['text'],
                        'order' => $index + 1,
                        'is_active' => true,
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Question updated successfully',
                'data' => $question->load('answers')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating question'
            ], 500);
        }
    }

    /**
     * Delete a question
     */
    public function destroy($id): JsonResponse
    {
        try {
            $question = HelpFormQuestion::findOrFail($id);
            $question->delete();

            return response()->json([
                'success' => true,
                'message' => 'Question deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting question'
            ], 500);
        }
    }

    /**
     * Toggle question active status
     */
    public function toggleStatus($id): JsonResponse
    {
        try {
            $question = HelpFormQuestion::findOrFail($id);
            $question->update(['is_active' => !$question->is_active]);

            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully',
                'is_active' => $question->is_active
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating status'
            ], 500);
        }
    }

    /**
     * Update question order
     */
    public function updateOrder(Request $request): JsonResponse
    {
        try {
            foreach ($request->questions as $index => $questionId) {
                HelpFormQuestion::where('id', $questionId)
                    ->update(['order' => $index + 1]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Order updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating order'
            ], 500);
        }
    }

    /**
     * Get active questions for frontend
     */
    public function getActiveQuestions(): JsonResponse
    {
        try {
            $questions = HelpFormQuestion::with(['answers' => function($query) {
                $query->where('is_active', true)->orderBy('order');
            }])
            ->where('is_active', true)
            ->orderBy('order')
            ->get();

            return response()->json([
                'success' => true,
                'questions' => $questions
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading questions'
            ], 500);
        }
    }

    /**
     * Get question responses for a specific lead
     */
    public function getLeadResponses($leadId): JsonResponse
    {
        try {
            $responses = UserHelpFormResponse::with(['question', 'answer'])
                ->where('lead_id', $leadId)
                ->get()
                ->groupBy('question_id')
                ->map(function($group) {
                    $first = $group->first();
                    
                    // If multiple responses (checkbox), combine them
                    if ($group->count() > 1) {
                        $answers = $group->map(function($r) {
                            return $r->answer ? $r->answer->answer_text : '';
                        })->filter()->implode(', ');
                        
                        return [
                            'question' => $first->question,
                            'answer_text' => $answers,
                            'answer' => null
                        ];
                    }
                    
                    return $first;
                })
                ->values();

            return response()->json([
                'success' => true,
                'responses' => $responses
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading responses'
            ], 500);
        }
    }
}

