<?php

namespace Modules\Communication\app\Repositories\Eloquent;

use Modules\Communication\app\Repositories\Contracts\SmsInterface;
use Modules\Communication\app\Models\CommunicationSettings;
use Illuminate\Support\Facades\Http;
use Vonage\Client;
use Vonage\Client\Credentials\Basic;
use Vonage\SMS\Message\SMS;

class SmsRepository implements SmsInterface
{
    public function sendSms(array $data): array
    {
        \Illuminate\Support\Facades\Log::info('=== sendSms called ===', [
            'has_otp' => !empty($data['otp']),
            'otp_value' => $data['otp'] ?? 'MISSING',
            'to_number' => $data['to_number'] ?? 'MISSING',
            'content' => substr($data['content'] ?? '', 0, 50)
        ]);
        
        $settings = $this->getActiveSmsSettings();
        
        if (empty($settings)) {
            \Illuminate\Support\Facades\Log::error('No active SMS settings found');
            return ['status' => 'error', 'message' => __('No active SMS settings found.'), 'code' => 400];
        }

        \Illuminate\Support\Facades\Log::info('SMS Provider detected', [
            'provider' => $settings['provider'],
            'whatsapp_enabled' => $settings['config']['twilio_whatsapp_enabled'] ?? 'NOT SET'
        ]);

        if ($settings['provider'] === 'nexmo') {
            return $this->sendViaNexmo($settings, $data);
        }

        if ($settings['provider'] === 'twilio') {
            return $this->sendViaTwilio($settings, $data);
        }

        return ['status' => 'error', 'message' => __('Unsupported SMS provider.'), 'code' => 400];
    }

    public function getActiveSmsSettings(): array
    {
        $settings = CommunicationSettings::where('settings_type', 2)
            ->where(function($query) {
                $query->where('key', 'nexmo_status')
                      ->orWhere('key', 'twilio_status');
            })
            ->where('value', 1)
            ->first();

        if (!$settings) {
            return [];
        }

        $provider = str_replace('_status', '', $settings->key);
        $config = CommunicationSettings::where('settings_type', 2)
            ->where('type', $provider)
            ->pluck('value', 'key')
            ->toArray();

        return [
            'provider' => $provider,
            'config' => $config
        ];
    }

    protected function sendViaNexmo(array $settings, array $data): array
    {
        try {
            $basic = new Basic($settings['config']['nexmo_api_key'], $settings['config']['nexmo_secret_key']);
            $client = new Client($basic);

            $response = $client->sms()->send(
                new SMS($data['to_number'], $settings['config']['nexmo_sender_id'], $data['content'])
            );

            $message = $response->current();

            if ($message->getStatus() == 0) {
                return ['code' => 200, 'message' => __('Message sent successfully.'), 'data' => []];
            }

            return ['status' => 'error', 'message' => $message->getStatus(), 'code' => 500];
        } catch (\Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    protected function sendViaTwilio(array $settings, array $data): array
    {
        // Check if WhatsApp is enabled for Twilio
        $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
        
        if ($useWhatsApp) {
            return $this->sendViaTwilioWhatsApp($settings, $data);
        }
        
        try {
            \Illuminate\Support\Facades\Log::info('Sending SMS via Twilio', [
                'to_number' => $data['to_number'],
                'from' => $settings['config']['twilio_sender_id'] ?? 'NOT SET',
                'has_api_key' => !empty($settings['config']['twilio_api_key']),
                'has_secret' => !empty($settings['config']['twilio_secret_key'])
            ]);

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'], 
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', [
                'From' => $settings['config']['twilio_sender_id'],
                'To' => $data['to_number'],
                'Body' => strip_tags($data['content']),
            ]);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Twilio SMS sent successfully');
                return ['code' => 200, 'message' => __('Message sent successfully.'), 'data' => []];
            }

            \Illuminate\Support\Facades\Log::error('Twilio SMS failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => 500];
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Twilio SMS exception', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    protected function sendViaTwilioWhatsApp(array $settings, array $data): array
    {
        try {
            \Illuminate\Support\Facades\Log::info('Sending WhatsApp message via Twilio', [
                'to_number' => $data['to_number'],
                'from' => $settings['config']['twilio_whatsapp_number'] ?? 'NOT SET',
                'has_api_key' => !empty($settings['config']['twilio_api_key']),
                'has_secret' => !empty($settings['config']['twilio_secret_key']),
                'has_template' => !empty($settings['config']['twilio_whatsapp_template_otp'])
            ]);

            // Format phone numbers for WhatsApp
            $fromNumber = $settings['config']['twilio_whatsapp_number'] ?? $settings['config']['twilio_sender_id'];
            // Ensure the from number has the whatsapp: prefix
            if (strpos($fromNumber, 'whatsapp:') !== 0) {
                $fromNumber = 'whatsapp:' . $fromNumber;
            }
            
            $toNumber = $data['to_number'];
            // Ensure the to number has the whatsapp: prefix
            if (strpos($toNumber, 'whatsapp:') !== 0) {
                $toNumber = 'whatsapp:' . $toNumber;
            }

            // Check if we have a WhatsApp template configured
            $useTemplate = !empty($settings['config']['twilio_whatsapp_template_otp']);
            
            if ($useTemplate && !empty($data['otp'])) {
                // Send using WhatsApp template (for OTP messages)
                return $this->sendWhatsAppWithTemplate($settings, $fromNumber, $toNumber, $data);
            }
            
            // Fallback: Send plain WhatsApp message (for non-OTP messages)
            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'], 
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', [
                'From' => $fromNumber,
                'To' => $toNumber,
                'Body' => strip_tags($data['content']),
            ]);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Twilio WhatsApp message sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => __('WhatsApp message sent successfully.'), 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Twilio WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Twilio WhatsApp exception', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    protected function sendWhatsAppWithTemplate(array $settings, string $fromNumber, string $toNumber, array $data): array
    {
        try {
            $contentSid = $settings['config']['twilio_whatsapp_template_sid'] ?? null;
            $otp = $data['otp'] ?? '';
            
            \Illuminate\Support\Facades\Log::info('Sending WhatsApp with Content Template', [
                'content_sid' => $contentSid,
                'otp' => $otp,
                'from' => $fromNumber,
                'to' => $toNumber
            ]);

            // Use Twilio Content API with the approved template SID
            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::error('WhatsApp Content SID not configured');
                return ['status' => 'error', 'message' => 'WhatsApp template not configured', 'code' => 400];
            }
            
            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $otp // {{1}} in template = OTP code
                ])
            ];
            
            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'], 
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('WhatsApp template message sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => __('WhatsApp OTP sent successfully.'), 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('WhatsApp template message failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];
            
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('WhatsApp template exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send provider welcome WhatsApp message
     */
    public function sendProviderWelcomeWhatsApp(string $toNumber, string $providerName, string $categoryName = 'Provider'): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            // Check if WhatsApp is enabled
            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping provider welcome WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_welcome_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Provider welcome WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending provider welcome WhatsApp', [
                'to' => $toNumber,
                'provider_name' => $providerName,
                'category_name' => $categoryName,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $providerName,  // {{1}} = Provider Name
                    '2' => $categoryName   // {{2}} = Category Name (e.g., "Counsellor")
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Provider welcome WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Provider welcome WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Provider welcome WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send provider verified WhatsApp message
     */
    public function sendProviderVerifiedWhatsApp(string $toNumber, string $providerName, string $categoryName, string $loginUrl): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            // Check if WhatsApp is enabled
            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping provider verified WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_verified_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Provider verified WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending provider verified WhatsApp', [
                'to' => $toNumber,
                'provider_name' => $providerName,
                'category_name' => $categoryName,
                'login_url' => $loginUrl,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $providerName,  // {{1}} = Provider Name
                    '2' => $categoryName   // {{2}} = Category Name (e.g., "Counsellor")
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Provider verified WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Provider verified WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Provider verified WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send admin notification WhatsApp for new provider signup
     */
    public function sendAdminProviderSignupWhatsApp(string $categoryName, string $providerName, string $providerEmail, string $providerPhone): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            // Check if WhatsApp is enabled
            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping admin notification WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            // Get admin WhatsApp number
            $adminNumber = $settings['config']['admin_whatsapp_number'] ?? null;
            if (!$adminNumber) {
                \Illuminate\Support\Facades\Log::warning('Admin WhatsApp number not configured');
                return ['status' => 'error', 'message' => 'Admin WhatsApp number not configured', 'code' => 400];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $adminNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_admin_provider_signup_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Admin provider signup WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending admin provider signup WhatsApp', [
                'to' => $toNumber,
                'category_name' => $categoryName,
                'provider_name' => $providerName,
                'provider_email' => $providerEmail,
                'provider_phone' => $providerPhone,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $categoryName,   // {{1}} = Category Name (e.g., "Counsellor")
                    '2' => $providerName,   // {{2}} = Provider Name
                    '3' => $providerEmail,  // {{3}} = Provider Email
                    '4' => $providerPhone   // {{4}} = Provider Phone
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Admin provider signup WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Admin provider signup WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Admin provider signup WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send user welcome WhatsApp message
     */
    public function sendUserWelcomeWhatsApp(string $toNumber, string $firstName, string $lastName, string $companyName): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            // Check if WhatsApp is enabled
            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping user welcome WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_user_welcome_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('User welcome WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            $userName = $firstName . ' ' . $lastName;

            \Illuminate\Support\Facades\Log::info('Sending user welcome WhatsApp', [
                'to' => $toNumber,
                'user_name' => $userName,
                'company_name' => $companyName,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $userName,      // {{1}} = First Name + Last Name
                    '2' => $companyName    // {{2}} = Company Name
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('User welcome WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('User welcome WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('User welcome WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send admin notification WhatsApp for new user signup
     */
    public function sendAdminUserSignupWhatsApp(string $firstName, string $lastName, string $userEmail, string $userPhone): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            // Check if WhatsApp is enabled
            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping admin notification WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            // Get admin WhatsApp number
            $adminNumber = $settings['config']['admin_whatsapp_number'] ?? null;
            if (!$adminNumber) {
                \Illuminate\Support\Facades\Log::warning('Admin WhatsApp number not configured');
                return ['status' => 'error', 'message' => 'Admin WhatsApp number not configured', 'code' => 400];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $adminNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_admin_user_signup_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Admin user signup WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            $userName = $firstName . ' ' . $lastName;

            \Illuminate\Support\Facades\Log::info('Sending admin user signup WhatsApp', [
                'to' => $toNumber,
                'user_name' => $userName,
                'user_email' => $userEmail,
                'user_phone' => $userPhone,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $userName,    // {{1}} = First Name + Last Name
                    '2' => $userEmail,   // {{2}} = User Email
                    '3' => $userPhone    // {{3}} = User Phone
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Admin user signup WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Admin user signup WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Admin user signup WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send wallet top-up receipt WhatsApp
     */
    public function sendWalletTopUpWhatsApp(string $toNumber, string $userName, string $amountPaid, string $creditsAdded): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping wallet top-up WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_wallet_topup_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Wallet top-up WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending wallet top-up WhatsApp', [
                'to' => $toNumber,
                'user_name' => $userName,
                'amount_paid' => $amountPaid,
                'credits_added' => $creditsAdded,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $userName,        // {{1}} = Name
                    '2' => $amountPaid,      // {{2}} = Amount Paid
                    '3' => $creditsAdded     // {{3}} = Credits Added
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Wallet top-up WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Wallet top-up WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Wallet top-up WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send Paradigm invitation WhatsApp to client
     * Template: userpurchaseassessment (WhatsApp Card)
     * {{1}} = Client Name
     * {{2}} = Assessment Link
     */
    public function sendParadigmInvitationClientWhatsApp(string $toNumber, string $clientName, ?string $assessmentLink = null): array
    {
        \Illuminate\Support\Facades\Log::info('sendParadigmInvitationClientWhatsApp CALLED', [
            'to' => $toNumber,
            'client_name' => $clientName,
            'assessment_link' => $assessmentLink
        ]);
        
        try {
            $settings = $this->getActiveSmsSettings();
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping paradigm invitation client WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_paradigm_invitation_client_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Paradigm invitation client WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            // Build content variables - Template: {{1}} = Client Name, {{2}} = Assessment Link
            $contentVariables = [
                '1' => $clientName  // {{1}} = Client Name
            ];
            
            // Add assessment link as variable {{2}} if provided
            if ($assessmentLink) {
                $contentVariables['2'] = $assessmentLink;  // {{2}} = Assessment Link
            }

            \Illuminate\Support\Facades\Log::info('Sending paradigm invitation client WhatsApp', [
                'to' => $toNumber,
                'client_name' => $clientName,
                'assessment_link' => $assessmentLink,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode($contentVariables)
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Paradigm invitation client WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Paradigm invitation client WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Paradigm invitation client WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send profile completed WhatsApp to client
     */
    public function sendProfileCompletedClientWhatsApp(string $toNumber, string $clientName, ?string $assessmentLink = null): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping profile completed WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_profile_completed_client_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Profile completed client WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            // Build content variables - include link if provided
            $contentVariables = [
                '1' => $clientName  // {{1}} = Client Name
            ];
            
            // Add assessment link as variable {{2}} if provided
            if ($assessmentLink) {
                $contentVariables['2'] = $assessmentLink;  // {{2}} = Assessment Link
            }

            \Illuminate\Support\Facades\Log::info('Sending profile completed client WhatsApp', [
                'to' => $toNumber,
                'client_name' => $clientName,
                'assessment_link' => $assessmentLink,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode($contentVariables)
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Profile completed client WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Profile completed client WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Profile completed client WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send profile completed WhatsApp to provider
     */
    public function sendProfileCompletedProviderWhatsApp(string $toNumber, string $providerName, string $clientName): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping profile completed provider WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_profile_completed_provider_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Profile completed provider WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending profile completed provider WhatsApp', [
                'to' => $toNumber,
                'provider_name' => $providerName,
                'client_name' => $clientName,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $providerName,  // {{1}} = Provider Name
                    '2' => $clientName     // {{2}} = Client Name
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Profile completed provider WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Profile completed provider WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Profile completed provider WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send admin notification when provider buys credits
     */
    public function sendAdminProviderCreditsWhatsApp(string $toNumber, string $providerName, string $categoryName, string $amountPaid, string $creditsAdded): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping admin provider credits WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_admin_provider_credits_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Admin provider credits WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending admin provider credits WhatsApp', [
                'to' => $toNumber,
                'provider_name' => $providerName,
                'category' => $categoryName,
                'amount_paid' => $amountPaid,
                'credits_added' => $creditsAdded,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $providerName,      // {{1}} = Provider Name
                    '2' => $categoryName,      // {{2}} = Category
                    '3' => $amountPaid,        // {{3}} = Amount Paid
                    '4' => $creditsAdded       // {{4}} = Credits Added
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Admin provider credits WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Admin provider credits WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Admin provider credits WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Send admin notification when user buys credits
     */
    public function sendAdminUserCreditsWhatsApp(string $toNumber, string $userName, string $amountPaid, string $creditsAdded): array
    {
        try {
            $settings = $this->getActiveSmsSettings();
            if (empty($settings) || $settings['provider'] !== 'twilio') {
                \Illuminate\Support\Facades\Log::warning('Twilio not configured for WhatsApp');
                return ['status' => 'error', 'message' => 'Twilio not configured', 'code' => 400];
            }

            $useWhatsApp = !empty($settings['config']['twilio_whatsapp_enabled']) && $settings['config']['twilio_whatsapp_enabled'] == '1';
            if (!$useWhatsApp) {
                \Illuminate\Support\Facades\Log::info('WhatsApp not enabled, skipping admin user credits WhatsApp');
                return ['status' => 'skipped', 'message' => 'WhatsApp not enabled', 'code' => 200];
            }

            $fromNumber = 'whatsapp:' . $settings['config']['twilio_whatsapp_number'];
            $toNumber = 'whatsapp:' . $toNumber;
            $contentSid = $settings['config']['twilio_whatsapp_template_admin_user_credits_sid'] ?? null;

            if (!$contentSid) {
                \Illuminate\Support\Facades\Log::warning('Admin user credits WhatsApp template SID not configured');
                return ['status' => 'error', 'message' => 'Template not configured', 'code' => 400];
            }

            \Illuminate\Support\Facades\Log::info('Sending admin user credits WhatsApp', [
                'to' => $toNumber,
                'user_name' => $userName,
                'amount_paid' => $amountPaid,
                'credits_added' => $creditsAdded,
                'content_sid' => $contentSid
            ]);

            $payload = [
                'From' => $fromNumber,
                'To' => $toNumber,
                'ContentSid' => $contentSid,
                'ContentVariables' => json_encode([
                    '1' => $userName,          // {{1}} = User Name
                    '2' => $amountPaid,        // {{2}} = Amount Paid
                    '3' => $creditsAdded       // {{3}} = Credits Added
                ])
            ];

            $response = Http::withBasicAuth(
                $settings['config']['twilio_api_key'],
                $settings['config']['twilio_secret_key']
            )
            ->asForm()
            ->post('https://api.twilio.com/2010-04-01/Accounts/' . $settings['config']['twilio_api_key'] . '/Messages.json', $payload);

            if ($response->successful()) {
                \Illuminate\Support\Facades\Log::info('Admin user credits WhatsApp sent successfully', [
                    'response' => $response->json()
                ]);
                return ['code' => 200, 'message' => 'WhatsApp sent successfully', 'data' => $response->json()];
            }

            \Illuminate\Support\Facades\Log::error('Admin user credits WhatsApp failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return ['status' => 'error', 'message' => $response->body(), 'code' => $response->status()];

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Admin user credits WhatsApp exception', [
                'message' => $e->getMessage()
            ]);
            return ['status' => 'error', 'message' => $e->getMessage(), 'code' => 500];
        }
    }
}