<?php

namespace Modules\Communication\app\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\PackageTrx;
use App\Models\User;
use Illuminate\Http\Request;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Modules\GlobalSetting\app\Models\Templates;
use Illuminate\Support\Facades\Hash;
use App\Models\ProviderDetail;
use App\Models\UserDetail;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Modules\Communication\app\Models\OtpSetting;
use Modules\GlobalSetting\app\Models\SubscriptionPackage;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Modules\Communication\app\Helpers\MailConfigurator;
use Illuminate\Support\Facades\Mail;
use Modules\Communication\app\Mail\Samplemail;

class CommunicationController extends Controller
{
    public function getOtpSettings(Request $request): JsonResponse
    {
        $email = $request->input('email');


        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return response()->json(['error' => 'Invalid email address'], 400);
        }

        $user = User::where('email', $email)->first();
        $type = $request->input('type');

        $providerApprovalStatus = providerApprovalStatus();
        if ($user && $user->user_type == 2 && $user->provider_verified_status == 0 && $providerApprovalStatus == 1) {
            return response()->json([
                'code'    => 200,
                'message' => __('provider_not_verified_info'),
                'provider_verified_status' => 0,
            ], 200);
        }

        if (!$user || ($type === 'forgot' && ($email === 'demouser@gmail.com' || $email === 'demoprovider@gmail.com'))) {
            return response()->json(['error' => 'The given email is not registered.'], 400);
        }

        $settings = GlobalSetting::whereIn('key', ['otp_digit_limit', 'otp_expire_time', 'otp_type'])
            ->pluck('value', 'key');

        if (!in_array($settings['otp_type'], ['email', 'sms'])) {
            return response()->json(['error' => 'Unsupported OTP type'], 400);
        }

        if ($email === 'demouser@gmail.com') {
            $otp = '1234';
        } elseif ($email === 'demoprovider@gmail.com') {
            $otp = '1234';
        } else {
            $otp = $this->generateOtp($settings['otp_digit_limit']);
        }

        $otpExpireMinutes = (int) filter_var($settings['otp_expire_time'], FILTER_SANITIZE_NUMBER_INT);
        $expiresAt = now()
            ->addMinutes($otpExpireMinutes)
            ->format('Y-m-d H:i:s');

        $existingOtp = DB::table('otp_settings')->where('email', $email)->first();


        if ($existingOtp) {
            DB::table('otp_settings')
                ->where('email', $email)
                ->update([
                    'otp' => $otp,
                    'expires_at' => $expiresAt,
                ]);
        } else {
            DB::table('otp_settings')->insert([
                'email' => $email,
                'otp' => $otp,
                'expires_at' => $expiresAt,
            ]);
        }

        $subject = null;
        $content = null;

        if ($settings['otp_type'] === 'email') {
            // Retrieve email template
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'Email template not found'], 404);
            }

            //for email
            $subject = $template->subject;
            $content = str_replace(
                ['{{user_name}}', '{{otp}}'],
                [$user->name, $otp],
                $template->content
            );
        } elseif ($settings['otp_type'] === 'sms') {

            // Retrieve sms template
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 2)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'SMS template not found'], 404);
            }

            //for SMS
            $subject = $template->subject;
            $content = str_replace(
                ['{{user_name}}', '{{otp}}'],
                [$user->name, $otp],
                $template->content
            );
        }

        return response()->json([
            'name' => $user->name,
            'phone_number' => $user->phone_number,
            'otp_digit_limit' => $settings['otp_digit_limit'],
            'otp_expire_time' => $settings['otp_expire_time'],
            'otp_type' => $settings['otp_type'],
            'otp' => $otp,
            'expires_at' => $expiresAt,
            'email_subject' => $subject,
            'email_content' => $content,
        ]);
    }

    public function getOtpSettingsApi(Request $request): JsonResponse
    {
        $email = $request->input('email');


        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return response()->json(['error' => 'Invalid email address'], 400);
        }

        $user = User::where('email', $email)->first();
        $type = $request->input('type');

        $providerApprovalStatus = providerApprovalStatus();
        if ($user && $user->user_type == 2 && $user->provider_verified_status == 0 && $providerApprovalStatus == 1) {
            return response()->json([
                'code'    => 200,
                'message' => __('provider_not_verified_info'),
                'provider_verified_status' => 0,
            ], 200);
        }

        if (!$user || ($type === 'forgot' && ($email === 'demouser@gmail.com' || $email === 'demoprovider@gmail.com'))) {
            return response()->json(['error' => 'The given email is not registered.'], 400);
        }

        $settings = GlobalSetting::whereIn('key', ['otp_digit_limit', 'otp_expire_time', 'otp_type'])
            ->pluck('value', 'key');

        if (!in_array($settings['otp_type'], ['email', 'sms'])) {
            return response()->json(['error' => 'Unsupported OTP type'], 400);
        }

        if ($email === 'demouser@gmail.com') {
            $otp = '1234';
        } elseif ($email === 'demoprovider@gmail.com') {
            $otp = '1234';
        } else {
            $otp = $this->generateOtp($settings['otp_digit_limit']);
        }

        $otpExpireMinutes = (int) filter_var($settings['otp_expire_time'], FILTER_SANITIZE_NUMBER_INT);
        $expiresAt = now()
            ->addMinutes($otpExpireMinutes)
            ->format('Y-m-d H:i:s');

        $existingOtp = DB::table('otp_settings')->where('email', $email)->first();


        if ($existingOtp) {
            DB::table('otp_settings')
                ->where('email', $email)
                ->update([
                    'otp' => $otp,
                    'expires_at' => $expiresAt,
                ]);
        } else {
            DB::table('otp_settings')->insert([
                'email' => $email,
                'otp' => $otp,
                'expires_at' => $expiresAt,
            ]);
        }

        $subject = null;
        $content = null;

        if ($settings['otp_type'] === 'email') {
            // Retrieve email template
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'Email template not found'], 404);
            }

            //for email
            $subject = $template->subject;
            $content = str_replace(
                ['{{user_name}}', '{{otp}}'],
                [$user->name, $otp],
                $template->content
            );
        } elseif ($settings['otp_type'] === 'sms') {

            // Retrieve sms template
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 2)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'SMS template not found'], 404);
            }

            //for SMS
            $subject = $template->subject;
            $content = str_replace(
                ['{{user_name}}', '{{otp}}'],
                [$user->name, $otp],
                $template->content
            );
        }

        MailConfigurator::configureMail();

        $tomail = $request->input('email');
        $subject = $subject;
        $content = $content;
        $attachment = $request->input('attachment');

        $data = [
            'message' => $content,
            'subject' => $subject,
            'attachment' => $attachment,
        ];

        // Check if the email address is provided
        if (empty($tomail)) {
            return response()->json([
                'code' => 400,
                'message' => 'Recipient email is required.',
            ], 400);
        }

        if (is_array($tomail)) {
            foreach ($tomail as $email) {
                Mail::to($email)->send(new Samplemail($data));
            }
        } else {
            Mail::to($tomail)->send(new Samplemail($data));
        }

        $data = [
            'name' => $user->name,
            'phone_number' => $user->phone_number,
            'otp_digit_limit' => $settings['otp_digit_limit'],
            'otp_expire_time' => $settings['otp_expire_time'],
            'otp_type' => $settings['otp_type'],
            'otp' => $otp,
            'expires_at' => $expiresAt,
            'email_subject' => $subject,
            'email_content' => $content,
        ];

        return response()->json(['code' => 200, 'message' => __('Otp and Email sent successfully.'), 'data' => $data], 200);
    }

    public function getRegisterOtpSettings(Request $request): JsonResponse
    {

        $email = $request->email;

        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return response()->json(['error' => 'Invalid email address'], 400);
        }

        $settings = GlobalSetting::whereIn('key', ['otp_digit_limit', 'otp_expire_time', 'otp_type'])
            ->pluck('value', 'key');

        if (!in_array($settings['otp_type'], ['email', 'sms', 'whatsapp'])) {
            return response()->json(['error' => 'Unsupported OTP type'], 400);
        }

        $otp = $this->generateOtp($settings['otp_digit_limit']);

        $otpExpireMinutes = (int) filter_var($settings['otp_expire_time'], FILTER_SANITIZE_NUMBER_INT);
        $expiresAt = now()
            ->addMinutes($otpExpireMinutes)
            ->format('Y-m-d H:i:s');

        $existingOtp = DB::table('otp_settings')->where('email', $email)->first();

        if ($existingOtp) {
            DB::table('otp_settings')
                ->where('email', $email)
                ->update([
                    'otp' => $otp,
                    'expires_at' => $expiresAt,
                ]);
        } else {
            DB::table('otp_settings')->insert([
                'email' => $email,
                'otp' => $otp,
                'expires_at' => $expiresAt,
            ]);
        }

        $subject = null;
        $content = null;

        if ($settings['otp_type'] === 'email') {
            // Retrieve email template
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'Email template not found'], 404);
            }

            //for email
            $subject = $template->subject;
            $content = str_replace(
                ['{{user_name}}', '{{otp}}'],
                [$request->name, $otp],
                $template->content
            );
        } elseif ($settings['otp_type'] === 'sms' || $settings['otp_type'] === 'whatsapp') {

            // Retrieve sms/whatsapp template (same template for both - type 2)
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 2)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'SMS/WhatsApp template not found'], 404);
            }

            //for SMS/WhatsApp - send template with placeholders, replacement happens in sendOtpSms
            $subject = $template->subject;
            $content = $template->content;  // Don't replace yet - send template as-is
        }

        return response()->json([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'name' => $request->name,
            'phone_number' => $request->phone_number,
            'email' => $request->email,
            'password' => $request->password,
            'otp_digit_limit' => $settings['otp_digit_limit'],
            'otp_expire_time' => $settings['otp_expire_time'],
            'otp_type' => $settings['otp_type'],
            'otp' => $otp,
            'expires_at' => $expiresAt,
            'email_subject' => $subject,
            'email_content' => $content,
        ]);
    }

    public function getProviderRegisterOtpSettings(Request $request): JsonResponse
    {
        $email = $request->email;

        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return response()->json(['error' => 'Invalid email address'], 400);
        }

        $settings = GlobalSetting::whereIn('key', ['otp_digit_limit', 'otp_expire_time', 'otp_type'])
            ->pluck('value', 'key');

        // Use the requested OTP type from frontend, fallback to global setting
        $otpType = $settings['otp_type'];
        
        if (!in_array($otpType, ['email', 'sms', 'whatsapp'])) {
            return response()->json(['error' => 'Unsupported OTP type'], 400);
        }

        $otp = $this->generateOtp($settings['otp_digit_limit']);

        $otpExpireMinutes = (int) filter_var($settings['otp_expire_time'], FILTER_SANITIZE_NUMBER_INT);
        $expiresAt = now()
            ->addMinutes($otpExpireMinutes)
            ->format('Y-m-d H:i:s');

        $existingOtp = DB::table('otp_settings')->where('email', $email)->first();

        if ($existingOtp) {
            DB::table('otp_settings')
                ->where('email', $email)
                ->update([
                    'otp' => $otp,
                    'expires_at' => $expiresAt,
                ]);
        } else {
            DB::table('otp_settings')->insert([
                'email' => $email,
                'otp' => $otp,
                'expires_at' => $expiresAt,
            ]);
        }

        $subject = null;
        $content = null;

        // Always return BOTH email and SMS templates so frontend can choose
        if ($otpType === 'email') {
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'Email template not found'], 404);
            }

            $subject = $template->subject;
            $content = str_replace(
                ['{{user_name}}', '{{otp}}'],
                [$request->provider_name, $otp],
                $template->content
            );
        } elseif ($otpType === 'sms' || $otpType === 'whatsapp') {

            // Retrieve sms template (same template used for WhatsApp)
            $notificationType = 2;
            $template = Templates::select('subject', 'content')
                ->where('type', 2)
                ->where('notification_type', $notificationType)
                ->first();

            if (!$template) {
                return response()->json(['error' => 'SMS/WhatsApp template not found'], 404);
            }

            //for SMS/WhatsApp - send template with placeholders, replacement happens in sendOtpSms
            $subject = $template->subject;
            $content = $template->content;  // Don't replace yet - send template as-is
        }

        return response()->json([
            'provider_first_name' => $request->provider_first_name,
            'provider_last_name' => $request->provider_last_name,
            'name' => $request->name,
            'phone_number' => $request->phone_number,
            'email' => $request->email,
            'password' => $request->password,
            'category_id' => $request->category_id,
            'subcategory_ids' => $request->subcategory_ids,
            'company_name' => $request->company_name,
            'company_website' => $request->company_website,
            'otp_digit_limit' => $settings['otp_digit_limit'],
            'otp_expire_time' => $settings['otp_expire_time'],
            'otp_type' => $settings['otp_type'],
            'otp' => $otp,
            'expires_at' => $expiresAt,
            'email_subject' => $subject,
            'email_content' => $content,
        ]);
    }

    private function generateOtp(int $digitLimit): string
    {
        return str_pad((string) random_int(0, pow(10, $digitLimit) - 1), $digitLimit, '0', STR_PAD_LEFT);
    }

    /**
     * Add welcome credits to new provider
     */
    public function addWelcomeCredits(int $userId): void
    {
        try {
            // Use WalletRepository to add 25 free credits
            $walletRepository = app(\App\Repositories\Contracts\WalletInterface::class);
            
            $walletRepository->addWalletAmount([
                'user_id' => $userId,
                'amount' => 25.00,
                'payment_method' => 'Others', // Use existing enum value
                'status' => 'Completed',
                'transaction_id' => 'WELCOME-' . $userId . '-' . time(),
                'type' => 1, // Credit
            ]);
            
            \Log::info("Welcome credits added to provider", [
                'user_id' => $userId,
                'amount' => 25.00,
                'transaction_id' => 'WELCOME-' . $userId . '-' . time()
            ]);
            
        } catch (\Exception $e) {
            \Log::error("Failed to add welcome credits to provider", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
        }
    }

    public function verifyOtp(Request $request): JsonResponse
    {
        \Log::info('=== verifyOtp called ===', [
            'login_type' => $request->login_type,
            'email' => $request->email ?? 'N/A'
        ]);
        
        if ($request->login_type == "provider_register") {
            \Log::info('Provider registration OTP verification starting');

            $request->validate([
                'otp' => 'required',
            ]);

            $otpSetting = OtpSetting::where('email', $request->email)->first();

            $currentDateTime = now();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }
            $providerApprovalStatus = providerApprovalStatus();
            $data = [
                'name' => $request->name,
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'password' => Hash::make($request->password),
                'category_id' => $request->category_id,
                'user_type' => 2,
                'sub_service_type' => $request->sub_service_type ?? 'individual',
                'provider_verified_status' => $providerApprovalStatus == 1 ? 0 : 1,
                'dob' => $request->dob ?? $request->provider_dob,
                'gender' => $request->gender ?? $request->provider_gender,
                'address' => $request->address ?? $request->provider_address,
                'city' => $request->city ?? $request->provider_city,
                'state' => $request->state ?? $request->provider_state,
                'country' => $request->country ?? $request->provider_country,
                'postal_code' => $request->postal_code ?? $request->provider_postal_code,
                'bio' => $request->bio ?? $request->provider_bio,
                'language' => $request->language ?? $request->provider_language
            ];

            if (request()->has('sub_service_type') && !empty(request()->get('sub_service_type'))) {
                $data['sub_service_type'] = request()->get('sub_service_type');
            }

            $save = User::create($data);

            if (is_array($request->subcategory_ids) && count($request->subcategory_ids) > 0) {
                foreach ($request->subcategory_ids as $subcategoryId) {
                    ProviderDetail::create([
                        'user_id' => $save->id,
                        'category_id' => $request->category_id,
                        'subcategory_id' => $subcategoryId,
                    ]);
                }
            }

            $company_details = [
                'user_id' => $save->id,
                'category_id' => $request->category_id,
                'first_name' => $request->provider_first_name ?? $request->first_name,
                'last_name' => $request->provider_last_name ?? $request->last_name,
                'company_name' => $request->company_name,
                'company_website' => $request->company_website,
                'dob' => $request->dob ?? $request->provider_dob,
                'gender' => $request->gender ?? $request->provider_gender,
                'bio' => $request->bio ?? $request->provider_bio,
                'address' => $request->address ?? $request->provider_address,
                'city' => $request->city ?? $request->provider_city,
                'state' => $request->state ?? $request->provider_state,
                'country' => $request->country ?? $request->provider_country,
                'postal_code' => $request->postal_code ?? $request->provider_postal_code,
                'language' => $request->language ?? $request->provider_language,
                'profile_image' => $request->profile_image ?? null,
            ];

            $company = UserDetail::create($company_details);

            $feeSub = SubscriptionPackage::select('id', 'is_default', 'price', 'package_term', 'package_duration')
                ->where('is_default', 1)
                ->first();

            if ($feeSub) {
                $currentDate = Carbon::now();

                $trx_date = $currentDate->toDateString();

                $end_date = match ($feeSub->package_term) {
                    'day' => $currentDate->copy()->addDays($feeSub->package_duration)->toDateTimeString(),
                    'month' => $currentDate->copy()->addMonths($feeSub->package_duration)->toDateTimeString(),
                    'yearly' => $currentDate->copy()->addYears($feeSub->package_duration)->toDateTimeString(),
                    'lifetime' => '9999-12-31',
                    default => null,
                };

                $pacakge_trx = [
                    'provider_id' => $save->id,
                    'package_id' => $feeSub->id,
                    'transaction_id' => null,
                    'trx_date' => $trx_date,
                    'end_date' => $end_date,
                    'amount' => $feeSub->price,
                    'payment_status' => 2,
                    'created_by' => $save->id,
                ];

                $create = PackageTrx::create($pacakge_trx);
            }

            DB::table('otp_settings')->where('email', $request->email)->delete();

            $signupDate = formatDateTime($save->created_at);

            if ($providerApprovalStatus == 1) {
                // Send welcome email to provider
                $this->sendProviderWelcomeEmail($request, 2, $signupDate);
                // Send notification to admin
                $this->sendProviderSignupEmailToAdmin($request, 32, $signupDate);

                // Log in the user so they can access ID verification page
                Auth::login($save);
                session(['user_id' => $save->id]);
                Cache::forget('provider_auth_id');
                Cache::forever('provider_auth_id', $save->id);

                // Check if ID verification is enabled
                if (providerIdVerificationEnabled()) {
                    return response()->json([
                        'message' => 'OTP verified successfully',
                        'provider_approval_status' => $providerApprovalStatus,
                        'next_step' => 'id_verification'
                    ]);
                }

                return response()->json([
                    'message' => 'OTP verified successfully',
                    'provider_approval_status' => $providerApprovalStatus
                ]);
            } else {
                $this->sendProviderWelcomeEmail($request, 2, $signupDate);
                $this->sendProviderSignupEmailToAdmin($request, 32, $signupDate);

                Auth::login($save);

                session(['user_id' => $save->id]);
                Cache::forget('provider_auth_id');
                Cache::forever('provider_auth_id',  $save->id);

                // Check if ID verification is enabled
                if (providerIdVerificationEnabled()) {
                    return response()->json([
                        'message' => 'OTP verified successfully',
                        'provider_approval_status' => 0,
                        'next_step' => 'id_verification'
                    ]);
                }

                return response()->json([
                    'message' => 'OTP verified successfully',
                    'provider_approval_status' => 0
                ]);
            }
        } elseif ($request->login_type == "register") {
            $request->validate([
                'otp' => 'required',
            ]);

            $otpSetting = DB::table('otp_settings')->where('email', $request->email)->first();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $currentDateTime = now();
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }

            $data = [
                'name' => $request->name,
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'password' => Hash::make($request->password),
                'user_type' => 3,
            ];

            $save = User::create($data);

            $company_details = [
                'user_id' => $save->id,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'mobile_number' => $request->phone_number,
                'gender' => $request->gender,
                'dob' => $request->dob,
                'address' => $request->address,
                'country' => $request->country,
                'state' => $request->state,
                'city' => $request->city,
                'postal_code' => $request->postal_code,
            ];

            $company = UserDetail::create($company_details);

            Auth::login($save);

            session(['user_id' => $save->id]);
            Cache::forget('user_auth_id');
            Cache::forever('user_auth_id',  $save->id);
            DB::table('otp_settings')->where('email', $request->email)->delete();

            try {
                $this->sendUserWelcomeEmail($request);
            } catch (\Exception $e) {
                \Log::error('User welcome email/WhatsApp failed: ' . $e->getMessage());
                // Don't break the registration flow
            }

            return response()->json([
                'code' => 200,
                'message' => 'Registration successful! Redirecting to dashboard...',
                'user_id' => $save->id,
                'user_type' => $save->user_type
            ]);
        } elseif ($request->login_type == "forgot_email") {
            $request->validate([
                'forgot_email' => 'required|email',
                'otp' => 'required',
            ]);

            $user = User::where('email', $request->forgot_email)->first();

            if (!$user) {
                return response()->json(['error' => 'User not found'], 404);
            }

            $otpSetting = DB::table('otp_settings')->where('email', $request->forgot_email)->first();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $currentDateTime = now();
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }
            DB::table('otp_settings')->where('email', $request->forgot_email)->delete();

            $data = "done";

            return response()->json(['message' => 'OTP verified successfully', 'data' => $data, 'email' => $request->forgot_email]);
        } else {
            $request->validate([
                'email' => 'required|email',
                'otp' => 'required',
            ]);

            $user = User::where('email', $request->email)->first();

            if (!$user) {
                return response()->json(['error' => 'User not found'], 404);
            }

            $otpSetting = DB::table('otp_settings')->where('email', $request->email)->first();

            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $currentDateTime = now();
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }

            Auth::login($user);

            session(['user_id' => $user->id]);

            if ($user->user_type == '2') {
                Cache::forget('provider_auth_id');
                Cache::forever('provider_auth_id', $user->id);
            } else {
                Cache::forget('user_auth_id');
                Cache::forever('user_auth_id', $user->id);
            }

            $token = $user->createToken('user-token')->plainTextToken;

            DB::table('otp_settings')->where('email', $request->email)->delete();

            return response()->json([
                'message' => 'OTP verified successfully',
                'user_id' => $user->id,
                'user_type' => $user->user_type,
                'token' => $token
            ]);
        }

    }

    public function verifyOtpApi(Request $request): JsonResponse
    {
        if ($request->login_type == "provider_register") {

            $request->validate([
                'otp' => 'required',
            ]);

            $otpSetting = OtpSetting::where('email', $request->email)->first();

            $currentDateTime = now();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }
            
            $providerApprovalStatus = providerApprovalStatus();
            $providerVerifiedStatus = $providerApprovalStatus == 1 ? 0 : 1;
            $data = [
                'name' => $request->name,
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'password' => Hash::make($request->password),
                'user_type' => 2,
                'provider_verified_status' => $providerVerifiedStatus
            ];
            
            if (request()->has('sub_service_type') && !empty(request()->get('sub_service_type'))) {
                $data['sub_service_type'] = request()->get('sub_service_type');
            }

            $save = User::create($data);

            if (is_array($request->subcategory_ids) && count($request->subcategory_ids) > 0) {
                foreach ($request->subcategory_ids as $subcategoryId) {
                    ProviderDetail::create([
                        'user_id' => $save->id,
                        'category_id' => $request->category_id,
                        'subcategory_id' => $subcategoryId,
                    ]);
                }
            }

            $company_details = [
                'user_id' => $save->id,
                'category_id' => $request->category_id,
                'first_name' => $request->provider_first_name,
                'last_name' => $request->provider_last_name,
                'company_name' => $request->company_name,
                'company_website' => $request->company_website
            ];

            $company = UserDetail::create($company_details);

            $feeSub = SubscriptionPackage::select('id', 'is_default', 'price', 'package_term', 'package_duration')
                ->where('is_default', 1)
                ->first();

            if ($feeSub) {
                $currentDate = Carbon::now();

                $trx_date = $currentDate->toDateString();

                $end_date = match ($feeSub->package_term) {
                    'day' => $currentDate->copy()->addDays($feeSub->package_duration)->toDateTimeString(),
                    'month' => $currentDate->copy()->addMonths($feeSub->package_duration)->toDateTimeString(),
                    'yearly' => $currentDate->copy()->addYears($feeSub->package_duration)->toDateTimeString(),
                    'lifetime' => '9999-12-31',
                    default => null,
                };


                $pacakge_trx = [
                    'provider_id' => $save->id,
                    'package_id' => $feeSub->id,
                    'transaction_id' => null,
                    'trx_date' => $trx_date,
                    'end_date' => $end_date,
                    'amount' => $feeSub->price,
                    'payment_status' => 2,
                    'created_by' => $save->id,
                ];

                $create = PackageTrx::create($pacakge_trx);
            }

            DB::table('otp_settings')->where('email', $request->email)->delete();
            
            $signupDate = formatDateTime($save->created_at);

            if ($providerApprovalStatus == 1) {
                try {
                    // Send provider welcome email (Template 98) even when approval is required
                    $this->sendProviderWelcomeEmail($request, 2, $signupDate);
                    $this->sendProviderSignupEmailToAdmin($request, 32, $signupDate);
                } catch (\Exception $e) {
                    \Log::warning('Provider welcome emails failed but continuing: ' . $e->getMessage());
                }

                // Check if ID verification is enabled
                if (providerIdVerificationEnabled()) {
                    return response()->json([
                        'code' => 200,
                        'message' => 'Registration successful! Your account is pending admin approval.',
                        'provider_verified_status' => $providerVerifiedStatus,
                        'provider_approval_status' => 1,
                        'next_step' => 'id_verification'
                    ]);
                }

                return response()->json([
                    'code' => 200,
                    'message' => 'Registration successful! Your account is pending admin approval.',
                    'provider_verified_status' => $providerVerifiedStatus,
                    'provider_approval_status' => 1
                ]);
            } else {
                \Log::info('CommunicationController: About to send provider welcome emails', [
                    'provider_email' => $request->email ?? 'N/A',
                    'signup_date' => $signupDate
                ]);
                
                try {
                    $this->sendProviderWelcomeEmail($request, 2, $signupDate);
                    \Log::info('CommunicationController: sendProviderWelcomeEmail completed');
                } catch (\Exception $e) {
                    \Log::error('CommunicationController: Provider welcome email FAILED', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                
                try {
                    $this->sendProviderSignupEmailToAdmin($request, 32, $signupDate);
                    \Log::info('CommunicationController: sendProviderSignupEmailToAdmin completed');
                } catch (\Exception $e) {
                    \Log::error('CommunicationController: Admin email FAILED', [
                        'error' => $e->getMessage()
                    ]);
                }

                Auth::login($save);

                session(['user_id' => $save->id]);
                Cache::forget('provider_auth_id');
                Cache::forever('provider_auth_id',  $save->id);

                // Check if ID verification is enabled
                if (providerIdVerificationEnabled()) {
                    return response()->json([
                        'code' => 200,
                        'message' => 'Registration successful! Redirecting to your dashboard...',
                        'provider_verified_status' => 1,
                        'provider_approval_status' => 0,
                        'next_step' => 'id_verification',
                        'user_id' => $save->id,
                        'user_type' => $save->user_type
                    ]);
                }

                return response()->json([
                    'code' => 200,
                    'message' => 'Registration successful! Redirecting to your dashboard...',
                    'provider_verified_status' => 1,
                    'provider_approval_status' => 0,
                    'user_id' => $save->id,
                    'user_type' => $save->user_type
                ]);
            }
        } else if ($request->login_type == "register") {
            $request->validate([
                'otp' => 'required',
            ]);

            $otpSetting = DB::table('otp_settings')->where('email', $request->email)->first();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $currentDateTime = now();
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }

            $data = [
                'name' => $request->name,
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'password' => Hash::make($request->password),
                'user_type' => 3,
            ];

            $save = User::create($data);

            $company_details = [
                'user_id' => $save->id,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'mobile_number' => $request->phone_number,
                'gender' => $request->gender,
                'dob' => $request->dob,
                'address' => $request->address,
                'country' => $request->country,
                'state' => $request->state,
                'city' => $request->city,
                'postal_code' => $request->postal_code,
            ];

            $company = UserDetail::create($company_details);

            Auth::login($save);

            session(['user_id' => $save->id]);
            Cache::forget('user_auth_id');
            Cache::forever('user_auth_id',  $save->id);
            DB::table('otp_settings')->where('email', $request->email)->delete();

            $this->sendUserWelcomeEmail($request);

            return response()->json(['message' => 'OTP verified successfully']);
        } else if ($request->login_type == "forgot_email") {
            $request->validate([
                'forgot_email' => 'required|email',
                'otp' => 'required',
            ]);

            $user = User::where('email', $request->forgot_email)->first();

            if (!$user) {
                return response()->json([ 'code' => 200, 'error' => 'User not found'], 404);
            }

            $otpSetting = DB::table('otp_settings')->where('email', $request->forgot_email)->first();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $currentDateTime = now();
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['code' => 422, 'error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }
            DB::table('otp_settings')->where('email', $request->forgot_email)->delete();

            $data = "done";

            return response()->json(['code' => 200, 'message' => 'OTP verified successfully', 'data' => $data, 'email' => $request->forgot_email]);
        } else {
            $request->validate([
                'email' => 'required|email',
                'otp' => 'required',
            ]);

            $user = User::where('email', $request->email)->first();

            if (!$user) {
                return response()->json(['error' => 'User not found'], 404);
            }

            $otpSetting = DB::table('otp_settings')->where('email', $request->email)->first();
            if (isset($otpSetting)) {
                $expire = $otpSetting->expires_at ?? "";
                if ($expire != '') {
                    $currentDateTime = now();
                    $expiresAt = \Carbon\Carbon::parse($expire);
                    if ($currentDateTime->greaterThanOrEqualTo($expiresAt)) {
                        return response()->json(['error' => 'OTP is expired'], 400);
                    }
                }
                $otp = $otpSetting->otp ?? "";
                if ($otp != '') {
                    if ($otp !== $request->otp) {
                        return response()->json(['error' => 'The OTP you entered is invalid. Please check and try again.'], 400);
                    }
                }
            }

            Auth::login($user);

            session(['user_id' => $user->id]);
            if ($user->user_type == '2') {
                Cache::forget('provider_auth_id');
                Cache::forever('provider_auth_id',  $user->id);
            } else {
                Cache::forget('user_auth_id');
                Cache::forever('user_auth_id',  $user->id);
            }
            DB::table('otp_settings')->where('email', $request->email)->delete();
            $token = $user->createToken('user-token')->plainTextToken;
            return response()->json([
                'message' => 'OTP verified successfully',
                'user_id' => $user->id,
                'user_type' => $user->user_type,
                'token' => $token
            ]);
        }
    }

    private function sendUserWelcomeEmail(Request $request)
    {
        // Send WhatsApp welcome message
        try {
            $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
            $phoneNumber = $request->phone_number ?? '';
            $firstName = $request->first_name ?? '';
            $lastName = $request->last_name ?? '';
            
            // Get company name from GlobalSetting using Eloquent
            $companySetting = \Modules\GlobalSetting\Entities\GlobalSetting::where('key', 'company_name')->first();
            $companyName = $companySetting->value ?? 'Our Platform';
            
            if (!empty($phoneNumber) && !empty($firstName)) {
                $smsRepository->sendUserWelcomeWhatsApp($phoneNumber, $firstName, $lastName, $companyName);
            }
        } catch (\Exception $e) {
            \Log::error('Failed to send user welcome WhatsApp: ' . $e->getMessage());
        }
        
        $notificationType = 1;

        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->first();

        if ($template) {
            $settings = GlobalSetting::whereIn('key', ['company_name', 'website', 'phone_no', 'site_email', 'site_address'])->pluck('value', 'key');
            $companyName = $settings['company_name'] ?? '';
            $companyWebsite = $settings['website'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyEmail = $settings['site_email'] ?? '';
            $companyAddress = $settings['site_address'] ?? '';
            $contact = $companyEmail . ' | ' . $companyPhone;
            $customerName = $request->first_name . ' ' . $request->last_name;

            // Prepare email data
            $subject = str_replace(
                ['{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', '{{website_link}}', '{{contact}}', '{{company_address}}', '{{company_email}}', '{{company_phone}}'],
                [$request->name, $request->first_name, $request->last_name, $customerName, $request->phone_number, $request->email, $companyName, $companyWebsite, $contact, $companyAddress, $companyEmail, $companyPhone],
                $template->subject
            );

            $content = str_replace(
                ['{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', '{{website_link}}', '{{contact}}', '{{company_address}}', '{{company_email}}', '{{company_phone}}'],
                [$request->name, $request->first_name, $request->last_name, $customerName, $request->phone_number, $request->email, $companyName, $companyWebsite, $contact, $companyAddress, $companyEmail, $companyPhone],
                $template->content
            );

            $emailData = [
                'to_email' => $request->email,
                'subject' => $subject,
                'content' => $content
            ];

            try {
                $emailRequest = new Request($emailData);
                $emailController = new EmailController();
                $emailController->sendEmail($emailRequest);
                
                // Also send admin notification (Template 100)
                $this->sendUserRegistrationToAdmin($request);
                
            } catch (\Exception $e) {
                Log::error('Failed to send registration email: ' . $e->getMessage());
            }
        }
    }
    
    private function sendUserRegistrationToAdmin(Request $request)
    {
        try {
            \Modules\Communication\app\Helpers\MailConfigurator::configureMail();
            
            $template = Templates::select('subject', 'content')
                ->where('type', 1)
                ->where('notification_type', 4) // Template 100 - New User Registration to Admin
                ->where('status', 1)
                ->first();
            
            if (!$template) {
                return;
            }
            
            $settings = GlobalSetting::whereIn('key', ['company_name', 'website', 'phone_no', 'site_email', 'company_address'])->pluck('value', 'key');
            $companyName = $settings['company_name'] ?? '';
            $companyWebsite = $settings['website'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyEmail = $settings['site_email'] ?? '';
            $companyAddress = $settings['company_address'] ?? '';
            
            $userName = $request->name ?? '';
            $firstName = $request->first_name ?? explode(' ', $request->name)[0] ?? '';
            $lastName = $request->last_name ?? (explode(' ', $request->name)[1] ?? '');
            $email = $request->email ?? '';
            $phoneNumber = $request->phone_number ?? '';
            $signupDate = date('d/m/Y');
            
            // All possible placeholders for user registration
            $placeholders = [
                '{{user_name}}', '{{first_name}}', '{{last_name}}', 
                '{{email_id}}', '{{user_email}}', '{{email}}',
                '{{phone_number}}', '{{user_phone}}', '{{phone}}',
                '{{company_name}}', '{{signup_date}}', '{{registration_date}}',
                '{{company_website}}', '{{company_email}}', '{{company_phone}}',
                '{{company_address}}'
            ];
            
            $replacements = [
                $userName, $firstName, $lastName,
                $email, $email, $email,
                $phoneNumber, $phoneNumber, $phoneNumber,
                $companyName, $signupDate, $signupDate,
                $companyWebsite, $companyEmail, $companyPhone,
                $companyAddress
            ];
            
            $subject = str_replace($placeholders, $replacements, $template->subject);
            $content = str_replace($placeholders, $replacements, $template->content);
            
            // Send WhatsApp to admin
            try {
                $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
                $smsRepository->sendAdminUserSignupWhatsApp($firstName, $lastName, $email, $phoneNumber);
            } catch (\Exception $e) {
                \Log::error('Failed to send admin user signup WhatsApp: ' . $e->getMessage());
            }
            
            // Send email to ALL admins (not just one)
            $AdminUsers = User::where('user_type', 1)->get();
            foreach ($AdminUsers as $AdminUser) {
                $emailData = [
                    'to_email' => $AdminUser->email,
                    'subject' => $subject,
                    'content' => $content
                ];
                
                try {
                    $emailRequest = new Request($emailData);
                    $emailController = new EmailController();
                    $emailController->sendEmail($emailRequest);
                    Log::info('User registration admin email sent to: ' . $AdminUser->email);
                } catch (\Exception $innerEx) {
                    Log::error('Failed to send admin email to ' . $AdminUser->email . ': ' . $innerEx->getMessage());
                }
            }
        } catch (\Exception $e) {
            Log::error('Failed to send user registration admin email: ' . $e->getMessage());
        }
    }

    private function sendProviderSignupEmailToAdmin(Request $request, $notificationType = '', $signupDate = '')
    {
        Log::info('sendProviderSignupEmailToAdmin called', [
            'notification_type' => $notificationType,
            'provider_email' => $request->email ?? 'N/A'
        ]);

        // Send WhatsApp notification to admin FIRST (regardless of email template status)
        try {
            $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
            $providerEmail = $request->email ?? '';
            $providerPhone = $request->phone_number ?? $request->mobile ?? '';
            
            // Get first and last name from provider fields
            $firstName = $request->provider_first_name ?? $request->first_name ?? explode(' ', $request->name)[0] ?? '';
            $lastName = $request->provider_last_name ?? $request->last_name ?? (explode(' ', $request->name)[1] ?? '');
            $customerName = trim($firstName . ' ' . $lastName);
            
            // Get category name
            $categoryName = 'Provider'; // Default
            if (!empty($request->category_id)) {
                $category = \DB::table('categories')->where('id', $request->category_id)->first();
                $categoryName = $category->name ?? 'Provider';
            }
            
            if (!empty($customerName) && !empty($providerEmail)) {
                $smsRepository->sendAdminProviderSignupWhatsApp(
                    $categoryName,
                    $customerName,
                    $providerEmail,
                    $providerPhone
                );
            }
        } catch (\Exception $e) {
            \Log::warning('Failed to send admin provider signup WhatsApp: ' . $e->getMessage());
        }

        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->where('status', 1)
            ->first();
        
        if (!$template) {
            Log::info('Provider signup admin email template not found or not active', [
                'type' => 1,
                'notification_type' => $notificationType
            ]);
            return;
        }

        if ($template) {
            $settings = GlobalSetting::whereIn('key', ['company_name', 'website', 'phone_no', 'site_email', 'site_address', 'company_address'])->pluck('value', 'key');
            $companyName = $settings['company_name'] ?? '';
            $companyWebsite = $settings['website'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyEmail = $settings['site_email'] ?? '';
            $companyAddress = $settings['company_address'] ?? $settings['site_address'] ?? '';
            $contact = $companyEmail . ' | ' . $companyPhone;
            $adminProvidersLink = url('/admin/providers');
            
            // Get first and last name from provider fields
            $firstName = $request->provider_first_name ?? $request->first_name ?? explode(' ', $request->name)[0] ?? '';
            $lastName = $request->provider_last_name ?? $request->last_name ?? (explode(' ', $request->name)[1] ?? '');
            $customerName = trim($firstName . ' ' . $lastName);
            $providerEmail = $request->email;
            
            // Get category name for template (already retrieved above for WhatsApp, reuse it)
            // Note: categoryName was already set at line 1339-1343

            // Prepare email data - use correct parameter names with admin_providers_link, category_name, provider_name, provider_email
            $placeholders = [
                '{{provider_name}}', '{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', 
                '{{category_name}}', '{{provider_email}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', 
                '{{website_link}}', '{{contact}}', '{{signup_date}}', '{{company_address}}', '{{company_email}}', 
                '{{company_phone}}', '{{admin_providers_link}}'
            ];
            
            $replacements = [
                $customerName, $customerName, $firstName, $lastName, $customerName, 
                $categoryName, $providerEmail, $request->phone_number, $request->email, $companyName, 
                $companyWebsite, $contact, $signupDate, $companyAddress, $companyEmail, 
                $companyPhone, $adminProvidersLink
            ];
            
            $subject = str_replace($placeholders, $replacements, $template->subject);
            $content = str_replace($placeholders, $replacements, $template->content);

            // Send email to ALL admins (not just one)
            $AdminUsers = User::where('user_type', 1)->get();
            
            foreach ($AdminUsers as $AdminUser) {
                $emailData = [
                    'to_email' => $AdminUser->email,
                    'subject' => $subject,
                    'content' => $content
                ];

                try {
                    Log::info('Sending provider signup email to admin', [
                        'to_email' => $emailData['to_email'],
                        'subject' => $emailData['subject']
                    ]);
                    $emailRequest = new Request($emailData);
                    $emailController = new EmailController();
                    $result = $emailController->sendEmail($emailRequest);
                    Log::info('Provider signup admin email sent to: ' . $AdminUser->email, ['result' => $result]);
                } catch (\Exception $e) {
                    Log::error('Failed to send provider signup admin email to ' . $AdminUser->email, [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }
        }
    }

    private function sendProviderWelcomeEmail(Request $request, $notificationType = '', $signupDate = '')
    {
        Log::info('sendProviderWelcomeEmail called', [
            'notification_type' => $notificationType,
            'provider_email' => $request->email ?? 'N/A'
        ]);

        // Send WhatsApp welcome message FIRST (regardless of email template status)
        try {
            $smsRepository = app(\Modules\Communication\app\Repositories\Contracts\SmsInterface::class);
            $phoneNumber = $request->phone_number ?? $request->mobile ?? '';
            
            // Get first and last name from provider fields
            $firstName = $request->provider_first_name ?? $request->first_name ?? explode(' ', $request->name)[0] ?? '';
            $lastName = $request->provider_last_name ?? $request->last_name ?? (explode(' ', $request->name)[1] ?? '');
            $customerName = trim($firstName . ' ' . $lastName);
            
            // Get category name
            $categoryName = 'Provider'; // Default
            if (!empty($request->category_id)) {
                $category = \DB::table('categories')->where('id', $request->category_id)->first();
                $categoryName = $category->name ?? 'Provider';
            }
            
            if (!empty($phoneNumber)) {
                $smsRepository->sendProviderWelcomeWhatsApp(
                    $phoneNumber,
                    $customerName,
                    $categoryName
                );
            }
        } catch (\Exception $e) {
            \Log::warning('Failed to send provider welcome WhatsApp: ' . $e->getMessage());
        }
        
        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->where('status', 1)
            ->first();
        
        if (!$template) {
            Log::info('Provider welcome email template not found or not active', [
                'type' => 1,
                'notification_type' => $notificationType
            ]);
            return;
        }

        if ($template) {
            $settings = GlobalSetting::whereIn('key', ['company_name', 'website', 'phone_no', 'site_email', 'site_address'])->pluck('value', 'key');
            $companyName = $settings['company_name'] ?? '';
            $companyWebsite = $settings['website'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyEmail = $settings['site_email'] ?? '';
            $companyAddress = $settings['site_address'] ?? '';
            $contact = $companyEmail . ' | ' . $companyPhone;
            
            // Get first and last name from provider fields
            $firstName = $request->provider_first_name ?? $request->first_name ?? explode(' ', $request->name)[0] ?? '';
            $lastName = $request->provider_last_name ?? $request->last_name ?? (explode(' ', $request->name)[1] ?? '');
            $customerName = trim($firstName . ' ' . $lastName);

            // Prepare email data - use correct parameter names including provider_name, category_name, provider_email
            $providerName = $firstName . ' ' . $lastName;
            $providerEmail = $request->email;
            
            // Get category name for template
            $categoryName = 'Provider'; // Default
            if (!empty($request->category_id)) {
                $category = \DB::table('categories')->where('id', $request->category_id)->first();
                $categoryName = $category->name ?? 'Provider';
            }
            
            $subject = str_replace(
                ['{{provider_name}}', '{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', '{{category_name}}', '{{provider_email}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', '{{website_link}}', '{{contact}}', '{{signup_date}}', '{{company_address}}', '{{company_email}}', '{{company_phone}}'],
                [$customerName, $customerName, $firstName, $lastName, $customerName, $categoryName, $providerEmail, $request->phone_number, $request->email, $companyName, $companyWebsite, $contact, $signupDate, $companyAddress, $companyEmail, $companyPhone],
                $template->subject
            );
            
            $content = str_replace(
                ['{{provider_name}}', '{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', '{{category_name}}', '{{provider_email}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', '{{website_link}}', '{{contact}}', '{{signup_date}}', '{{company_address}}', '{{company_email}}', '{{company_phone}}'],
                [$customerName, $customerName, $firstName, $lastName, $customerName, $categoryName, $providerEmail, $request->phone_number, $request->email, $companyName, $companyWebsite, $contact, $signupDate, $companyAddress, $companyEmail, $companyPhone],
                $template->content
            );

            // Use correct email field
            $toEmail = $request->email;

            Log::info('sendProviderWelcomeEmail: Email address resolved', [
                'to_email' => $toEmail
            ]);

            $emailData = [
                'to_email' => $toEmail,
                'subject' => $subject,
                'content' => $content
            ];

            try {
                Log::info('Sending provider welcome email', [
                    'to_email' => $emailData['to_email'],
                    'subject' => $emailData['subject']
                ]);
                $emailRequest = new Request($emailData);
                $emailController = new EmailController();
                $result = $emailController->sendEmail($emailRequest);
                Log::info('Provider welcome email sent', ['result' => $result]);
            } catch (\Exception $e) {
                Log::error('Failed to send provider welcome email', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }
    }
}
